<?php

include_once 'config.php';

class branch_model
{
    public $link;

    public function __construct()
    {
        $db_connection = new dbConnection();
        $this->link = $db_connection->connect();
    }

    public function checkBranchCode($code) {
        $stmt = $this->link->prepare("
            SELECT COUNT(*) FROM (
                SELECT branch_code FROM branches WHERE branch_code = ?
                UNION ALL
                SELECT warehouse_code FROM warehouses WHERE warehouse_code = ?
            ) t
        ");
        $stmt->execute([$code, $code]);
        return $stmt->fetchColumn() > 0;
    }


    public function doesLocationAlreadyHaveWarehouse($location_id) {
        $sql = "SELECT COUNT(*) as total FROM branches WHERE location = :location_id";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':location_id', $location_id, PDO::PARAM_STR);
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row && $row['total'] > 0;
    }

    public function generateNextID() {
        // Get max ID from warehouses
        $sql1 = "SELECT MAX(warehouse_id) AS max_warehouse_id FROM warehouses";
        $stmt1 = $this->link->prepare($sql1);
        $stmt1->execute();
        $result1 = $stmt1->fetch(PDO::FETCH_ASSOC);
        $maxWarehouseId = (int) $result1['max_warehouse_id'];

        // Get max ID from branches
        $sql2 = "SELECT MAX(branch_id) AS max_branch_id FROM branches";
        $stmt2 = $this->link->prepare($sql2);
        $stmt2->execute();
        $result2 = $stmt2->fetch(PDO::FETCH_ASSOC);
        $maxBranchId = (int) $result2['max_branch_id'];

        // Determine the highest of the two
        $maxId = max($maxWarehouseId, $maxBranchId);

        // If no records exist in either table
        if ($maxId < 7000) {
            return 7000;
        }

        return $maxId + 1;
    }

    public function isManagerAssignedElsewhere($shop_manager) {
        $query = "SELECT COUNT(*) FROM branches WHERE shop_manager = :shop_manager AND branch_state = 1";
        $stmt = $this->link->prepare($query);
        $stmt->bindParam(':shop_manager', $shop_manager, PDO::PARAM_INT);
        $stmt->execute();

        $count = $stmt->fetchColumn();

        return $count > 0;
    }

    public function isSupervisorAssignedToDifferentLocation($employee_code, $new_location) {
        $query = "SELECT location FROM warehouses WHERE supervisor = :supervisor AND warehouse_state = 1 LIMIT 1";
        $stmt = $this->link->prepare($query);
        $stmt->bindParam(':supervisor', $employee_code, PDO::PARAM_INT);
        $stmt->execute();

        $existing_location = $stmt->fetchColumn();

        if ($existing_location && $existing_location !== $new_location) {
            return true; // Assigned to a different location
        }

        return false; // Not assigned or location matches
    }




    
    
    public function getBranches() {
        try {
            $stmt = $this->link->prepare("SELECT 
                                    b.*,
                                    l.location_name,

                                    GROUP_CONCAT(DISTINCT e.fullname ORDER BY e.fullname SEPARATOR ', ') 
                                        AS manager_names,

                                    GROUP_CONCAT(DISTINCT e.phone_number ORDER BY e.fullname SEPARATOR ', ') 
                                        AS phone_numbers,

                                    COUNT(DISTINCT bp.product_id) AS product_count,

                                    b.branch_state,
                                    CASE 
                                        WHEN b.branch_state = 1 THEN 'Active Branch'
                                        WHEN b.branch_state = 0 THEN 'Deactivated Branch'
                                    END AS branch_status

                                FROM branches b

                                LEFT JOIN locations l 
                                    ON b.location = l.loc_id

                                LEFT JOIN users uu 
                                    ON uu.location = b.branch_id
                                AND uu.role = 'Cashier'
                                AND uu.user_state = 1

                                LEFT JOIN employees e 
                                    ON uu.employee_code = e.employee_code

                                LEFT JOIN branch_products bp 
                                    ON b.branch_id = bp.branch_id

                                WHERE b.branch_state IN (0,1)

                                GROUP BY b.branch_id

                                ORDER BY b.branch_name ASC
            ");
            $stmt->execute();
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            error_log("Get branches error: " . $e->getMessage());
            return [];
        }
    }
    

    public function saveBranch($data) {
        try {
            $stmt = $this->link->prepare("INSERT INTO branches (branch_id, branch_name, branch_code, location, datecaptured, operator)
                VALUES (:branch_id, :branch_name, :branch_code, :location, :datecaptured, :operator)
            ");

            $stmt->bindParam(':branch_id', $data['branch_id']);
            $stmt->bindParam(':branch_name', $data['branch_name']);
            $stmt->bindParam(':branch_code', $data['branch_code']);
            $stmt->bindParam(':location', $data['location']);
            $stmt->bindParam(':datecaptured', $data['datecaptured']);
            $stmt->bindParam(':operator', $data['operator']);
    
            return $stmt->execute();
        } catch (PDOException $e) {
            // Optionally log error
            error_log("Save Branch error: " . $e->getMessage());
            return false;
        }
    }

    public function branch_stock($branch_id) {
        try {
            $sql = "SELECT 
                        bp.*, 
                        p.product_name, 
                        p.product_code, 
                        bp.opening_stock,
                        bp.quantity,
                        pc.categoryname,
                        pu.unitname,
                        b.branch_name,
                        p.selling_price,
                        bp.min_level as min_order_level
                    FROM 
                        branch_products bp
                    LEFT JOIN
                        branches b ON bp.branch_id = b.branch_id
                    LEFT JOIN 
                        products p ON bp.product_id = p.product_id
                    LEFT JOIN
                        product_categories pc ON p.category = pc.category_id
                    LEFT JOIN
                        product_units pu ON p.unit = pu.unit_id
                    WHERE 
                        bp.branch_id = :branch_id";
    
            $stmt = $this->link->prepare($sql);
            $stmt->bindParam(':branch_id', $branch_id, PDO::PARAM_INT);
            $stmt->execute();
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
    
        } catch (PDOException $e) {
            error_log("Warehouse stock fetch error: " . $e->getMessage());
            return [];
        }
    }

    public function getBranchDetails($branch_id) {
        try {
            $sql = "SELECT 
                        b.*, 
                        b.location,
                        l.location_name, 
                        (
                            SELECT COUNT(*) 
                            FROM branch_products bp 
                            WHERE bp.branch_id = bp.branch_id
                        ) AS total_products
                    FROM 
                        branches b
                    LEFT JOIN 
                        locations l ON b.location = l.loc_id
                    WHERE 
                        b.branch_id = :branch_id";
    
            $stmt = $this->link->prepare($sql);
            $stmt->bindParam(':branch_id', $branch_id, PDO::PARAM_INT);
            $stmt->execute();
    
            return $stmt->fetch(PDO::FETCH_ASSOC);
    
        } catch (PDOException $e) {
            error_log("Get warehouse details error: " . $e->getMessage());
            return false;
        }
    }

    

    public function updatebranch($id, $name, $supervisor, $location) {
        $sql = "UPDATE branches SET branch_name = ?, shop_manager = ?, location = ? WHERE branch_id = ?";
        $stmt = $this->link->prepare($sql);
        return $stmt->execute([$name, $supervisor, $location, $id]);
    }

    public function doesLocationExistExcludingBranch($location_id, $exclude_branch_id) {
        $sql = "SELECT COUNT(*) as total FROM branches WHERE location = :location_id AND branch_id != :exclude_id";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':location_id', $location_id, PDO::PARAM_STR);
        $stmt->bindParam(':exclude_id', $exclude_branch_id, PDO::PARAM_INT);
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row && $row['total'] > 0;
    }

    public function isManagerInAnotherWarehouseExcludingCurrent($shop_manager, $excluded_branch_id) {
        $sql = "SELECT COUNT(*) as total FROM branches 
                WHERE shop_manager = :shop_manager AND branch_id != :exclude_id";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':shop_manager', $shop_manager, PDO::PARAM_INT);
        $stmt->bindParam(':exclude_id', $excluded_branch_id, PDO::PARAM_INT);
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row && $row['total'] > 0;
    }

    public function doesBranchNameExistExcludingCurrent($branch_name, $branch_id) {
        $sql = "SELECT COUNT(*) FROM branches WHERE branch_name = :name AND branch_id != :id";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':name', $branch_name);
        $stmt->bindParam(':id', $branch_id);
        $stmt->execute();
        return $stmt->fetchColumn() > 0;
    }
    

    public function getWarehouseBySupervisor($employee_code) {
        $sql = "SELECT warehouse_id, location, warehouse_state 
                FROM warehouses 
                WHERE supervisor = :employee_code 
                LIMIT 1";

        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':employee_code', $employee_code, PDO::PARAM_INT);
        $stmt->execute();

        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
    
}