<?php

include_once 'config.php';

class company_model
{
    public $link;

    public function __construct()
    {
        $db_connection = new dbConnection();
        $this->link = $db_connection->connect();
    }

    public function getCompanyDetails() {
    try {
        $sql = "SELECT
                    ss.id, company_id, company_name, tpin, ss.location, ll.location_name, address, phone_number, company_phon2, email, 
                    website, logo, fiscal_start, fiscal_end, directors_signature 
                FROM 
                    settings ss
                LEFT JOIN
                    locations ll ON ss.location = ll.loc_id  
                WHERE 
                    ss.id=1";

            $stmt = $this->link->prepare($sql);
            $stmt->execute();

            $company = $stmt->fetch(PDO::FETCH_ASSOC); // Fetch single row as associative array

            return $company ?: []; // Return empty array if no record found
        } catch (PDOException $e) {
            // Optional: Log the error or handle it gracefully
            return ['error' => $e->getMessage()];
        }
    }

    public function updateCompanyDetails($id, $data) {
        $query = "UPDATE settings SET company_id = ?, company_name = ?, tpin = ?, location = ?, address = ?, phone_number = ?, company_phon2 = ?, email = ?, website = ?";

        $params = [
            $data['company_id'], $data['company_name'], $data['tpin'], $data['location'], $data['address'], $data['phone_number'], $data['company_phone2'], $data['email'], $data['website']
        ];

        if (!empty($data['logo'])) {
            $query .= ", logo = ?";
            $params[] = $data['logo'];
        }

        if (!empty($data['directors_signature'])) {
            $query .= ", directors_signature = ?";
            $params[] = $data['directors_signature'];
        }

        $query .= " WHERE id = ?";
        $params[] = $id;

        $stmt = $this->link->prepare($query);
        return $stmt->execute($params);
    }

    public function getThirdPartyCompanies() {
        $sql = "SELECT
                    tpc.company_id,
                    tpc.company_name,
                    ll.location_name,
                    tpc.phone_number,
                    tpc.contact_person,
                    tpc.email_address,
                    tpc.datecreated,    
                    COUNT(tpp.product_id) AS total_products,
                    COALESCE(SUM(tpp.quantity),0) AS total_stock_quantity,
                    CASE 
                        WHEN tpc.company_state = 1 THEN 'Active Company'
                        ELSE 'Deactivated Company'
                    END as company_status,
                    tpc.company_state   
                FROM 
                    third_party_companies tpc
                LEFT JOIN
                    locations ll ON tpc.location = ll.loc_id
                LEFT JOIN
                    third_party_products tpp ON tpc.company_id = tpp.company_id
                GROUP BY 
                    tpc.company_id, tpc.company_name, ll.location_name";
        $smtp = $this->link->prepare($sql);
        $smtp->execute();
        $companies = $smtp->fetchAll(PDO::FETCH_ASSOC);
        
        return $companies;
    }
    public function getNextCompanyId() {
        $sql = "SELECT MAX(id) as max_id FROM third_party_companies";
        $stmt = $this->link->prepare($sql);
        $stmt->execute();
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        $max_id = $result['max_id'];

        if ($max_id) {
            $new_id = $max_id + 1;
        } else {
            
            $new_id = 1;
        }
        
        // Format the new ID with the "TPC-" prefix and zero-padded to 3 digits
        $new_company_id = "TPC" . str_pad($new_id, 3, "0", STR_PAD_LEFT);
        
        return $new_company_id;
    }

    public function checkEmail  ($email) {
        $sql = "SELECT * FROM third_party_companies WHERE email_address = :email_address";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':email_address', $email);
        $stmt->execute();
        
        return $stmt->fetch(PDO::FETCH_ASSOC); // Returns false if no match found
    }

    public function checkCompanyName  ($company_name) {
        $sql = "SELECT * FROM third_party_companies WHERE company_name = :company_name";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':company_name', $company_name);
        $stmt->execute();
        
        return $stmt->fetch(PDO::FETCH_ASSOC); // Returns false if no match found
    }

    // Add new company to the database
    public function addNewCompany($data) {
        $sql = "INSERT INTO third_party_companies (company_id, company_name, contact_person, phone_number, email_address, location, datecreated, operator, company_state)
                VALUES (:company_id, :company_name, :contact_person, :phone_number, :email_address, :location, :datecreated, :operator, :company_state)";
        $stmt = $this->link->prepare($sql);

        $stmt->bindParam(':company_id', $data['company_id']);
        $stmt->bindParam(':company_name', $data['company_name']);
        $stmt->bindParam(':contact_person', $data['contact_person']);
        $stmt->bindParam(':phone_number', $data['phone_number']);
        $stmt->bindParam(':email_address', $data['email_address']);
        $stmt->bindParam(':location', $data['location']);
        $stmt->bindParam(':datecreated', $data['datecreated']);
        $stmt->bindParam(':operator', $data['operator']);
        $stmt->bindParam(':company_state', $data['company_state']);

        return $stmt->execute();
    }

    public function getProductsByCompanyId(string $company_id): array {
        $sql = "SELECT 
                    p.product_code,
                    p.product_name,
                    pc.categoryname AS category,
                    tpp.quantity,
                    tpp.min_level,
                    tpp.opening_stock,
                    p.product_state
                FROM 
                    third_party_products tpp
                JOIN 
                    products p ON p.product_id = tpp.product_id
                JOIN 
                    product_categories pc ON p.category = pc.category_id
                WHERE 
                    tpp.company_id = :company_id
        ";

        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':company_id', $company_id);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getProductCountByCategory($company_id) {
        $sql = "SELECT 
                    pc.categoryname AS category_name,
                    COUNT(p.product_id) AS product_count
                FROM 
                    third_party_products tpp
                JOIN 
                    products p ON p.product_id = tpp.product_id
                JOIN 
                    product_categories pc ON p.category = pc.category_id
                WHERE 
                    tpp.company_id = :company_id
                GROUP BY 
                    pc.categoryname";
        
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':company_id', $company_id);
        $stmt->execute();
        
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function isCompanyNameExists($company_name, $company_id) {
        $sql = "SELECT COUNT(*) FROM third_party_companies WHERE company_name = :company_name AND company_id != :company_id";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':company_name', $company_name);
        $stmt->bindParam(':company_id', $company_id);
        $stmt->execute();
        return $stmt->fetchColumn() > 0;  
    }

    
    public function isEmailExists($email_address, $company_id) {
        $sql = "SELECT COUNT(*) FROM third_party_companies WHERE email_address = :email_address AND company_id != :company_id";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':email_address', $email_address);
        $stmt->bindParam(':company_id', $company_id);
        $stmt->execute();
        return $stmt->fetchColumn() > 0; 
    }

    public function update_third_party_company_details($company_id, $company_name, $contact_person, $phone_number, $email_address, $location_id, $company_state) {
        $sql = "UPDATE third_party_companies SET company_name = :company_name, contact_person = :contact_person, phone_number = :phone_number, email_address = :email_address,
                    location = :location_id, company_state = :company_state WHERE company_id = :company_id";

        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':company_name', $company_name);
        $stmt->bindParam(':contact_person', $contact_person);
        $stmt->bindParam(':phone_number', $phone_number);
        $stmt->bindParam(':email_address', $email_address);
        $stmt->bindParam(':location_id', $location_id);
        $stmt->bindParam(':company_state', $company_state);
        $stmt->bindParam(':company_id', $company_id);

        return $stmt->execute();
    }




    public function getThirdPartyCompanyDetails($company_id) {
        try {
            $sql = "SELECT 
                        tpc.company_id,
                        tpc.company_name,
                        tpc.contact_person,
                        tpc.phone_number,
                        tpc.email_address,
                        tpc.company_state,
                        tpc.location,
                        ll.location_name,
                        CASE 
                            WHEN tpc.company_state = 1 THEN 'Active Company'
                            WHEN tpc.company_state = 0 THEN 'Deactivated Company'
                        END as company_status
                    FROM
                        third_party_companies tpc
                    LEFT JOIN
                        locations ll ON tpc.location = ll.loc_id
                    WHERE
                        tpc.company_id = :company_id";

            $stmt = $this->link->prepare($sql);
            $stmt->bindParam(':company_id', $company_id, PDO::PARAM_STR);
            $stmt->execute();

            return $stmt->fetch(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            error_log("Company details error: " . $e->getMessage());
            return false;
        }
    }

    public function addNewSupplier($data) {
        $sql = "INSERT INTO suppliers (supplier_name, contact_person, contact_number, email_address, datecreated, operator, supplier_state) VALUES (:supplier_name, :contact_person, :phone_number, :email_address, :datecreated, :operator, :supplier_state)";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':supplier_name', $data['supplier_name']);
        $stmt->bindParam(':contact_person', $data['contact_person']);
        $stmt->bindParam(':phone_number', $data['phone_number']);
        $stmt->bindParam(':email_address', $data['email']);
        $stmt->bindParam(':datecreated', $data['datecreated']);
        $stmt->bindParam(':operator', $data['operator']);
        $stmt->bindParam(':supplier_state', $data['state']);

        return $stmt->execute();
    }

    public function updateSupplier($supplier_id, $data) {
        $sql = "UPDATE suppliers SET supplier_name = :supplier_name, contact_person = :contact_person, contact_number = :phone_number, email_address = :email_address WHERE supplier_id = :supplier_id";

        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':supplier_name', $data['supplier_name']);
        $stmt->bindParam(':contact_person', $data['contact_person']);
        $stmt->bindParam(':phone_number', $data['phone_number']);
        $stmt->bindParam(':email_address', $data['email']);
        $stmt->bindParam(':supplier_id', $supplier_id);

        return $stmt->execute();
    }

    public function updateSupplierState($supplier_id, $new_state) {
        $sql = "UPDATE suppliers SET supplier_state = :supplier_state WHERE supplier_id = :supplier_id";

        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':supplier_state', $new_state);
        $stmt->bindParam(':supplier_id', $supplier_id);

        return $stmt->execute();
    }


}