<?php

include_once 'config.php';

class finance_model
{
    public $link;

    public function __construct()
    {
        $db_connection = new dbConnection();
        $this->link = $db_connection->connect();
    }

    public function getsalesTotals($startDate, $endDate){
        $sql = "SELECT 
                    COALESCE(SUM(total), 0) AS total_sales
                FROM (
                    -- Cash / direct payments
                    SELECT 
                        SUM(amount_paid) AS total
                    FROM 
                        payments
                    WHERE 
                        payment_state = 1
                        AND payment_type = 'Cash Sale'
                        AND payment_date >= :startDate
                        AND payment_date <= :endDate

                UNION ALL

                    -- Invoice sales
                    SELECT 
                        SUM(total_bill) AS total
                    FROM 
                        saved_orders
                    WHERE 
                        order_state IN (0,1,2)
                        AND orderdate >= :startDate
                        AND orderdate <= :endDate
                    ) AS combined_sales";

            $stmt = $this->link->prepare($sql);
            $stmt->execute([
                'startDate' => $startDate,
                'endDate'   => $endDate
            ]);

        return (float) $stmt->fetchColumn();
    }


    public function countOpenInvoices() {
        $sql = "SELECT COUNT(order_number) AS total FROM saved_orders WHERE order_state IN (0, 1)";
        $stmt = $this->link->prepare($sql);
        $stmt->execute();
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        return $result['total'];
    }

    public function getTotalSoldThisMonth($product_code) {
        $sql = "SELECT SUM(quantity) AS total FROM sales_order WHERE product_code = :product_code AND MONTH(date_captured) = MONTH(CURDATE()) AND YEAR(date_captured) = YEAR(CURDATE())";
                
        $stmt = $this->link->prepare($sql);
        $stmt->execute(['product_code' => $product_code]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        return $result['total'] ?? 0;
    }

    public function getTotalSoldLastMonth($product_code) {
        $sql = "SELECT SUM(quantity) AS total 
                FROM sales_order 
                WHERE product_code = :product_code 
                AND MONTH(date_captured) = MONTH(CURDATE() - INTERVAL 1 MONTH) 
                AND YEAR(date_captured) = YEAR(CURDATE() - INTERVAL 1 MONTH)";
                
        $stmt = $this->link->prepare($sql);
        $stmt->execute(['product_code' => $product_code]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        return $result['total'] ?? 0;
    }

    public function getYesterdaySales() {
        $yesterday = date('Y-m-d', strtotime('-1 day'));
        $sql = "SELECT SUM(amount_paid) AS total FROM payments WHERE DATE(datecaptured) = ?";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(1, $yesterday);
        $stmt->execute();
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        return $result['total'] ?? 0;
    }

    public function getLastWeekSales() {
        // Get today's date (end of today)
        $end_of_today = date('Y-m-d');

        // Get the current day of the week (0 = Sunday, 1 = Monday, ..., 6 = Saturday)
        $day_of_week = date('w');  // 0 (Sunday) to 6 (Saturday)

        // Calculate the start of this week (Monday)
        $start_of_this_week = date('Y-m-d', strtotime("-$day_of_week days"));

        // Calculate the end of last week (Sunday) - 1 day before the start of this week
        $end_of_last_week = date('Y-m-d', strtotime("$start_of_this_week -1 day"));

        // Calculate the start of last week (Monday) - 7 days before the end of last week
        $start_of_last_week = date('Y-m-d', strtotime("$end_of_last_week -6 days"));

        // Debugging - log or print the dates
        //error_log("Start of last week: $start_of_last_week, End of last week: $end_of_last_week");

        // Query to get the total sales for last week
        $sql = "SELECT SUM(amount_paid) AS total 
                FROM payments 
                WHERE DATE(datecaptured) BETWEEN ? AND ?";

        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(1, $start_of_last_week);
        $stmt->bindParam(2, $end_of_last_week);
        $stmt->execute();

        $result = $stmt->fetch(PDO::FETCH_ASSOC);

        /* if ($result === false) {
            error_log("No data found for last week.");
        } else {
            error_log("Total sales for last week: " . $result['total']);
        } */

        return $result['total'] ?? 0;
    }






    public function getThisWeekSales($start_day, $end_date) {
        

        $sql = "SELECT 
                    SUM(total) AS total_sales
                FROM (
                -- Cash / direct payments
                    SELECT 
                        SUM(amount_paid) AS total
                    FROM 
                        payments
                    WHERE
                        payment_state = 1
                        AND payment_type = 'Cash Sale'
                        AND DATE(payment_date) BETWEEN :start AND :end

                UNION ALL

                -- Invoice sales
                SELECT SUM(total_bill) AS total
                FROM saved_orders
                WHERE order_state IN (0,1,2)
                AND DATE(orderdate) BETWEEN :start AND :end
            ) AS combined_sales
        ";

        $stmt = $this->link->prepare($sql);
        // Bind parameters: first two for payments, next two for saved_orders
        $stmt->bindParam(':start', $start_day);
        $stmt->bindParam(':end', $end_date);

        $stmt->execute();
        $result = $stmt->fetch(PDO::FETCH_ASSOC);

        return $result['total_sales'] ?? 0;
    }




    public function getSalesLastMonth() {
        // Get the first day of the last month
        $start_of_last_month = date('Y-m-d', strtotime('first day of last month'));

        // Get the last day of the last month
        $end_of_last_month = date('Y-m-d', strtotime('last day of last month'));

        $sql = "SELECT SUM(amount_paid) AS total 
                FROM payments 
                WHERE DATE(datecaptured) BETWEEN ? AND ?";

        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(1, $start_of_last_month);
        $stmt->bindParam(2, $end_of_last_month);
        $stmt->execute();

        $result = $stmt->fetch(PDO::FETCH_ASSOC);

        return $result['total'] ?? 0;
    }

    public function getSalesThisMonth() {
        // Get the first day of the current month
        $start_of_this_month = date('Y-m-d', strtotime('first day of this month'));

        // Get today's date
        $end_of_today = date('Y-m-d');

        $sql = "SELECT SUM(total) AS total_sales
            FROM (
                -- Cash / direct payments
                SELECT SUM(amount_paid) AS total
                FROM payments
                WHERE payment_type = 'Cash Sale'
                AND payment_state = 1
                AND MONTH(datecaptured) = MONTH(NOW())
                AND YEAR(datecaptured) = YEAR(NOW())

                UNION ALL

                -- Invoice sales
                SELECT SUM(total_bill) AS total
                FROM saved_orders
                WHERE order_state IN (0,1,2)
                AND MONTH(orderdate) = MONTH(NOW())
                AND YEAR(orderdate) = YEAR(NOW())
            ) AS combined_sales
        ";

        $stmt = $this->link->prepare($sql);
        $stmt->execute();
        $result = $stmt->fetch(PDO::FETCH_ASSOC);

        return $result['total_sales'] ?? 0;
    }


    public function getReceiptDetails($receipt_number) {
        $sql = "SELECT
                    cc.customer_name,
                    p.datecaptured as payment_date,
                    p.amount_paid,
                    p.sale_id,
                    p.balance,
                    p.payment_state,
                    p.customer
                FROM 
                    payments p
                LEFT JOIN
                    customers cc ON p.customer = cc.customer_id
                WHERE 
                    p.receipt_number = :receipt_number";
        $smtp = $this->link->prepare($sql);
        $smtp->bindParam(':receipt_number', $receipt_number, PDO::PARAM_STR);
        $smtp->execute();
        $receipt_data = $smtp->fetch(PDO::FETCH_ASSOC);
        return $receipt_data;
    }

    public function getInvoicePayments(){
        $sql = "SELECT
                ip.id as pay_id,
                cc.customer_name,
                py.receipt_number,
                ip.date_paid as payment_date,    
                py.tx_id,
                ip.customer,
                ip.invoice_number,
                py.payment_method,
                py.transction_id,
                ip.amount_paid,
                py.datecaptured,
                py.operator,
                py.payment_state,
                CASE 
                    WHEN py.payment_state = 1 THEN
                    'Aprroved Payment'
                    WHEN py.payment_state = 0 THEN
                    'Cancelled Payment'
                END as payment_status,
                COALESCE(emp.fullname, 'Default User') as operator_name

            FROM 
                invoice_payments ip
            LEFT JOIN
                payments py ON ip.payment_id = py.id
            LEFT JOIN
                customers cc ON py.customer = cc.customer_id
            LEFT JOIN
                employees emp ON py.operator = emp.employee_code
            WHERE
                py.payment_state = 1
            ORDER BY 
                py.datecaptured DESC
        ";
        $smtp = $this->link->prepare($sql);
        $smtp->execute();
        $data = $smtp->fetchAll();
        return $data;
    }

    public function getDistinctLocationsByDate($start, $end) {
        $sql = "SELECT DISTINCT 
                    p.location AS id,
                    COALESCE(w.warehouse_name, b.branch_name) AS location_name
                FROM 
                    payments p
                LEFT JOIN 
                    warehouses w ON p.location = w.warehouse_id
                LEFT JOIN 
                    branches b ON p.location = b.branch_id
                WHERE 
                    p.payment_state = 1
                    AND DATE(p.datecaptured) BETWEEN :start AND :end

            UNION ALL

                /* Customer Invoices */
                SELECT DISTINCT 
                    so.department AS id,
                    COALESCE(w.warehouse_name, b.branch_name) AS location_name
                FROM 
                    sales_order so
                LEFT JOIN 
                    warehouses w ON so.department = w.warehouse_id
                LEFT JOIN 
                    branches b ON so.department = b.branch_id
                WHERE 
                    so.status = 1
                    AND DATE(so.date_captured) BETWEEN :start AND :end
        ";

        $stmt = $this->link->prepare($sql);
        $stmt->execute([':start' => $start, ':end' => $end]);

        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }



    public function getSalesDataPerHour($date){
        $sql = "SELECT 
                    hour,
                    SUM(value) AS value
                FROM (

                    /* Cash sales (from payments) */
                    SELECT 
                        HOUR(datecaptured) AS hour,
                        SUM(amount_paid) AS value
                    FROM 
                        payments
                    WHERE 
                        DATE(datecaptured) = ?
                        AND payment_type = 'Cash Sale'
                        AND payment_state = 1
                    GROUP BY 
                        HOUR(datecaptured)

                    UNION ALL

                    /* Invoice totals (from saved_orders) */
                    SELECT 
                        HOUR(datecaptured) AS hour,
                        SUM(total_bill) AS value
                    FROM saved_orders
                    WHERE 
                        DATE(datecaptured) = ?
                        AND order_state IN (0,1,2)
                    GROUP BY 
                        HOUR(datecaptured)

                    ) combined
                GROUP BY 
                    hour
                ORDER BY 
                    hour
        ";

        $stmt = $this->link->prepare($sql);
        $stmt->execute([$date, $date]);

        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }



    public function getTopSellingProducts() {
        // Get the first and last day of the current month
        $startOfMonth = date('Y-m-01');
        $endOfMonth = date('Y-m-t');

        // SQL query to get the top-selling product
        $query ="SELECT 
                    p.product_id as product_code,
                    p.product_name,
                    COUNT(so.product_code) as total_sales
                FROM 
                    sales_order so 
                LEFT JOIN 
                    products p ON so.product_code = p.product_id 
                WHERE 
                    MONTH(date_captured) = MONTH(NOW()) 
                    AND YEAR(date_captured) = YEAR(NOW()) 
                GROUP BY 
                    so.product_code 
                ORDER BY total_sales DESC LIMIT 1";

        $stmt = $this->link->prepare($query);
        $stmt->execute();

        // Fetch the top-selling product
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    public function getSalesDataByMonth($year){
        $sql = "SELECT 
                    month,
                    SUM(value) AS value
                FROM (

                    /* ALL payments (cash + invoice payments) */
                    SELECT 
                        DATE_FORMAT(payment_date, '%b %Y') AS month,
                        SUM(amount_paid) AS value
                    FROM 
                        payments
                    WHERE 
                        YEAR(payment_date) = :year
                        AND payment_type = 'Cash Sale'
                        AND payment_state = 1
                    GROUP BY MONTH(payment_date)

                UNION ALL

                    /* Outstanding invoice totals (not yet paid) */
                    SELECT 
                        DATE_FORMAT(orderdate, '%b %Y') AS month,
                        SUM(total_bill) AS value
                    FROM 
                        saved_orders
                    WHERE 
                        YEAR(datecreated) = :year
                        AND order_state IN(0,1,2)
                    GROUP BY 
                        MONTH(orderdate)

                    ) combined
                GROUP BY 
                    month
                ORDER BY 
                    STR_TO_DATE(month, '%b %Y')";

        $stmt = $this->link->prepare($sql);
        $stmt->execute([':year' => $year]);

        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }


    public function getSalaryProjections() {
        $sql = "SELECT 
                pp.postname as month, 
                SUM(emp.gross_salary) AS amount
            FROM 
                employees emp
            LEFT JOIN
                posts pp ON emp.post = pp.id
            WHERE 
                emp.employee_state = 1
            GROUP BY 
                pp.department
            ORDER BY 
                pp.department
        ";
        $stmt = $this->link->prepare($sql);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }



    public function getSalesDataByDateRange($start, $end){
        try {

            $sql = "SELECT 
                        date,
                        SUM(value) AS value
                    FROM (

                        /* Cash sales */
                        SELECT 
                            DATE(datecaptured) AS date,
                            SUM(amount_paid) AS value
                        FROM 
                            payments
                        WHERE 
                            DATE(datecaptured) BETWEEN :start AND :end
                            AND payment_type = 'Cash Sale'
                            AND payment_state = 1
                        GROUP BY 
                            DATE(datecaptured)

                    UNION ALL

                        /* Invoice totals */
                        SELECT 
                            DATE(orderdate) AS date,
                            SUM(total_bill) AS value
                        FROM 
                            saved_orders
                        WHERE 
                            DATE(orderdate) BETWEEN :start AND :end
                            AND order_state IN (0,1,2)
                        GROUP BY 
                            DATE(orderdate)

                        ) combined
                    GROUP BY
                        date
                    ORDER BY 
                        date ASC
            ";

            $stmt = $this->link->prepare($sql);
            $stmt->execute([
                ':start' => $start,
                ':end'   => $end
            ]);

            return $stmt->fetchAll(PDO::FETCH_ASSOC);

        } catch (PDOException $e) {
            error_log("Database Error: " . $e->getMessage());
            return [];
        }
    }


    public function getTopCustomers($start, $end){
        $sql = "SELECT 
                    customer_name,
                    SUM(total_spent) AS total_spent
                FROM (

                    /* Cash sales */
                    SELECT 
                        c.customer_name,
                        SUM(p.amount_paid) AS total_spent
                    FROM 
                        payments p
                    INNER JOIN 
                        customers c ON p.customer = c.customer_id
                    WHERE 
                        DATE(p.datecaptured) BETWEEN :start AND :end
                        AND p.payment_type = 'Cash Sale'
                        AND p.payment_state = 1
                    GROUP BY 
                        p.customer

                    UNION ALL

                    /* Invoice totals */
                    SELECT 
                        c.customer_name,
                        SUM(s.total_bill) AS total_spent
                    FROM 
                        saved_orders s
                    INNER JOIN 
                        customers c ON s.customer = c.customer_id
                    WHERE 
                        DATE(s.orderdate) BETWEEN :start AND :end
                        AND s.order_state IN (0,1,2)
                    GROUP BY 
                        s.customer

                ) combined
            GROUP BY 
                customer_name
            ORDER BY 
                total_spent DESC
            LIMIT 5
        ";

        $stmt = $this->link->prepare($sql);
        $stmt->execute([
            ':start' => $start,
            ':end'   => $end
        ]);

        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }





    public function getDailySales(){
        $sql = "SELECT SUM(total) AS daily_sales
            FROM (
                -- Cash / direct payments
                SELECT 
                    SUM(amount_paid) AS total
                FROM 
                    payments
                WHERE 
                    payment_state = 1
                    AND DATE(payment_date) = CURDATE()
                    AND payment_type = 'Cash Sale'

                UNION ALL

                -- Invoice sales
                SELECT 
                    SUM(total_bill) AS total
                FROM 
                    saved_orders
                WHERE 
                    order_state IN(0,1,2)
                    AND DATE(orderdate) = CURDATE()
            ) AS combined_sales
        ";

        $stmt = $this->link->prepare($sql);
        $stmt->execute();
        $result = $stmt->fetch(PDO::FETCH_ASSOC);

        return $result['daily_sales'] ?? 0;
    }


    public function getWeeklySales(){
        $sql = "SELECT 
                    SUM(amount_paid) AS total
                FROM 
                    payments
                WHERE 
                    payment_state = 1
                    AND payment_type = 'Cash Sale'
                    AND WEEK(payment_date, 1) = WEEK(CURDATE(), 1)
                    AND YEAR(payment_date) = YEAR(CURDATE())";

        $stmt = $this->link->prepare($sql);
        $stmt->execute();
        $result = $stmt->fetch(PDO::FETCH_ASSOC);

        return $result['weekly_sales'] ?? 0;
    }


    public function getMonthlysales() {
        $sql = "SELECT SUM(amount_paid) AS monthly_sales FROM payments WHERE payment_state = 1 AND MONTH(datecaptured) = MONTH(NOW()) AND YEAR(datecaptured) = YEAR(NOW())";
    
        $stmt = $this->link->prepare($sql); // Prepare the query
        $stmt->execute(); // Execute the query
        $result = $stmt->fetch(PDO::FETCH_ASSOC); // Fetch the result
    
        return $result['monthly_sales'] ?? 0; // Return the total or 0 if no sales found
    }

    public function getSalesByDateRange($start_date, $end_date) {
        $stmt = $this->link->prepare("SELECT datecaptured, SUM(amount_paid) as total FROM payments WHERE DATE(datecaptured) >= :start_date AND DATE(datecaptured) <= :end_date GROUP BY DATE(datecaptured) ORDER BY DATE(datecaptured) ASC");
        $stmt->bindParam(':start_date', $start_date, PDO::PARAM_STR);
        $stmt->bindParam(':end_date', $end_date, PDO::PARAM_STR);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    
    
    

    

}