<?php

include_once 'config.php';

class sales_model
{
    public $link;

    public function __construct()
    {
        $db_connection = new dbConnection();
        $this->link = $db_connection->connect();
    }

    public function getDeliveryNotes($states = null) {

        $sql = "SELECT 
                    dn.dn_number,
                    dn.reference,
                    dn.dn_date,
                    dn.dn_state,
                    COALESCE(w.warehouse_name, b.branch_name) AS location,
                    c.customer_name,
                    COUNT(so.product_code) AS number_of_products,
                    SUM(so.quantity) AS total_items,
                    'customer_delivery_note' AS sale_type,
                    CASE
                        WHEN dn.dn_state = 0 THEN 'Cancelled'
                        WHEN dn.dn_state = 1 THEN 'Pending Invoice'
                        WHEN dn.dn_state = 2 THEN 'Invoiced'
                        WHEN dn.dn_state = 3 THEN 'Cancel Requested'
                        ELSE 'Unknown'
                    END AS dn_status
                FROM delivery_notes dn
                LEFT JOIN customers c ON dn.customer = c.customer_id
                LEFT JOIN branches b ON dn.shop_location = b.branch_id
                LEFT JOIN warehouses w ON dn.shop_location = w.warehouse_id
                LEFT JOIN sales_order so ON so.sales_number = dn.dn_number";

        $params = [];

        // ✅ FILTER IF STATE PROVIDED
        if ($states !== null) {

            // if array (multiple states)
            if (is_array($states)) {
                $placeholders = implode(',', array_fill(0, count($states), '?'));
                $sql .= " WHERE dn.dn_state IN ($placeholders)";
                $params = $states;
            } else {
                // single state
                $sql .= " WHERE dn.dn_state = ?";
                $params[] = $states;
            }
        }

        $sql .= " GROUP BY dn.dn_number
                ORDER BY dn.dn_date DESC";

        $stmt = $this->link->prepare($sql);
        $stmt->execute($params);

        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }



    public function getDailySales() {
        // SQL query to select daily sales data
        $sql = "SELECT 
                p.sale_id AS sale_id,
                p.datecaptured AS sale_date,
                p.amount_paid AS total_sales_amount,
                p.payment_state AS sale_status,
                CASE
                    WHEN p.payment_state = 1 THEN 
                    'Completed'
                    WHEN p.payment_state = 0 THEN
                    'Cancelled Payment'
                END as status,
                c.customer_name,
                p.payment_method,
                p.transction_id,
                p.payment_type as transaction_type,
                p.receipt_number
            FROM 
                payments p
            JOIN 
                customers c ON p.customer = c.customer_id
            WHERE 
                DATE(p.datecaptured) = DATE(CURDATE()) 
                AND p.payment_state = 1
                AND payment_type  = 'Cash Sale'
        UNION ALL
            SELECT 
                so.order_number as sale_id,
                so.orderdate as sale_date,
                so.total_bill as total_sales_amount,
                so.order_state as sale_status,
                CASE
                    WHEN so.order_state IN (0,1) THEN 
                    'Outstanding Invoice'
                    WHEN so.order_state = 2 THEN 
                    'Fully Paid'
                END as status,
                c.customer_name,
                '' as payment_method,
                so.order_number as transaction_id,
                'Customer Invoice' as transaction_type,
                '' as receipt_number
            FROM 
                saved_orders so
            LEFT JOIN
                customers c ON so.customer = c.customer_id
            WHERE
                so.order_state IN (0,1,2)
                AND DATE(so.orderdate) = CURDATE()  
        ";

        // Prepare and execute the query
        $stmt = $this->link->prepare($sql);
        $stmt->execute();
        
        // Fetch the results as an associative array
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    public function getSales($range){
        // Sanitize & normalize the range param
        $range = strtolower(trim($range));

        // Build SQL based on range
        if ($range === "monthly") {

            // GROUP BY MONTH
            $query = "SELECT 
                p.sale_id AS sale_id,
                p.payment_date AS sale_date,
                p.datecaptured as transaction_date,
                p.amount_paid AS total_sales_amount,
                p.payment_state AS sale_status,
                CASE
                    WHEN p.payment_state = 1 THEN 
                    'Completed'
                    WHEN p.payment_state = 0 THEN
                    'Cancelled Payment'
                END as status,
                c.customer_name,
                p.payment_method,
                p.transction_id,
                p.payment_type as transaction_type,
                p.receipt_number
            FROM 
                payments p
            JOIN 
                customers c ON p.customer = c.customer_id
            WHERE 
                YEAR(p.payment_date) = YEAR(CURDATE()) 
                AND MONTH(p.payment_date) = MONTH(p.payment_date)
                AND p.payment_state = 1
                AND payment_type  = 'Cash Sale'
        UNION ALL
            SELECT 
                so.order_number as sale_id,
                so.orderdate as sale_date,
                so.datecaptured as transaction_date,
                so.total_bill as total_sales_amount,
                so.order_state as sale_status,
                CASE
                    WHEN so.order_state = 0 THEN
                    'Outstanding Invoice'
                    WHEN so.order_state  =1 THEN 
                    'Outstanding Invoice - Partially Paid'
                    WHEN so.order_state = 2 THEN 
                    'Fully Paid Invoice'
                END as status,
                c.customer_name,
                '' as payment_method,
                so.order_number as transaction_id,
                'Customer Invoice' as transaction_type,
                '' as receipt_number
            FROM 
                saved_orders so
            LEFT JOIN
                customers c ON so.customer = c.customer_id
            WHERE
                so.order_state IN (0,1,2)
                AND MONTH(so.orderdate) = MONTH(NOW())
                AND YEAR(so.orderdate) = YEAR(NOW())
            ORDER BY
                transaction_date DESC";

        }else if($range === "weekly"){
            $day_of_week = date('w');
            $start_of_this_week = date('Y-m-d', strtotime("-$day_of_week days"));
            $end_of_today = date('Y-m-d');

            $query = "SELECT 
                        p.sale_id AS sale_id,
                        p.payment_date AS sale_date,
                        p.datecaptured AS transaction_date,
                        p.amount_paid AS total_sales_amount,
                        p.payment_state AS sale_status,
                        'Completed' AS status,
                        c.customer_name,
                        p.payment_method,
                        p.transction_id,
                        p.payment_type AS transaction_type,
                        p.receipt_number
                    FROM payments p
                    JOIN customers c ON p.customer = c.customer_id
                    WHERE 
                        p.payment_state = 1
                        AND p.payment_type = 'Cash Sale'
                        AND DATE(p.payment_date) BETWEEN :start AND :end

                    UNION ALL

                    SELECT 
                        so.order_number AS sale_id,
                        so.orderdate AS sale_date,
                        so.datecaptured AS transaction_date,
                        so.total_bill AS total_sales_amount,
                        so.order_state AS sale_status,
                        CASE
                            WHEN so.order_state = 0 THEN 'Outstanding Invoice'
                            WHEN so.order_state = 1 THEN 'Outstanding Invoice - Partially Paid'
                            WHEN so.order_state = 2 THEN 'Fully Paid Invoice'
                        END AS status,
                        c.customer_name,
                        '' AS payment_method,
                        so.order_number AS transaction_id,
                        'Customer Invoice' AS transaction_type,
                        '' AS receipt_number
                    FROM saved_orders so
                    LEFT JOIN customers c ON so.customer = c.customer_id
                    WHERE 
                        so.order_state IN (0,1,2)
                        AND DATE(so.orderdate) BETWEEN :start AND :end

                    ORDER BY transaction_date DESC
                ";

                $stmt = $this->link->prepare($query);
                $stmt->execute([
                    ':start' => $start_of_this_week,
                    ':end'   => $end_of_today
                ]);

                return $stmt->fetchAll(PDO::FETCH_ASSOC);

            
        } else {

            // GROUP BY DAY (default)
            $query = "SELECT 
                p.sale_id AS sale_id,
                p.payment_date AS sale_date,
                p.datecaptured as transaction_date,
                p.amount_paid AS total_sales_amount,
                p.payment_state AS sale_status,
                CASE
                    WHEN p.payment_state = 1 THEN 
                    'Completed'
                    WHEN p.payment_state = 0 THEN
                    'Cancelled Payment'
                END as status,
                c.customer_name,
                p.payment_method,
                p.transction_id,
                p.payment_type as transaction_type,
                p.receipt_number
            FROM 
                payments p
            JOIN 
                customers c ON p.customer = c.customer_id
            WHERE 
                DATE(p.payment_date) = DATE(CURDATE()) 
                AND p.payment_state = 1
                AND payment_type  = 'Cash Sale'
        UNION ALL
            SELECT 
                so.order_number as sale_id,
                so.orderdate as sale_date,
                so.datecaptured as transaction_date,
                so.total_bill as total_sales_amount,
                so.order_state as sale_status,
                CASE
                    WHEN so.order_state IN (0,1) THEN 
                    'Outstanding Invoice'
                    WHEN so.order_state = 2 THEN 
                    'Fully Paid Invoice'
                END as status,
                c.customer_name,
                '' as payment_method,
                so.order_number as transaction_id,
                'Customer Invoice' as transaction_type,
                '' as receipt_number
            FROM 
                saved_orders so
            LEFT JOIN
                customers c ON so.customer = c.customer_id
            WHERE
                so.order_state IN (0,1,2)
                AND DATE(so.orderdate) = DATE(NOW())
            ORDER BY
                sale_date DESC";
            }

        // Run the query
        $stmt = $this->link->prepare($query);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    public function getWeeklySales() {
        $end_of_today = date('Y-m-d');
        $day_of_week = date('w');

        $start_of_this_week = date('Y-m-d', strtotime("-$day_of_week days"));

        // SQL query to select weekly sales data (from last Monday until today)
        $sql ="SELECT 
                p.id,
                p.sale_id,
                p.datecaptured AS sale_date,
                p.amount_paid AS total_sales_amount,
                p.payment_state AS sale_status,
                CASE
                    WHEN p.payment_state = 1 THEN 'Completed'
                    WHEN p.payment_state = 0 THEN 'Cancelled Payment'
                END AS payment_status,
                c.customer_name,
                p.payment_method,
                p.transction_id,
                p.payment_type,
                p.receipt_number
            FROM 
                payments p
            JOIN 
                customers c ON p.customer = c.customer_id
            WHERE 
                DATE(datecaptured) BETWEEN ? AND ?
                AND p.payment_state = 1
        ";

        // Prepare and execute the query
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(1, $start_of_this_week);
        $stmt->bindParam(2, $end_of_today);
        $stmt->execute();
        
        // Fetch the results as an associative array
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getPaymentDetails($sales_id){
        $sql = "SELECT * FROM payments WHERE id = :sales_id";
        $smtp = $this->link->prepare($sql);
        $smtp->bindParam(':sales_id', $sales_id, PDO::PARAM_INT);
        $smtp->execute();
        $data = $smtp->fetch(PDO::FETCH_ASSOC);

        return $data;
    }

    public function logCancelledPayment($data) {
        $sql = "INSERT INTO cancelled_payments 
                (tx_id, sale_id, customer, receipt_number, payment_method, payment_type, transction_id,
                payment_date, amount, datecaptured, reason, date_cancelled, cancelled_by, operator, cancel_state)
                VALUES 
                (:tx_id, :sale_id, :customer, :receipt_number, :payment_method, :payment_type, :transction_id,
                :payment_date, :amount, :datecaptured, :reason, :date_cancelled, :cancelled_by, :operator, :cancel_state)";
        
        $stmt = $this->link->prepare($sql);

        $stmt->bindParam(':tx_id', $data['tx_id']);
        $stmt->bindParam(':sale_id', $data['sale_id']);
        $stmt->bindParam(':customer', $data['customer']);
        $stmt->bindParam(':receipt_number', $data['receipt_number']);
        $stmt->bindParam(':payment_method', $data['payment_method']);
        $stmt->bindParam(':payment_type', $data['payment_type']);
        $stmt->bindParam(':transction_id', $data['transction_id']);
        $stmt->bindParam(':payment_date', $data['payment_date']);
        $stmt->bindParam(':amount', $data['amount']);
        $stmt->bindParam(':datecaptured', $data['datecaptured']);
        $stmt->bindParam(':reason', $data['reason']);
        $stmt->bindParam(':date_cancelled', $data['date_cancelled']);
        $stmt->bindParam(':cancelled_by', $data['cancelled_by']);
        $stmt->bindParam(':operator', $data['operator']);
        $stmt->bindParam(':cancel_state', $data['cancel_state']);

        return $stmt->execute();
    }


    public function getCashSales(){
        $sql = "SELECT 
                p.tx_id,
                p.id AS sale_id,
                p.datecaptured AS sale_date,
                p.amount_paid AS total_sales_amount,
                p.payment_state AS sale_status,
                CASE
                    WHEN p.payment_state = 1 THEN 'Completed Payment'
                    WHEN p.payment_state = 0 THEN 'Cancelled Payment'
                END AS payment_status,
                c.customer_name,
                p.payment_method,
                p.transction_id,
                p.payment_type,
                p.receipt_number,
                cp.transction_id as original_transaction_id,
                COALESCE(emp.fullname,'Default Account') as operator
            FROM 
                payments p
            JOIN 
                customers c ON p.customer = c.customer_id
            LEFT JOIN
                cancelled_payments cp ON p.receipt_number = cp.receipt_number
            LEFT JOIN
                employees emp ON p.operator = emp.employee_code
            WHERE
                p.payment_type = 'Cash Sale'
                AND p.payment_state = 1
            ORDER BY 
                p.payment_state DESC,
                p.datecaptured DESC";
        // Prepare and execute the query
        $stmt = $this->link->prepare($sql);
        $stmt->execute();
        
        // Fetch the results as an associative array
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
            
    }


    public function getcancelledPayments(){
        $sql = "SELECT 
                p.tx_id,
                p.id AS sale_id,
                p.datecaptured AS sale_date,
                p.amount AS total_sales_amount,
                p.cancel_state AS sale_status,
                CASE
                    WHEN p.cancel_state = 1 THEN 'Cancelled Payment'
                    WHEN p.cancel_state = 0 THEN 'Reapproved Payment'
                END AS payment_status,
                c.customer_name,
                p.payment_method,
                p.transction_id,
                p.payment_type,
                p.receipt_number,
                cp.transction_id as original_transaction_id,
                emp.fullname as cancelled_by_operator,
                p.reason
            FROM 
                cancelled_payments p
            JOIN 
                customers c ON p.customer = c.customer_id
            LEFT JOIN
                cancelled_payments cp ON p.receipt_number = cp.receipt_number
            LEFT JOIN
                employees emp ON p.cancelled_by = emp.employee_code
            WHERE
                p.cancel_state = 1
            ORDER BY 
                p.cancel_state DESC,
                p.datecaptured DESC";
        // Prepare and execute the query
        $stmt = $this->link->prepare($sql);
        $stmt->execute();
        
        // Fetch the results as an associative array
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
            
    }
    
    public function getMonthlySales() {
        // SQL query to select weekly sales data (from last Monday until today)
        $sql ="SELECT 
                p.id AS sale_id,
                p.datecaptured AS sale_date,
                p.amount_paid AS total_sales_amount,
                p.payment_state AS sale_status,
                CASE
                    WHEN p.payment_state = 1 THEN 'Completed'
                    WHEN p.payment_state = 0 THEN 'Cancelled Payment'
                END AS payment_status,
                c.customer_name,
                p.payment_method,
                p.transction_id,
                p.payment_type,
                p.receipt_number
            FROM 
                payments p
            JOIN 
                customers c ON p.customer = c.customer_id
            WHERE 
                MONTH(p.datecaptured) = MONTH(NOW()) 
                AND YEAR(p.datecaptured) = YEAR(NOW())
                AND p.payment_state = 1
        ";

        // Prepare and execute the query
        $stmt = $this->link->prepare($sql);
        $stmt->execute();
        
        // Fetch the results as an associative array
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getInvoiceDetails($invoice_number){
        $sql = "SELECT
                c.customer_name,
                so.order_number,
                so.customer,
                so.orderdate,
                so.due_date,
                so.datecaptured,
                so.subtotal,
                so.total_tax,
                so.total_bill,
                so.order_state
            FROM 
                saved_orders so
            LEFT JOIN
                customers c ON so.customer = c.customer_id
            WHERE 
                so.order_number = :invoice_number
            ";
        $smtp = $this->link->prepare($sql);
        $smtp->bindParam(':invoice_number', $invoice_number);
        $smtp->execute();
        $data = $smtp->fetch();

        return $data;
    }

    public function updateSaleStatus($sale_id, $status) {
        $sql = "UPDATE payments SET transction_id = :sale_id, payment_state = :status WHERE id = :sale_id";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':sale_id', $sale_id, PDO::PARAM_INT);        
        $stmt->bindParam(':status', $status, PDO::PARAM_STR);        

        return $stmt->execute();
    }

    public function restoreSaleStatusAndTransaction($sale_id, $status, $original_transaction_id) {
        $sql = "UPDATE payments 
                SET transction_id = :transction_id, payment_state = :status 
                WHERE id = :sale_id";
        
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':transction_id', $original_transaction_id, PDO::PARAM_STR);
        $stmt->bindParam(':status', $status, PDO::PARAM_INT);
        $stmt->bindParam(':sale_id', $sale_id, PDO::PARAM_INT);

        return $stmt->execute();
    }


    public function deleteCancelledPayment($receipt_number) {
        $sql = "DELETE FROM cancelled_payments WHERE receipt_number = :receipt_number";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':receipt_number', $receipt_number, PDO::PARAM_STR);
        return $stmt->execute();
    }



    public function getSalesForTopSellingProduct($product_code) {
        // SQL query to select all sales for the top-selling product
        $sql = "    SELECT 
                        s.sales_number AS sale_id,
                        s.date_captured AS sale_date,
                        s.total AS sale_amount,
                        s.selling_price,
                        s.type,
                        s.quantity,
                        s.status AS sale_state,
                        p.product_name,
                        CASE
                            WHEN s.type = 'Cash Sale' THEN c.customer_name
                            WHEN s.type = 'Customer Order' THEN so_c.customer_name
                            ELSE 'Unknown'
                        END AS customer_name,
                        
                        CASE
                            WHEN s.status = 1 THEN 'Completed Sale'
                            WHEN s.type = 'Customer Order' THEN so_c.customer_name
                            ELSE 'Unknown'
                        END AS sale_status
                    FROM 
                        sales_order s
                    JOIN 
                        products p ON s.product_code = p.product_id
                    LEFT JOIN 
                        payments pay ON s.sales_number = pay.sale_id AND s.type = 'Cash Sale'
                    LEFT JOIN 
                        saved_orders so ON s.sales_number = so.order_number AND s.type = 'Customer Order'
                    LEFT JOIN 
                        customers c ON pay.customer = c.customer_id  -- For Cash Sale, join with payments table
                    LEFT JOIN 
                        customers so_c ON so_c.customer_id = so.customer  -- For Order Sale, join with saved_orders table
                    WHERE 
                        p.product_id = :product_code
                        AND MONTH(s.date_captured) = MONTH(NOW())
                        AND YEAR(s.date_captured) = YEAR(NOW())
                    AND 
                        s.status = 1  -- Successful payments only
                    ORDER BY 
                        s.quantity DESC";

        // Prepare and execute the query
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':product_code', $product_code, PDO::PARAM_INT); // Bind the product code parameter
        $stmt->execute();
        
        // Fetch the results as an associative array
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    public function getReceiptData($id) {
        $sql = "SELECT p.*, 
                c.customer_name,
                p.payment_state,
                p.sale_id
            FROM 
                payments p 
            LEFT JOIN
                customers c ON p.customer = c.customer_id
            WHERE 
                p.sale_id = :id";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':id', $id, PDO::PARAM_STR);
    
        if ($stmt->execute()) {
            return $stmt->fetch(PDO::FETCH_ASSOC); // return single row as associative array
        } else {
            return false;
        }
    }

    public function getReceiptItems($sales_number) {
        $sql = "SELECT *,
                p.product_name as description,
                so.selling_price,
                so.total,
                so.subtotal
            FROM 
                sales_order so
            LEFT JOIN
                products p ON p.product_id = so.product_code
            WHERE 
                sales_number = :sales_number";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':sales_number', $sales_number, PDO::PARAM_STR);
    
        if ($stmt->execute()) {
            return $stmt->fetchAll(PDO::FETCH_ASSOC); // return all matching rows
        } else {
            return false;
        }
    }

    public function getIncompleteTransactions() {
        $sql = "SELECT 
                so.sales_number AS transaction_id,
                so.type AS transaction_type,
                so.date_captured,
                so.total AS total_amount,
                emp.fullname AS operator_name,
                CASE 
                    WHEN py.sale_id IS NULL THEN 'Incomplete Cash Sale'
                    WHEN py.sale_id IS NOT NULL AND py.payment_state = 0 THEN 'Cancelled Payment'
                    ELSE 'Unknown'
                END AS transaction_status
            FROM 
                sales_order so
            LEFT JOIN 
                payments py ON so.sales_number = py.sale_id
            LEFT JOIN
                employees emp ON so.operator = emp.employee_code
            WHERE 
                so.type = 'Cash Sale'
                AND (
                    py.sale_id IS NULL 
                    OR (py.sale_id IS NOT NULL AND py.payment_state != 1)
                )

            UNION

            SELECT 
                so.sales_number AS transaction_id,
                so.type AS transaction_type,
                so.date_captured,
                so.total AS total_amount,
                emp.fullname AS operator_name,
                'Unsaved Invoice' AS transaction_status
            FROM 
                sales_order so
            LEFT JOIN 
                saved_orders sa ON so.sales_number = sa.order_number
            LEFT JOIN
                employees emp ON so.operator = emp.employee_code
            WHERE 
                so.type = 'Customer Order'
                AND sa.order_number IS NULL

            ORDER BY 
                date_captured ASC
        ";

        $stmt = $this->link->prepare($sql);
        $stmt->execute();
        $transactions = $stmt->fetchAll(PDO::FETCH_ASSOC);

        // Calculate the total amount
        $totalAmount = array_sum(array_column($transactions, 'total_amount'));

        return [
            'transactions' => $transactions,
            'total_amount' => $totalAmount
        ];
    }

    // In sales_model.php
    public function reverseCancelledPayment($receipt_number, $original_transaction_id, $payment_state) {
        $sql = "UPDATE payments SET payment_state = :payment_state, transction_id = :original_transaction_id WHERE receipt_number = :receipt_number";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':receipt_number', $receipt_number);
        $stmt->bindParam(':original_transaction_id', $original_transaction_id);
        $stmt->bindParam(':payment_state', $payment_state);
        return $stmt->execute();
    }

    public function getCancelledPaymentData($receipt_number) {
        $sql = "SELECT * FROM cancelled_payments WHERE receipt_number = :receipt_number LIMIT 1";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':receipt_number', $receipt_number);
        $stmt->execute();
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
    
    public function getCurrentDetailsBeforeupdate($receipt_number) {
        $sql = "SELECT * FROM payments WHERE receipt_number = :receipt_number LIMIT 1";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':receipt_number', $receipt_number);
        $stmt->execute();
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    public function cancelcancelledRecord($receipt_number, $state, $reason){
        $sql = "UPDATE cancelled_payments SET cancel_state =:state, reason =:reason WHERE receipt_number=:receipt_number";
        $smtp = $this->link->prepare($sql);
        $smtp->bindParam(':receipt_number', $receipt_number);
        $smtp->bindParam(':reason', $reason);
        $smtp->bindParam(':state', $state);
        return $smtp->execute();    
    }

    public function updateInvoiceBalance($invoice_id, $balance) {
        $sql = "UPDATE payments SET balance = :balance WHERE sale_id = :invoice_id";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':balance', $balance);
        $stmt->bindParam(':invoice_id', $invoice_id);
        return $stmt->execute();
    }

    public function checkCancelledPayment($receipt_number) {
        $sql = "SELECT * FROM cancelled_payments WHERE receipt_number = :receipt_number ";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':receipt_number', $receipt_number, PDO::PARAM_STR);
        $stmt->execute();
        return $stmt->fetch(PDO::FETCH_ASSOC); 
    }

    public function getSalesTransactionsByDate($start_date, $end_date){
        $sql = "SELECT
                    MIN(p.tx_id) AS unique_tx,
                    p.sale_id AS transaction_number,
                    TIMESTAMP(
                        p.payment_date,
                        TIME(p.datecaptured)
                    ) AS sale_date,
                    c.customer_name AS customer,
                    COALESCE(w.warehouse_name, b.branch_name) AS location_from,
                    'Cash Sale' AS transaction_type,

                    so_totals.total_amount AS sales_amount,
                    SUM(p.amount_paid) AS total_paid,
                    MAX(p.discount) AS discount,

                    ROUND(
                        so_totals.total_amount 
                        - (SUM(p.amount_paid) + MAX(p.discount)),
                        2
                    ) AS balance,

                    /* 🔑 Backdated date + real capture time */
                    TIMESTAMP(
                        p.payment_date,
                        TIME(p.datecaptured)
                    ) AS transaction_datetime

                FROM 
                    payments p
                LEFT JOIN 
                    customers c ON p.customer = c.customer_id
                LEFT JOIN 
                    warehouses w ON p.location = w.warehouse_id
                LEFT JOIN 
                    branches b ON p.location = b.branch_id

                /* Get TOTAL from sales_order */
                LEFT JOIN (
                    SELECT
                        sales_number,
                        SUM(total) AS total_amount
                    FROM 
                        sales_order
                    GROUP BY 
                        sales_number
                    ) so_totals ON so_totals.sales_number = p.sale_id

                WHERE
                    p.payment_state = 1
                    AND p.payment_type = 'Cash Sale'
                    AND DATE(p.payment_date) BETWEEN DATE(:start_date) AND DATE(:end_date)

                GROUP BY
                    p.sale_id

                    /* ============================
                    CASH SALES
                    ============================ */
            UNION ALL


                    /* ============================
                    CUSTOMER INVOICES
                    ============================ */
                SELECT
                    so.id AS unique_tx,
                    so.order_number AS transaction_number,
                    TIMESTAMP(
                        so.orderdate,
                        TIME(so.datecaptured)
                    ) AS sale_date,
                    c.customer_name AS customer,
                    COALESCE(w.warehouse_name, b.branch_name) AS location_from,
                    'Customer Invoice' AS transaction_type,

                    so.total_bill AS sales_amount,
                    COALESCE(ip.total_paid, 0) AS total_paid,
                    0 AS discount,

                    ROUND(
                        so.total_bill - COALESCE(ip.total_paid, 0),
                        2
                    ) AS balance,

                    /* Use invoice capture timestamp */
                    so.datecaptured AS transaction_datetime

                FROM 
                    saved_orders so
                LEFT JOIN 
                    customers c ON so.customer = c.customer_id
                LEFT JOIN 
                    warehouses w ON so.invoice_location = w.warehouse_id
                LEFT JOIN 
                    branches b ON so.invoice_location = b.branch_id
                LEFT JOIN (
                    SELECT
                        invoice_number,
                        SUM(amount_paid) AS total_paid
                    FROM invoice_payments
                    WHERE payment_state = 1
                    GROUP BY invoice_number
                ) ip ON ip.invoice_number = so.order_number

                WHERE
                    so.order_state IN (0,1,2)
                    AND DATE(so.orderdate) BETWEEN DATE(:start_date) AND DATE(:end_date)
                ORDER BY 
                    transaction_datetime DESC
        ";

        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':start_date', $start_date, PDO::PARAM_STR);
        $stmt->bindParam(':end_date', $end_date, PDO::PARAM_STR);
        $stmt->execute();

        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }



    public function getPaymentsByDate($start_date, $end_date){
        $sql = "SELECT
                    p.tx_id,
                    p.sale_id as transaction_number,
                    p.receipt_number,
                    p.id,
                    COALESCE(w.warehouse_name, b.branch_name) as location_from,
                    p.payment_method,
                    p.transction_id,
                    p.payment_date as sale_date,
                    cc.customer_name as customer,
                    CASE 
                        WHEN p.payment_type = 'Cash Sale' THEN
                        'Cash Sale'
                        WHEN p.payment_type = 'Order Payment' THEN
                        'Customer Invoice Payment'
                    END as transaction_type,
                    p.amount_paid as sales_amount,
                    p.datecaptured as time
                FROM
                    payments p
                LEFT JOIN
                    customers cc ON p.customer = cc.customer_id
                LEFT JOIN
                    warehouses w ON p.location = w.warehouse_id
                LEFT JOIN
                    branches b ON p.location = b.branch_id
                WHERE
                    p.payment_state = 1
                    AND DATE(p.payment_date) >= :start_date
                    AND DATE(p.payment_date) <= :end_date
                ORDER BY
                    sale_date DESC";
        $smtp = $this->link->prepare($sql);
        $smtp->bindParam(':start_date', $start_date, PDO::PARAM_STR);
        $smtp->bindParam(':end_date', $end_date, PDO::PARAM_STR);
        $smtp->execute();
        $payments = $smtp->fetchAll(PDO::FETCH_ASSOC);
        return $payments;
    }

    public function creditorsTransactions($customer){
        $sql = "SELECT 
                    t.date,
                    t.reference,
                    t.description,
                    t.debit,
                    t.credit
                FROM (
                    /* INVOICES (DEBITS) */
                    SELECT
                        so.orderdate AS date,
                        so.order_number AS reference,
                        'Credit Invoice' AS description,
                        so.total_bill AS debit,
                        0 AS credit
                    FROM saved_orders so
                    WHERE 
                        so.customer = :customer
                        AND so.invoice_type = 'credit'

                    UNION ALL

                    /* PAYMENTS (CREDITS) */
                    SELECT
                        ip.date_paid AS date,
                        ip.invoice_number AS reference,
                        'Invoice Payment' AS description,
                        0 AS debit,
                        ip.amount_paid AS credit
                    FROM 
                        invoice_payments ip
                    INNER JOIN 
                        saved_orders so ON so.order_number = ip.invoice_number
                    WHERE 
                        so.customer = :customer
                ) t
                ORDER BY t.date ASC, t.reference ASC
            ";

        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':customer', $customer, PDO::PARAM_INT);
        $stmt->execute();

        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getCustomerInvoiceAging($customer){
        $sql = "SELECT
                    bucket,
                    SUM(balance) AS amount
                FROM (
                    SELECT
                        so.order_number,
                        so.orderdate,
                        so.total_bill - IFNULL(SUM(ip.amount_paid), 0) AS balance,
                        DATEDIFF(CURDATE(), so.orderdate) AS age_days,
                        CASE
                            WHEN DATEDIFF(CURDATE(), so.orderdate) <= 30 THEN '0-30 Days'
                            WHEN DATEDIFF(CURDATE(), so.orderdate) BETWEEN 31 AND 60 THEN '31-60 Days'
                            WHEN DATEDIFF(CURDATE(), so.orderdate) BETWEEN 61 AND 90 THEN '61-90 Days'
                            ELSE '90+ Days'
                        END AS bucket
                    FROM 
                        saved_orders so
                    LEFT JOIN 
                        invoice_payments ip ON so.order_number = ip.invoice_number
                    WHERE
                        so.customer = :customer
                        AND so.invoice_type = 'credit'
                    GROUP BY
                        so.order_number,
                        so.orderdate,
                        so.total_bill
                    ) aging
                WHERE 
                    balance > 0
                GROUP BY 
                    bucket
                ORDER BY
                    FIELD(bucket, '0-30 Days', '31-60 Days', '61-90 Days', '90+ Days')";

        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':customer', $customer, PDO::PARAM_INT);
        $stmt->execute();

        return $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
    }

    /**
     * Compute the overall payment behavior for a credit customer
     *
     * @param int $customer_id
     * @return array ['status' => string, 'bootstrap_class' => string]
     */
    public function creditCustomerBehavior(int $customer_id): array{
        $today = new DateTime();

        // 1) Fetch all credit invoices for the customer
        $sql = "SELECT so.order_number, so.due_date, so.order_state,
                    MAX(ip.date_paid) AS last_payment_date,
                    IFNULL(SUM(ip.amount_paid),0) AS total_paid,
                    so.total_bill
                FROM saved_orders so
                LEFT JOIN invoice_payments ip 
                    ON so.order_number = ip.invoice_number 
                    AND ip.payment_state = 1
                WHERE so.customer = :customer
                AND so.invoice_type = 'credit'
                GROUP BY so.order_number, so.due_date, so.order_state, so.total_bill";

        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':customer', $customer_id, PDO::PARAM_INT);
        $stmt->execute();
        $invoices = $stmt->fetchAll(PDO::FETCH_ASSOC);

        if (empty($invoices)) {
            return [
                'status' => 'NO CREDIT HISTORY',
                'bootstrap_class' => 'default'
            ];
        }

        $categories = [];

        foreach ($invoices as $inv) {
            $due_date = new DateTime($inv['due_date']);
            $last_paid = $inv['last_payment_date'] ? new DateTime($inv['last_payment_date']) : null;

            switch ($inv['order_state']) {
                case 2: // fully paid
                    if (!$last_paid) {
                        $categories[] = 'VERY BAD';
                        break;
                    }

                    $days_late = (int)$last_paid->diff($due_date)->format('%r%a');
                    if ($days_late <= 0) {
                        $categories[] = 'VERY GOOD';
                    } elseif ($days_late <= 7) {
                        $categories[] = 'GOOD';
                    } elseif ($days_late <= 21) {
                        $categories[] = 'FAIR PAYER';
                    } elseif ($days_late <= 45) {
                        $categories[] = 'BAD';
                    } else {
                        $categories[] = 'VERY BAD';
                    }
                    break;

                case 0: // unpaid
                case 1: // partially paid
                    if ($today > $due_date) {
                        // Invoice past due
                        if ($inv['total_paid'] == 0) {
                            $categories[] = 'VERY BAD';
                        } else {
                            $categories[] = 'BAD';
                        }
                    } else {
                        // Not yet due
                        $categories[] = 'WAITING FOR PAYMENT';
                    }
                    break;

                case 3: // cancelled
                default:
                    // Ignore cancelled invoices
                    break;
            }
        }

        if (empty($categories)) {
            return [
                'status' => 'WAITING FOR PAYMENT',
                'bootstrap_class' => 'default'
            ];
        }

        // Majority rule
        $counts = array_count_values($categories);

        if (isset($counts['VERY BAD']) && $counts['VERY BAD'] > 0) {
            $status = 'VERY BAD';
            $class = 'danger';
        } elseif (isset($counts['BAD']) && $counts['BAD'] > 0) {
            $status = 'BAD';
            $class = 'text-danger';
        } elseif (isset($counts['FAIR PAYER']) && $counts['FAIR PAYER'] > 0) {
            $status = 'FAIR PAYER';
            $class = 'warning';
        } elseif (isset($counts['GOOD']) && $counts['GOOD'] > 0) {
            $status = 'GOOD';
            $class = 'info';
        } elseif (isset($counts['VERY GOOD']) && $counts['VERY GOOD'] > 0) {
            $status = 'VERY GOOD';
            $class = 'success';
        } else {
            $status = 'WAITING FOR PAYMENT';
            $class = 'default';
        }

        return [
            'status' => $status,
            'bootstrap_class' => $class
        ];
    }

}