<?php

include_once 'config.php';

class transfers_model
{
    public $link;

    public function __construct()
    {
        $db_connection = new dbConnection();
        $this->link = $db_connection->connect();
    }

    /**
     * Check if a waybill number already exists
     * 
     * @param string $waybill_no
     * @return bool
     */
    public function waybillExists(string $waybill_no): bool {
        $sql = "SELECT COUNT(*) as count FROM third_party_shipment_final WHERE waybill_number = :waybill_no";
        $stmt = $this->link->prepare($sql);
        $stmt->execute(['waybill_no' => $waybill_no]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);

        return ($result['count'] ?? 0) > 0;
    }

    /**
     * Check if a reference/document number already exists
     * 
     * @param string $reference_no
     * @return bool
     */
    public function referenceExists(string $reference_no): bool {
        $sql = "SELECT COUNT(*) as count FROM third_party_shipment_final WHERE reference_no = :reference_no";
        $stmt = $this->link->prepare($sql);
        $stmt->execute(['reference_no' => $reference_no]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);

        return ($result['count'] ?? 0) > 0;
    }

    /**
     * Create a new supplier shipment
     * 
     * @param array $data
     * @return int|false  Returns the inserted shipment ID or false on failure
     */
    public function createSupplierShipment(array $data) {
        $sql = "INSERT INTO third_party_shipment_final 
                (reference_no, waybill_number, supplier, transport_mode, shipment_date, third_party_receiver, num_items, currency, total_cost, datecaptured, operator, transfer_state) 
                VALUES 
                (:reference_no, :waybill_number, :supplier, :transport_mode, :shipment_date, :third_party_receiver, :num_items, :currency, :total_cost, :datecaptured, :operator, 2)";
        
        $stmt = $this->link->prepare($sql);

        $success = $stmt->execute([
            'reference_no'        => $data['reference_no'],
            'waybill_number'      => $data['waybill_no'] ?? null,           // optional
            'supplier'            => $data['supplier_id'],
            'transport_mode'      => $data['transport_mode'],
            'shipment_date'       => $data['shipment_date'],
            'third_party_receiver'=> $data['receiver_id'],
            'num_items'           => $data['num_items'] ?? 0,              // default to 0 if not provided
            'currency'            => $data['currency'] ?? 'USD',           // default currency
            'total_cost'          => $data['total_cost'] ?? 0,             // default to 0 if not provided
            'datecaptured'        => $data['datecaptured'] ?? date('Y-m-d H:i:s'), // default to current timestamp
            'operator'            => $data['operator']
        ]);

        if ($success) {
            return (int) $this->link->lastInsertId();
        }

        return false; // insertion failed
    }

    public function headerDetails($id){
        $sql = "SELECT
                    tpsf.reference_no,
                    tpsf.waybill_number waybill_no,
                    tpsf.shipment_date,
                    tpsf.transport_mode,
                    s.supplier_name,
                    tpc.company_name as receiver_name,
                    tpsf.transfer_state,
                    COALESCE(emp.fullname, 'Super User') as operator_name
                FROM 
                    third_party_shipment_final tpsf
                JOIN
                    suppliers s ON tpsf.supplier = s.supplier_id
                JOIN
                    third_party_companies tpc ON tpsf.third_party_receiver = tpc.company_id
                LEFT JOIN
                    employees emp ON tpsf.operator = emp.employee_code
                WHERE
                     tpsf.transfer_id =:id";
            $stmt = $this->link->prepare($sql);
            $stmt->bindParam(':id', $id, PDO::PARAM_INT);
            $stmt->execute();
            $header = $stmt->fetch(PDO::FETCH_ASSOC);
            return $header;
    }

    public function getShipmentTransferItems($id){
        $sql = "SELECT 
                    p.product_name,
                    sti.total_cost,
                    sti.opening_balance,
                    sti.quantity,
                    sti.order_price,
                    sti.closing_stock
                FROM
                    stock_transfer_in sti
                LEFT JOIN
                    products p ON sti.product_id = p.product_id
                WHERE
                    transfer_id =:id";
            $smt = $this->link->prepare($sql);
            $smt->bindParam(':id', $id, PDO::PARAM_INT);
            $smt->execute();
            $items = $smt->fetchAll(PDO::FETCH_ASSOC);
            return $items;
    }

    public function getTransfers($start, $end) {
        $sql = "SELECT 
                    st.transfer_id,
                    s.supplier_name,
                    tpc.company_name as receiver_name,
                    st.shipment_date,
                    st.transfer_state,
                    st.waybill_number,
                    st.reference_no,
                    st.num_items,
                    transport_mode,
                    CASE
                        WHEN st.transfer_state = 0 THEN 'Cancelled Shipment'
                        WHEN st.transfer_state = 1 THEN 'Pending Approval'
                        WHEN st.transfer_state = 2 THEN 'Completed Shipment'
                    END as transfer_status
                FROM 
                    third_party_shipment_final st
                JOIN
                    suppliers s ON st.supplier = s.supplier_id
                JOIN
                    third_party_companies tpc ON st.third_party_receiver = tpc.company_id
                WHERE 
                    DATE(st.shipment_date) BETWEEN :start AND :end
                ORDER BY 
                    st.shipment_date DESC";

        $stmt = $this->link->prepare($sql);
        $stmt->execute(['start' => $start, 'end' => $end]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }


    public function addShipmentItems($shipment_id, $supplier_id, $receiver, $items, $quantities, $prices, $operator) {
        $total_shipment_cost = 0;
        $insertSql = "INSERT INTO stock_transfer_in(transfer_id, supplier, company_id, product_id, opening_balance, quantity, approved_quantity, stock_difference, closing_stock, order_price, total_cost, operator, transfer_state)
        VALUES
        (:transfer_id, :supplier, :company_id, :product_id, :opening_balance, :quantity, :approved_quantity, :stock_difference, :closing_stock, :order_price, :total_cost, :operator, 2)";

        $insertStmt = $this->link->prepare($insertSql);

        $updateStockStmt = $this->link->prepare("UPDATE third_party_products SET quantity = :quantity WHERE company_id = :company_id AND product_id = :product_id");

        foreach ($items as $index => $product_id) {

            $quantity    = (float) ($quantities[$index] ?? 0);
            $order_price = (float) ($prices[$index] ?? 0);

            if ($quantity <= 0) {
                continue;
            }

            $line_total = $quantity * $order_price;

            // Current stock
            $opening_balance = $this->getOpeningBalance($receiver, $product_id);
            $closing_stock   = $opening_balance + $quantity;

            // Insert shipment line
            $insertStmt->execute(['transfer_id' => $shipment_id, 'transfer_id' => $shipment_id, 'supplier' => $supplier_id, 'company_id' => $receiver, 'product_id' => $product_id, 'opening_balance'   => $opening_balance, 'quantity' => $quantity, 'approved_quantity' => $quantity, 'stock_difference'  => 0, 'closing_stock'     => $closing_stock, 'order_price' => $order_price, 'total_cost' => $line_total, 'operator' => $operator]);

            // ✅ UPDATE THIRD PARTY STOCK
            $updateStockStmt->execute(['quantity' => $closing_stock, 'company_id' => $receiver, 'product_id' => $product_id]);
            $total_shipment_cost += $line_total;
        }

        // Update shipment header total
        $updateHeader = $this->link->prepare("UPDATE third_party_shipment_final SET total_cost = :total_cost WHERE transfer_id = :shipment_id ");

        $updateHeader->execute(['total_cost'  => $total_shipment_cost, 'shipment_id' => $shipment_id]);
        return true;
    }


    public function getOpeningBalance($company_id, $product_id) {
        $stmt = $this->link->prepare("
            SELECT quantity 
            FROM third_party_products 
            WHERE company_id = :company_id AND product_id = :product_id
            LIMIT 1
        ");
        $stmt->execute([
            'company_id' => $company_id,
            'product_id' => $product_id
        ]);

        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row['quantity'] ?? 0; // return 0 if not found
    }






    public function add_stock_warehouse($transaction, $product_id, $stock_from, $location_type_from,  $stock_to, $location_type_to, $movement, $opening_stock, $quantity, $closing_stock, $transaction_date, $datecaptured, $operator, $movement_state) {
        
        $sql = "INSERT INTO warehouse_stock_movement (transaction, product_id, stock_from, location_type_from, stock_to, location_type_to, movement, opening_stock, quantity, closing_stock, transaction_date, datecaptured, operator, movement_state) 
        VALUES (:transaction, :product_id, :stock_from, :location_type_from, :stock_to, :location_type_to, :movement, :opening_stock, :quantity, :closing_stock, :transaction_date, :datecaptured, :operator, :movement_state)";
    
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':transaction', $transaction, PDO::PARAM_STR);
        $stmt->bindParam(':product_id', $product_id, PDO::PARAM_INT);
        $stmt->bindParam(':stock_from', $stock_from, PDO::PARAM_INT);
        $stmt->bindParam(':location_type_from', $location_type_from, PDO::PARAM_STR);
        $stmt->bindParam(':stock_to', $stock_to, PDO::PARAM_INT);
        $stmt->bindParam(':location_type_to', $location_type_to, PDO::PARAM_STR);
        $stmt->bindParam(':movement', $movement, PDO::PARAM_STR);
        $stmt->bindParam(':opening_stock', $opening_stock, PDO::PARAM_STR);
        $stmt->bindParam(':quantity', $quantity, PDO::PARAM_STR);
        $stmt->bindParam(':closing_stock', $closing_stock, PDO::PARAM_STR);
        $stmt->bindParam(':transaction_date', $transaction_date, PDO::PARAM_STR);
        $stmt->bindParam(':datecaptured', $datecaptured, PDO::PARAM_STR);
        $stmt->bindParam(':operator', $operator, PDO::PARAM_STR);
        $stmt->bindParam(':movement_state', $movement_state, PDO::PARAM_INT);
        return $stmt->execute();
    }

    public function add_stock_branch($transaction, $product_id, $stock_from, $location_type_from,  $stock_to, $location_type_to, $movement, $opening_stock, $quantity, $closing_stock, $transaction_date, $datecaptured, $operator, $movement_state) {
        
        $sql = "INSERT INTO branch_stock_movement (transaction, product_id, stock_from, location_type_from, stock_to, location_type_to, movement, opening_stock, quantity, closing_stock, transaction_date, datecaptured, operator, movement_state) 
        VALUES (:transaction, :product_id, :stock_from, :location_type_from, :stock_to, :location_type_to, :movement, :opening_stock, :quantity, :closing_stock, :transaction_date, :datecaptured, :operator, :movement_state)";
    
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':transaction', $transaction, PDO::PARAM_STR);
        $stmt->bindParam(':product_id', $product_id, PDO::PARAM_INT);
        $stmt->bindParam(':stock_from', $stock_from, PDO::PARAM_INT);
        $stmt->bindParam(':location_type_from', $location_type_from, PDO::PARAM_STR);
        $stmt->bindParam(':stock_to', $stock_to, PDO::PARAM_INT);
        $stmt->bindParam(':location_type_to', $location_type_to, PDO::PARAM_STR);
        $stmt->bindParam(':movement', $movement, PDO::PARAM_STR);
        $stmt->bindParam(':opening_stock', $opening_stock, PDO::PARAM_STR);
        $stmt->bindParam(':quantity', $quantity, PDO::PARAM_STR);
        $stmt->bindParam(':closing_stock', $closing_stock, PDO::PARAM_STR);
        $stmt->bindParam(':transaction_date', $transaction_date, PDO::PARAM_STR);
        $stmt->bindParam(':datecaptured', $datecaptured, PDO::PARAM_STR);
        $stmt->bindParam(':operator', $operator, PDO::PARAM_STR);
        $stmt->bindParam(':movement_state', $movement_state, PDO::PARAM_INT);
        return $stmt->execute();
    }

    public function process_stock_transfer($order_number, $product_code, $location_from_id, $final_destination_id, $movement_out, $movement_in, $remaining, $quantity, $closing_stock_out, $remaining_income, $closing_stock_in, $transaction_date, $datecaptured, $operator, $movement_state, $location_from_type, $location_to_type) {
        // Start a transaction
        $this->link->beginTransaction();

        try {
            // Process stock removal
            $process_remove = false;
            if ($location_from_type === 'W') {
                $process_remove = $this->remove_stock_warehouse($order_number, $product_code, $location_from_id, $location_from_type, $final_destination_id, $location_to_type, $movement_out, $remaining, $quantity, $closing_stock_out, $transaction_date, $datecaptured, $operator, $movement_state);
            } else if ($location_from_type === 'B') {
                $process_remove = $this->remove_stock_branch($order_number, $product_code, $location_from_id, $location_from_type, $final_destination_id, $location_to_type, $movement_out, $remaining, $quantity, $closing_stock_out, $transaction_date, $datecaptured, $operator, $movement_state);
            } else {
                throw new Exception('Invalid Location Type');
            }

            if (!$process_remove) {
                throw new Exception('Failed to remove stock');
            }

            // Process stock addition
            $process_addstock = false;
            if ($location_to_type === 'W') {
                $process_addstock = $this->add_stock_warehouse($order_number, $product_code, $location_from_id, $location_from_type, $final_destination_id, $location_to_type, $movement_in, $remaining_income, $quantity, $closing_stock_in, $transaction_date, $datecaptured, $operator, $movement_state);
            } else if ($location_to_type === 'B') {
                $process_addstock = $this->add_stock_branch($order_number, $product_code, $location_from_id, $location_from_type, $final_destination_id, $location_to_type, $movement_in, $remaining_income, $quantity, $closing_stock_in, $transaction_date, $datecaptured, $operator, $movement_state);
            } else {
                throw new Exception('Invalid Destination of Selected Location');
            }

            if (!$process_addstock) {
                throw new Exception('Failed to add stock');
            }
 
            // If both processes succeed, commit the transaction
            $this->link->commit();
            return true; // Successful transfer

        } catch (Exception $e) {
            // Rollback if any error occurs
            $this->link->rollBack();
            return false; // Failed transfer
        }
    }  

    public function remove_stock_branch($transaction, $product_id, $stock_from, $location_type_from, $stock_to, $location_type_to, $movement, $opening_stock, $quantity, $closing_stock, $transaction_date, $datecaptured, $operator, $movement_state) {
        
        $sql = "INSERT INTO branch_stock_movement (transaction, product_id, stock_from, location_type_from, stock_to, location_type_to, movement, opening_stock, quantity, closing_stock, transaction_date, datecaptured, operator, movement_state) 
        VALUES (:transaction, :product_id, :stock_from, :location_type_from, :stock_to, :location_type_to, :movement, :opening_stock, :quantity, :closing_stock, :transaction_date, :datecaptured, :operator, movement_state)";
    
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':transaction', $transaction, PDO::PARAM_STR);
        $stmt->bindParam(':product_id', $product_id, PDO::PARAM_INT);
        $stmt->bindParam(':stock_from', $stock_from, PDO::PARAM_INT);
        $stmt->bindParam(':location_type_from', $location_type_from, PDO::PARAM_STR);
        $stmt->bindParam(':stock_to', $stock_to, PDO::PARAM_INT);
        $stmt->bindParam(':location_type_to', $location_type_to, PDO::PARAM_STR);
        $stmt->bindParam(':movement', $movement, PDO::PARAM_STR);
        $stmt->bindParam(':opening_stock', $opening_stock, PDO::PARAM_STR);
        $stmt->bindParam(':quantity', $quantity, PDO::PARAM_STR);
        $stmt->bindParam(':closing_stock', $closing_stock, PDO::PARAM_STR);
        $stmt->bindParam(':transaction_date', $transaction_date, PDO::PARAM_STR);
        $stmt->bindParam(':datecaptured', $datecaptured, PDO::PARAM_STR);
        $stmt->bindParam(':operator', $operator, PDO::PARAM_STR);
        $stmt->bindParam(':movement_state', $movement_state, PDO::PARAM_INT);
        return $stmt->execute();
    }

    public function remove_stock_warehouse($transaction, $product_id, $stock_from, $location_type_from, $stock_to, $location_type_to, $movement, $opening_stock, $quantity, $closing_stock, $transaction_date, $datecaptured, $operator, $movement_state) {
        
        $sql = "INSERT INTO warehouse_stock_movement (transaction, product_id, stock_from, location_type_from, stock_to, location_type_to, movement, opening_stock, quantity, closing_stock, transaction_date, datecaptured, operator, movement_state) 
        VALUES (:transaction, :product_id, :stock_from, :location_type_from, :stock_to, :location_type_to, :movement, :opening_stock, :quantity, :closing_stock, :transaction_date, :datecaptured, :operator, :movement_state)";
    
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':transaction', $transaction, PDO::PARAM_STR);
        $stmt->bindParam(':product_id', $product_id, PDO::PARAM_INT);
        $stmt->bindParam(':stock_from', $stock_from, PDO::PARAM_INT);
        $stmt->bindParam(':location_type_from', $location_type_from, PDO::PARAM_STR);
        $stmt->bindParam(':stock_to', $stock_to, PDO::PARAM_INT);
        $stmt->bindParam(':location_type_to', $location_type_to, PDO::PARAM_STR);
        $stmt->bindParam(':movement', $movement, PDO::PARAM_STR);
        $stmt->bindParam(':opening_stock', $opening_stock, PDO::PARAM_STR);
        $stmt->bindParam(':quantity', $quantity, PDO::PARAM_STR);
        $stmt->bindParam(':closing_stock', $closing_stock, PDO::PARAM_STR);
        $stmt->bindParam(':transaction_date', $transaction_date, PDO::PARAM_STR);
        $stmt->bindParam(':datecaptured', $datecaptured, PDO::PARAM_STR);
        $stmt->bindParam(':operator', $operator, PDO::PARAM_INT);
        $stmt->bindParam(':movement_state', $movement_state, PDO::PARAM_INT);
        return $stmt->execute();
    } 

    public function productExistsInLocation($product_id, $location_id, $location_type) {
        if ($location_type === 'B') {
            $sql = "SELECT COUNT(*) FROM branch_products WHERE product_id = :product_id AND branch_id = :location_id";
        } elseif ($location_type === 'W') {
            $sql = "SELECT COUNT(*) FROM warehouse_products WHERE product_id = :product_id AND warehouse_id = :location_id";
        } else {
            return false;
        }

        $stmt = $this->link->prepare($sql);
        $stmt->execute([
            ':product_id' => $product_id,
            ':location_id' => $location_id
        ]);

        return $stmt->fetchColumn() > 0;
    }

    public function getOrderItems($orderNumber){
        $sql = "SELECT 
            sti.id,
            sti.product_id, 
            sti.transfer_id,
            sti.transfer_from,
            sti.transfer_to,
            sti.qty_initiated,
            p.product_name,
            sti.datecaptured as order_date,
            pc.categoryname as category,
            pu.unitname as product_unit,

            CASE
                WHEN wf.warehouse_id IS NOT NULL THEN 'warehouse'
                ELSE 'branch'
            END AS from_type,


            -- Resolve 'location_from'
            CASE
                WHEN wf.warehouse_name IS NOT NULL THEN wf.warehouse_name
                ELSE bf.branch_name
            END AS location_from,

            -- Resolve 'location_to'
            CASE
                WHEN wt.warehouse_name IS NOT NULL THEN wt.warehouse_name
                ELSE bt.branch_name
            END AS location_to

        FROM stock_transfer_items sti

        LEFT JOIN products p ON sti.product_id = p.product_id
        LEFT JOIN product_categories pc ON p.category = pc.category_id
        LEFT JOIN product_units pu ON p.unit = pu.unit_id

        -- Joining possible sources
        LEFT JOIN warehouses wf ON sti.transfer_from = wf.warehouse_id
        LEFT JOIN branches bf ON sti.transfer_from = bf.branch_id

        -- Joining possible destinations
        LEFT JOIN warehouses wt ON sti.transfer_to = wt.warehouse_id
        LEFT JOIN branches bt ON sti.transfer_to = bt.branch_id

        WHERE sti.transfer_id =:order_number";
        $smtp = $this->link->prepare($sql);
        $smtp->bindParam(':order_number', $orderNumber, PDO::PARAM_STR);
        $smtp->execute();
        $items_list = $smtp->fetchAll(PDO::FETCH_ASSOC);
        return $items_list;
    }

    public function getPendingQuantityForProduct($company_id, $product_id) {
        $sql = "SELECT SUM(qty_initiated) AS pending_qty FROM stock_transfer_items WHERE transfer_from = :company_id AND product_id = :product_id AND transfer_state = 1";

        $stmt = $this->link->prepare($sql);
        $stmt->execute([':company_id' => $company_id, ':product_id' => $product_id]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        return floatval($result['pending_qty'] ?? 0);
    }

    public function transferStock(array $data) {
        try {
            $sql = "INSERT INTO stock_transfer_items (transfer_id, product_id, transfer_from, transfer_to, sender_opening_stock, receiver_opening_stock, qty_initiated, original_qty_initiated, sender_closing_stock, receiver_closing_stock, operator, transfer_date, transfer_state, datecaptured) VALUES (:transfer_id, :product_id, :transfer_from, :transfer_to, :sender_opening_stock, :receiver_opening_stock, :qty_initiated, :original_qty_initiated, :sender_closing_stock, :receiver_closing_stock, :operator, :transfer_date, :transfer_state, :datecaptured)";

            $stmt = $this->link->prepare($sql);

            $stmt->bindParam(':transfer_id',             $data['transfer_id']);
            $stmt->bindParam(':product_id',              $data['product_id'], PDO::PARAM_INT);
            $stmt->bindParam(':transfer_from',           $data['transfer_from']); // string like 'TP001'
            $stmt->bindParam(':transfer_to',             $data['transfer_to'], PDO::PARAM_INT);
            $stmt->bindParam(':sender_opening_stock',    $data['sender_opening_stock']);
            $stmt->bindParam(':receiver_opening_stock',  $data['receiver_opening_stock']);
            $stmt->bindParam(':qty_initiated',           $data['qty_initiated']);
            $stmt->bindParam(':original_qty_initiated',  $data['original_qty_initiated']);
            $stmt->bindParam(':sender_closing_stock',    $data['sender_closing_stock']);
            $stmt->bindParam(':receiver_closing_stock',  $data['receiver_closing_stock']);
            $stmt->bindParam(':operator',                $data['operator'], PDO::PARAM_INT);
            $stmt->bindParam(':transfer_date',           $data['transfer_date']);
            $stmt->bindParam(':transfer_state',          $data['transfer_state'], PDO::PARAM_INT);
            $stmt->bindParam(':datecaptured',            $data['datecaptured']);

            return $stmt->execute();
        } catch (PDOException $e) {
            // Optional: log or handle error
            error_log("TransferStock Error: " . $e->getMessage());
            return false;
        }

    }



    public function orderHasItems($orderNumber) {
        $query = "SELECT COUNT(*) FROM (SELECT id FROM stock_transfer_items WHERE transfer_id = :order_number) AS combined
        ";
        $stmt = $this->link->prepare($query);
        $stmt->execute(['order_number' => $orderNumber]);
        return $stmt->fetchColumn() > 0;
    }

    public function getTransferItems($transfer_id){
        $sql = "SELECT 
                sti.id AS item_id,
                sti.transfer_id,
                sti.product_id,
                pp.product_name,
                tpc.company_name,
                sti.qty_initiated AS quantity,
                COALESCE(w.warehouse_name, b.branch_name) AS destination,
                sti.transfer_from,
                sti.transfer_to as destination_id,
                sti.transfer_state,
                sti.qty_initiated,
                sti .qty_accepted,
                pc.categoryname as category_name,
                
                CASE 
                    WHEN w.warehouse_id IS NOT NULL THEN 'Warehouse'
                    WHEN b.branch_id IS NOT NULL THEN 'Branch'
                    ELSE 'Unknown'
                END AS destination_type,

                sti.transfer_date,
                
                CONCAT('Transfer requisition for ', sti.qty_initiated, ' units of ', pp.product_name) AS remarks

            FROM
                stock_transfer_items sti
            LEFT JOIN
                products pp ON sti.product_id = pp.product_id
            LEFT JOIN
                product_categories pc ON pp.category = pc.category_id
            LEFT JOIN
                third_party_companies tpc ON sti.transfer_from = tpc.company_id
            LEFT JOIN 
                warehouses w ON sti.transfer_to = w.warehouse_id
            LEFT JOIN 
                branches b ON sti.transfer_to = b.branch_id
            WHERE
                sti.transfer_id = :transfer_id";

        $smtp = $this->link->prepare($sql);
        $smtp->bindParam(':transfer_id', $transfer_id, PDO::PARAM_STR);
        $smtp->execute();
        $transfer_items  = $smtp->fetchAll(PDO::FETCH_ASSOC);

        return $transfer_items;
    }

    public function updateTransferQuantity($item_id, $quantity, $sender_opening, $receiver_opening, $sender_closing, $receiver_closing) {
        $sql = "UPDATE stock_transfer_items SET qty_initiated = :quantity, original_qty_initiated=:quantity, sender_opening_stock = :sender_opening, receiver_opening_stock = :receiver_opening, sender_closing_stock = :sender_closing, receiver_closing_stock = :receiver_closing WHERE id = :item_id";

        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':quantity', $quantity);
        $stmt->bindParam(':sender_opening', $sender_opening);
        $stmt->bindParam(':receiver_opening', $receiver_opening);
        $stmt->bindParam(':sender_closing', $sender_closing);
        $stmt->bindParam(':receiver_closing', $receiver_closing);
        $stmt->bindParam(':item_id', $item_id, PDO::PARAM_INT);

        return $stmt->execute();
    }

    public function updateTransferDate($transfer_id, $new_date) {
        $sql = "UPDATE stock_transfer_items SET transfer_date = :new_date WHERE transfer_id= :transfer_id";

        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':new_date', $new_date);
        $stmt->bindParam(':transfer_id', $transfer_id, PDO::PARAM_STR);

        return $stmt->execute();
    }

    public function isProductAlreadyInTransfer($transfer_id, $product_id) {
        $sql = "SELECT COUNT(*) as count FROM stock_transfer_items 
                WHERE transfer_id = :transfer_id AND product_id = :product_id";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':transfer_id', $transfer_id);
        $stmt->bindParam(':product_id', $product_id);
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row && intval($row['count']) > 0;
    }

    public function deleteTransferItemById($item_id) {
        $sql = "DELETE FROM stock_transfer_items WHERE id = :item_id";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':item_id', $item_id, PDO::PARAM_INT);
        return $stmt->execute();
    }




    public function getItemDetailsById($item_id) {
        $sql = "SELECT * FROM stock_transfer_items WHERE id = :item_id LIMIT 1";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':item_id', $item_id, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    public function getIncompleteTHirdCompanyStockTransfers() {
        $sql = "SELECT 
                so.transfer_id AS transaction_id,
                so.transfer_date,
                so.datecaptured,
                COALESCE(emp.fullname,'Default Account') AS operator_name,
                'Stock Transfer' as transaction_type

            FROM 
                stock_transfer_items so
            LEFT JOIN 
                stock_transfers_final py ON so.transfer_id = py.transfer_id
            LEFT JOIN
                employees emp ON so.operator = emp.employee_code
            WHERE
                py.transfer_id IS NULL
            GROUP BY
                so.transfer_id";

        $stmt = $this->link->prepare($sql);
        $stmt->execute();
        $transactions = $stmt->fetchAll(PDO::FETCH_ASSOC);

        return $transactions;
    }

    public function getFirstTransferItemDetails($transfer_id) {
        $sql = "SELECT 
                transfer_from, 
                transfer_to, 
                transfer_date,
                sti.transfer_id,
                COALESCE(wh.warehouse_name,br.branch_name) as destination_name

            FROM 
                stock_transfer_items sti
            LEFT JOIN
                warehouses wh ON sti.transfer_to = wh.warehouse_id
            LEFT JOIN
                branches br ON sti.transfer_to = br.branch_id
            WHERE 
                sti.transfer_id = :transfer_id
            ORDER BY 
                sti.id ASC 
            LIMIT 1";
        $stmt = $this->link->prepare($sql);
        $stmt->execute([':transfer_id' => $transfer_id]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    public function getFinalTransferDetails($transfer_id) {
        $sql = "SELECT 
                transfer_from, 
                transfer_to,
                stf.transfer_id,
                stf.operator,
                COALESCE(wh.warehouse_name,br.branch_name) as destination_name,
                stf.transfer_state,
                COALESCE(emp.fullname, 'Default Account') as initiated_by_name,
                COALESCE(em.fullname, 'Default Account') as declined_by_name,
                COALESCE(e.fullname, 'Default Account') as accepted_by_name,
                stf.transfer_date,
                stf.date_declined,
                stf.datereceived,
                stf.date_cancelled,
                stf.datereceived as date_accepted,
                emp.signature as accepted_by_signature,
                em.signature as declined_by_signature

            FROM 
                stock_transfers_final stf
            LEFT JOIN
                warehouses wh ON stf.transfer_to = wh.warehouse_id
            LEFT JOIN
                employees emp ON stf.operator = emp.employee_code
            LEFT JOIN
                employees em ON stf.declined_by = em.employee_code
            LEFT JOIN
                employees e ON stf.received_by = e.employee_code
            LEFT JOIN
                branches br ON stf.transfer_to = br.branch_id
            WHERE 
                stf.transfer_id = :transfer_id
            ORDER BY 
                stf.id ASC 
            LIMIT 1";
        $stmt = $this->link->prepare($sql);
        $stmt->execute([':transfer_id' => $transfer_id]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    public function finalizeStockTransfer($data) {
        $sql = "INSERT INTO stock_transfers_final (transfer_id, transfer_from, transfer_to, transfer_date, datecaptured, operator, transfer_state, driver) 
                VALUES ( :transfer_id, :transfer_from, :transfer_to, :transfer_date, :datecaptured, :operator, :transfer_state, :driver)";

        $stmt = $this->link->prepare($sql);
        return $stmt->execute([
            ':transfer_id'    => $data['transfer_id'],
            ':transfer_from'  => $data['transfer_from'],
            ':transfer_to'    => $data['transfer_to'],
            ':transfer_date'  => $data['transfer_date'],
            ':datecaptured'   => $data['datecaptured'],
            ':operator'       => $data['operator'],
            ':transfer_state' => $data['transfer_state'],
            ':driver'         => $data['driver']
        ]);
    }

    public function getAllTransfers(){
        $sql = "SELECT
            COALESCE(tpc.company_name, w.warehouse_name, b.branch_name) AS source_location,
            COALESCE(wh.warehouse_name, br.branch_name) AS destination_name,
            stf.transfer_id,
            stf.transfer_state,
            stf.transfer_date,
            stf.datecaptured,
            CASE
                WHEN stf.transfer_state = 2 THEN 'Accepted'
                WHEN stf.transfer_state = 1 THEN 'Pending Acceptance'
                WHEN stf.transfer_state = 0 THEN 'Transfer Declined'
                WHEN stf.transfer_state = 3 THEN 'Transfer Cancelled By Management'
            END AS transfer_status,
            COALESCE(emp.fullname, 'Default Account') AS operator_name,
            COUNT(sti.product_id) AS num_products

        FROM
            stock_transfers_final stf
        LEFT JOIN
            stock_transfer_items sti ON stf.transfer_id = sti.transfer_id

        LEFT JOIN warehouses wh ON stf.transfer_to = wh.warehouse_id
        LEFT JOIN warehouses w ON stf.transfer_from = w.warehouse_id
        LEFT JOIN branches b ON stf.transfer_from = b.branch_id
        LEFT JOIN branches br ON stf.transfer_to = br.branch_id
        LEFT JOIN third_party_companies tpc ON stf.transfer_from = tpc.company_id
        LEFT JOIN employees emp ON stf.operator = emp.employee_code

        GROUP BY
            stf.transfer_id
        ORDER BY
            stf.datecaptured DESC
        ";
        $stmt = $this->link->prepare($sql);
        $stmt->execute();
        $transfers = $stmt->fetchAll(PDO::FETCH_ASSOC);

        return $transfers;

    }

    public function getPendingTransfers(){
        $sql = "SELECT
            COALESCE(tpc.company_name, w.warehouse_name, b.branch_name) AS source_location,
            COALESCE(wh.warehouse_name, br.branch_name) AS destination_name,
            stf.transfer_id,
            stf.transfer_state,
            stf.transfer_date,
            stf.datecaptured,
            CASE
                WHEN stf.transfer_state = 2 THEN 'Accepted'
                WHEN stf.transfer_state = 1 THEN 'Pending Acceptance'
                WHEN stf.transfer_state = 0 THEN 'Transfer Declined'
                WHEN stf.transfer_state = 3 THEN 'Transfer Cancelled By Management'
            END AS transfer_status,
            COALESCE(emp.fullname, 'Default Account') AS operator_name,
            COUNT(sti.product_id) AS num_products

        FROM
            stock_transfers_final stf
        LEFT JOIN
            stock_transfer_items sti ON stf.transfer_id = sti.transfer_id

        LEFT JOIN warehouses wh ON stf.transfer_to = wh.warehouse_id
        LEFT JOIN warehouses w ON stf.transfer_from = w.warehouse_id
        LEFT JOIN branches b ON stf.transfer_from = b.branch_id
        LEFT JOIN branches br ON stf.transfer_to = br.branch_id
        LEFT JOIN third_party_companies tpc ON stf.transfer_from = tpc.company_id
        LEFT JOIN employees emp ON stf.operator = emp.employee_code
        
        WHERE stf.transfer_state = 1

        GROUP BY
            stf.transfer_id
        ORDER BY
            stf.datecaptured DESC
        ";
        $stmt = $this->link->prepare($sql);
        $stmt->execute();
        $transfers = $stmt->fetchAll(PDO::FETCH_ASSOC);

        return $transfers;

    }






}