<?php
    session_start();
    $customer_id = isset($_GET['customer_id']) ? intval($_GET['customer_id']) : 0;

    if ($customer_id === 0) {
        $_SESSION['notification'] = [
            'type' => 'error',
            'title' => 'FAILED!!',
            'message' => 'Invalid Customer ID, Try again!!'
        ];
        header('location:manage_customers.php');
        exit();
    }

    // 🔹 STEP 1: Fetch customer details first
    require_once __DIR__ . '/model/customers_model.php';
    $customers_model = new customers_model();
    $customer = $customers_model->getCustomerDetails($customer_id);
    $invoices = $customers_model->getActiveInvoicesByCustomer($customer_id);

    if (!$customer) {
        $_SESSION['notification'] = [
            'type' => 'error',
            'title' => 'FAILED!!',
            'message' => 'Customer not found!!'
        ];
        header('location:manage_customers.php');
        exit();
    }

    // 🔹 STEP 2: Validate balance
    $balance = 0;
    foreach ($invoices as $invoice) {
        $balance += floatval($invoice['balance']);
    }

    if ($balance <= 0) {
        $_SESSION['notification'] = [
            'type' => 'error',
            'title' => 'NO BALANCE!',
            'message' => 'Customer has no outstanding balance based on active invoices.'
        ];
        header('location:manage_customers.php');
        exit();
    }


    // 🔹 STEP 3: Validate phone number
    // Decide which number to use
    $raw_number = !empty($customer['whatsapp_number'])
        ? trim($customer['whatsapp_number'])
        : trim($customer['phone_number']);

    // Remove all non-digits
    $cleaned = preg_replace('/[^\d]/', '', $raw_number);

    // If WhatsApp number exists → treat as international
    if (!empty($customer['whatsapp_number'])) {

        // International numbers must be at least 8 digits (E.164 safe minimum)
        if (!ctype_digit($cleaned) || strlen($cleaned) < 8) {
            $_SESSION['notification'] = [
                'type' => 'error',
                'title' => 'INVALID WHATSAPP NUMBER!',
                'message' => 'Customer WhatsApp number is invalid. Please correct it before sending.'
            ];
            header('location:manage_customers.php');
            exit();
        }

        // Use as-is (already international)
        $wa_number = $cleaned;

    } else {
        // 🏠 Default phone number → assume Malawi

        // Remove country code if present
        if (strpos($cleaned, '265') === 0) {
            $cleaned = substr($cleaned, 3);
        }

        // Remove leading zero
        if (substr($cleaned, 0, 1) === '0') {
            $cleaned = substr($cleaned, 1);
        }

        // Validate Malawi number length
        if (!ctype_digit($cleaned) || strlen($cleaned) !== 9) {
            $_SESSION['notification'] = [
                'type' => 'error',
                'title' => 'INVALID PHONE NUMBER!',
                'message' => 'Customer phone number is invalid. Please correct it before sending WhatsApp.'
            ];
            header('location:manage_customers.php');
            exit();
        }

        // Prepend Malawi country code
        $wa_number = '265' . $cleaned;
    }

    $customer_name = $customer['customer_name'];


    // 🔹 STEP 4: Generate/upload invoices
    require_once __DIR__ . '/invoice_helper.php';
    $data = generateAndUploadInvoices($customer_id);





    if (!isset($data['status']) || !in_array($data['status'], ['success', 'partial'])) {
        $_SESSION['notification'] = [
            'type' => 'error',
            'title' => 'FAILED!!',
            'message' => 'Failed to generate PDF files. Cannot proceed!!'
        ];
        header('location:manage_customers.php');
        exit();
    }

    // 🔹 STEP 5: Build invoice links
    $base_url = "https://www.poweronline-mw.com/finance/invoices/";
    $invoice_links = "";

    foreach ($data['uploaded_files'] as $file) {
        if ($file['uploaded']) {
            $invoice_links .= "\n" . $base_url . $file['file'];
        }
    }

    if (empty($invoice_links)) {
        $_SESSION['notification'] = [
            'type' => 'error',
            'title' => 'FAILED!!',
            'message' => 'No invoice files were uploaded. Cannot send WhatsApp message!!'
        ];
        header('location:manage_customers.php');
        exit();
    }

    // 🔹 STEP 6: Redirect to WhatsApp
    $message = "Hello $customer_name,\n\n";
    $message .= "You have an outstanding balance of MWK " . number_format($balance, 2) . " with Powertex Ltd.\n\n";
    $message .= "Please find your active invoices below:" . $invoice_links . "\n\n";
    $message .= "Thank you.";

    $wa_link = "https://api.whatsapp.com/send?phone=$wa_number&text=" . urlencode($message);
    file_put_contents(__DIR__ . '/debug_whatsapp_message.txt', $message);
    file_put_contents(__DIR__ . '/debug_wa_link.txt', $wa_link);

    
    $_SESSION['wa_link'] = $wa_link;
    $_SESSION['notification'] = [
            'type' => 'info',
            'title' => 'Notification',
            'message' => 'Opening WhatsApp, Please wait..............'
        ];
    header('location:manage_customers.php');
    exit;