<?php

include_once 'config.php';

class quotations_model
{
    public $link;

    public function __construct()
    {
        $db_connection = new dbConnection();
        $this->link = $db_connection->connect();
    }

    public function insertQuotation($data){
        $sql = "INSERT INTO quotations 
            (quote_number, customer, quote_from, location_type, quotation_valid_until, transaction_date, vat_value,
            total_subtotal, total_tax, total, operator, quotation_state)
            VALUES
            (:quote_number, :customer, :quote_from, :location_type, :quotation_valid_until, :transaction_date,
            :vat_value, :total_subtotal, :total_tax, :total, :operator, :quotation_state)";
        $stmt = $this->link->prepare($sql);
        $stmt->execute([
            ':quote_number'          => $data['quote_number'],
            ':customer'              => $data['customer'],
            ':quote_from'            => $data['quote_from'],
            ':location_type'         => $data['location_type'],
            ':quotation_valid_until' => $data['quotation_valid_until'],
            ':transaction_date'      => $data['transaction_date'],
            ':vat_value'             => $data['vat_value'],
            ':total_subtotal'        => $data['total_subtotal'],
            ':total_tax'             => $data['total_tax'],
            ':total'                 => $data['total'],
            ':operator'              => $data['operator'],
            ':quotation_state'       => $data['quotation_state']
        ]);

        return $this->link->lastInsertId();
    }


    public function insertQuotationItem($data){
        $sql = "INSERT INTO quotation_items
            (quotation_id, product_id, quantity, price, subtotal, tax, total)
            VALUES
            (:quotation_id, :product_id, :quantity, :price, :subtotal, :tax, :total)";

        $stmt = $this->link->prepare($sql);
        return $stmt->execute([
            ':quotation_id' => $data['quotation_id'],
            ':product_id'   => $data['product_id'],
            ':quantity'     => $data['quantity'],
            ':price'        => $data['price'],
            ':subtotal'     => $data['subtotal'],
            ':tax'          => $data['tax'],
            ':total'        => $data['total']
        ]);
    }


    public function getQuotationData($quotation_number) {
        $sql = "SELECT qt.*, 
                c.customer_name,
                c.contact_number as customer_phone,
                c.email,
                qt.transaction_date as quotation_date,
                qt.quotation_valid_until as expiry_date,
                qt.total as total_amount,
                qt.quotation_state as state,
                qt.vat_value as vat_percentage,
                qt.total_subtotal as subtotal,
                qt.total_tax,
                qt.total
            FROM 
                quotations qt 
            LEFT JOIN
                customers c ON qt.customer = c.customer_id
            WHERE 
                qt.quote_number = :quote_number";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':quote_number', $quotation_number, PDO::PARAM_STR);
    
        if ($stmt->execute()) {
            return $stmt->fetch(PDO::FETCH_ASSOC); // return single row as associative array
        } else {
            return false;
        }
    }

    public function getquoteItems($quotation_number) {
        $sql = "SELECT *,
                p.product_name as description,
                qi.price,
                qi.quantity as qty,
                qi.total,
                qi.subtotal,
                qi.tax,
                cat.categoryname as categoryname
            FROM 
                quotation_items qi
            LEFT JOIN
                products p ON p.product_id = qi.product_id
            LEFT JOIN
                product_categories cat ON p.category = cat.category_id
            WHERE 
                quotation_id = :quote_number";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':quote_number', $quotation_number, PDO::PARAM_STR);
    
        if ($stmt->execute()) {
            return $stmt->fetchAll(PDO::FETCH_ASSOC); // return all matching rows
        } else {
            return false;
        }
    }

    public function fetchActiveQuotations($employee_code, $location) {
        $sql = "SELECT 
                    q.id,
                    q.quote_number,
                    q.transaction_date,
                    q.quotation_valid_until AS expiry_date,
                    q.quotation_state,
                    c.customer_name,
                    IFNULL(SUM(qi.total), 0) AS total,
                    CASE 
                        WHEN CURDATE() > q.quotation_valid_until AND q.quotation_state IN(0,1)  THEN
                        'Invalid(Expired) Quotation'
                        WHEN CURDATE() <= q.quotation_valid_until AND q.quotation_state IN(0,1) THEN
                        'Active Quotation'
                        WHEN q.quotation_state = 2 THEN
                        'Used Quotation'
                        ELSE ''
                    END as quotation_status
                FROM 
                    quotations q
                LEFT JOIN 
                    customers c ON q.customer = c.customer_id
                LEFT JOIN 
                    quotation_items qi ON q.quote_number = qi.quotation_id
                WHERE 
                    q.quotation_state IN (0,1)
                    AND CURDATE() <= q.quotation_valid_until 
                    AND (q.operator =:operator OR quote_from=:location)
                GROUP BY 
                    q.id ORDER BY 
                    q.transaction_date DESC";

        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':operator', $employee_code);
        $stmt->bindParam(':location', $location);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getQuotationItems($quote_number){
        $sql = "SELECT qi.product_id, p.product_name, qi.price, qi.quantity, qi.total
                FROM quotation_items qi
                JOIN products p ON p.product_id = qi.product_id
                WHERE qi.quotation_id = :quote_number";
        $stmt = $this->link->prepare($sql);
        $stmt->execute([':quote_number' => $quote_number]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getQuotationHeader($quote_number){
        $sql = "SELECT * FROM quotations WHERE quote_number = :quote_number";
        $stmt = $this->link->prepare($sql);
        $stmt->execute([':quote_number' => $quote_number]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    public function deleteQuotationItem($quote_number, $product_id){
    // Delete item
        $sql = "DELETE FROM quotation_items WHERE quotation_id = :quote_number AND product_id = :product_id";
        $stmt = $this->link->prepare($sql);
        $result = $stmt->execute([
            ':quote_number' => $quote_number,
            ':product_id' => $product_id
        ]);

        if($result){
            // Recalculate totals in quotations table
            $totals = $this->getQuotationTotals($quote_number);
            $sqlUpdate = "UPDATE quotations SET total_subtotal = :subtotal, total_tax = :tax, total = :total WHERE quote_number = :quote_number";
            $stmtUpdate = $this->link->prepare($sqlUpdate);
            $stmtUpdate->execute([
                ':subtotal' => $totals['subtotal'],
                ':tax' => $totals['tax'],
                ':total' => $totals['total'],
                ':quote_number' => $quote_number
            ]);
        }

        return $result;
    }

    public function getQuotationTotals($quote_number){
        $sql = "SELECT 
                    SUM(subtotal) as subtotal,
                    SUM(tax) as tax,
                    SUM(total) as total
                FROM quotation_items 
                WHERE quotation_id = :quote_number";
        $stmt = $this->link->prepare($sql);
        $stmt->execute([':quote_number' => $quote_number]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
    public function updateQuotationItems($quote_number, $items){
        if(!$quote_number) return false;

        // Delete all existing items first (optional: could update individually)
        $sqlDel = "DELETE FROM quotation_items WHERE quotation_id = :quote_number";
        $stmtDel = $this->link->prepare($sqlDel);
        $stmtDel->execute([':quote_number' => $quote_number]);

        // Insert updated items
        $sqlIns = "INSERT INTO quotation_items 
                    (quotation_id, product_id, price, quantity, subtotal, tax, total)
                VALUES (:quotation_id, :product_id, :price, :quantity, :subtotal, :tax, :total)";
        $stmtIns = $this->link->prepare($sqlIns);

        $totalSubtotal = 0;
        $totalTax = 0;
        $total = 0;

        // Fetch VAT rate from database
        $vatRate = 0;
        $stmtVat = $this->link->prepare("SELECT vat_value FROM settings LIMIT 1");
        $stmtVat->execute();
        if($row = $stmtVat->fetch(PDO::FETCH_ASSOC)){
            $vatRate = floatval($row['vat_value']); // e.g., 16.5% stored as 16.5
        }
        $vatRateDecimal = $vatRate; // convert to decimal for calculations


        foreach($items as $item){
            $price = floatval($item['price']);      // VAT-inclusive price
            $quantity = floatval($item['quantity']);

            $lineTotal = $price * $quantity;

            // Calculate subtotal and tax using dynamic VAT
            $subtotal = $lineTotal / (1 + $vatRateDecimal); 
            $tax = $lineTotal - $subtotal;

            $stmtIns->execute([
                ':quotation_id' => $quote_number,
                ':product_id' => $item['product_id'],
                ':price' => $price,
                ':quantity' => $quantity,
                ':subtotal' => $subtotal,
                ':tax' => $tax,
                ':total' => $lineTotal
            ]);

            $totalSubtotal += $subtotal;
            $totalTax += $tax;
            $total += $lineTotal;
        }



        // Update totals in quotations table
        $sqlUpdate = "UPDATE quotations 
                    SET total_subtotal = :subtotal, total_tax = :tax, total = :total
                    WHERE quote_number = :quote_number";
        $stmtUpdate = $this->link->prepare($sqlUpdate);
        $stmtUpdate->execute([
            ':subtotal' => $totalSubtotal,
            ':tax' => $totalTax,
            ':total' => $total,
            ':quote_number' => $quote_number
        ]);

        return true;
    }



}