<?php
require_once '../vendor/autoload.php';

use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;

use Twilio\Rest\Client;
class DocumentSender
{
    private $twilio;
    private $twilioFrom;

    public function __construct()
    {
        $config = include '../model/twilio.php';

        $this->twilio = new \Twilio\Rest\Client(
            $config['sid'],
            $config['token']
        );

        $this->twilioFrom = $config['from']; // whatsapp:+14155238886
    }


    public function send($channel, array $document)
    {
        switch ($channel) {
            case 'whatsapp':
                return $this->sendWhatsApp($document);

            case 'email':
                return $this->sendEmail($document);

            default:
                throw new Exception('Unsupported channel');
        }
    }

    /* ==========================
   WHATSAPP (DESKTOP / WEB)
========================== */
private function sendWhatsApp(array $doc)
{
    if (empty($doc['customer']['phone'])) {
        throw new Exception('Customer has no phone number.');
    }

    // 1️⃣ Generate PDF
    $pdf = $this->generatePdf($doc, 0);

    if (empty($pdf['path'])) {
        throw new Exception('PDF generation failed.');
    }

    // 2️⃣ Generate short URL for the PDF
    $shortCode = substr(md5(basename($pdf['path']) . time()), 0, 6);
    $shortUrl  = 'https://poweronline-mw.com/p.php?c=' . $shortCode;

    // TEMP storage of code → file mapping
    file_put_contents(
        __DIR__ . '/../pdf/tmp/links.json',
        json_encode([
            'code' => $shortCode,
            'file' => basename($pdf['path'])
        ]) . PHP_EOL,
        FILE_APPEND
    );

    // 3️⃣ Format phone (wa.me format – no +)
    $phone = $this->formatPhone($doc['customer']['phone']);

    // 4️⃣ WhatsApp message with short DOWNLOAD LINK
    // WhatsApp message with short DOWNLOAD LINK
    $lines = [
        "Hello {$doc['customer']['name']},",
        "Your {$doc['document_type']} is ready.",
        "📄 Download here:",
        $shortUrl,
        "Regards,",
        "Powertex Ltd"
    ];
    $text = urlencode(implode("\n", $lines));
    $waUrl = "https://wa.me/{$phone}?text={$text}";

    /*
    |--------------------------------------------------------------------------
    | 🔴 TWILIO API (DISABLED – KEEP FOR FUTURE)
    |--------------------------------------------------------------------------
    | $to   = 'whatsapp:+' . $phone;
    | $from = $this->twilioFrom;
    |
    | $this->twilio->messages->create(
    |     $to,
    |     [
    |         'from'     => $from,
    |         'body'     => "Hello {$doc['customer']['name']}",
    |         'mediaUrl' => [$pdf['path']]
    |     ]
    | );
    |--------------------------------------------------------------------------
    */

    return [
        'message' =>
            'Processed successfully.<br><br>' .
            'Finalize sending the document on WhatsApp.',
        'debug' => [
            'url' => $waUrl
        ]
    ];
}




   

    private function formatPhone($phone)
    {
        // Malawi example: remove spaces, ensure country code
        $phone = preg_replace('/\D/', '', $phone);

        if (strlen($phone) === 9) {
            return '265' . $phone;
        }

        if (strpos($phone, '265') === 0) {
            return $phone;
        }

        return $phone;
    }
    

    private function sendEmail(array $doc){
        if (empty($doc['customer']['email'])) {
            throw new Exception('Customer has no email address.<br>Update customer details and try again.');
        }


        // 1️⃣ Generate PDF (NO preview)
        $pdf = $this->generatePdf($doc, 0);

        if (empty($pdf['localPath']) || !file_exists($pdf['localPath'])) {
            throw new Exception('PDF file not found for email attachment.');
        }

        try {
            $mail = new PHPMailer(true);
            $mail->isSMTP();
            $mail->Host = 'www.poweronline-mw.com';
            $mail->SMTPAuth = true;
            $mail->SMTPDebug = 0;
            $mail->Username = 'noreply@smartplus-mw.com';
            $mail->Password = '#2024!@EMAIL';
            $mail->SMTPSecure = ''; // or 'tls'
            $mail->Port = 587;
            $mail->setFrom('noreply@smartplus-mw.com', 'Powertex Ltd');
            $mail->addAddress($doc['customer']['email'], $doc['customer']['name']);
            $mail->addAttachment($pdf['localPath'], basename($pdf['localPath']));
            $mail->isHTML(false);
            $mail->Subject = strtoupper($doc['document_type']) . ' - ' . $doc['document_number'];

            $mail->Body    = 
                "Hello {$doc['customer']['name']},\n\n" .
                "Please find your {$doc['document_type']} attached from Powertex Limited.\n\n" .
                "Thank you.";

            $mail->send();

        
            return [
                'message' => 'Document sent via email successfully'
            ];

        }catch (\Exception $e) {
            throw new Exception($e->getMessage());
        }

    }


    private function generatePdf(array $doc, $debug = 0)
    {
        $ch = curl_init();

        $postFields = [
            'document_type' => $doc['document_type'],
            'reference'     => $doc['document_number'],
            'debug'         => $debug
        ];

        // Auto-detect environment
        if (in_array($_SERVER['SERVER_NAME'], ['localhost', '127.0.0.1'])) {
            $url = 'http://localhost/powertex/supervisor/pdf/generate_view.php';
        } else {
            // Live server
            $url = 'https://' . $_SERVER['SERVER_NAME'] . '/supervisor/pdf/generate_view.php';
        }

        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => $postFields,
            CURLOPT_RETURNTRANSFER => true
        ]);

        $response = curl_exec($ch);
        curl_close($ch);

        // ✅ If debug=1, PDF streamed, stop execution immediately
        if ($debug == 1) {
            exit;
        }

        $data = json_decode($response, true);

        if (!$data || empty($data['path'])) {
            throw new Exception('PDF generation failed');
        }

        return $data;
    }




}
