<?php
session_start();
include_once '../model/stock_model.php';
include_once '../model/sales_model.php';
date_default_timezone_set('Africa/Blantyre');

$stock_model = new stock_model();
$sales_model = new sales_model();

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';

    switch($action){
        case 'create_delivery_note':
            createDeliveryNote($sales_model, $stock_model);
            break;
        
        default:
        $_SESSION['notification']=[
            'type' => 'error',
            'title' => 'FAILED',
            'message' => "Invalid action specified"
        ];
        header('location:');
        exit();
        break;
    }
}

    function createDeliveryNote($sales_model, $stock_model){
        $delivery_note_no  = $_POST['delivery_note_no'] ?? '';
        $customer          = $_POST['customer_id'] ?? 0;
        $delivery_date     = $_POST['delivery_date'] ?? '';
        $reference_no      = $_POST['reference_no'] ?? '';
        $location          = $_POST['location'] ?? '';
        $delivered_by      = $_POST['delivered_by'] ?? '';
        $datecaptured      = date('Y-m-d H:i:s');
        $operator          = $_SESSION['sess_employeecode'] ?? 0;
        $dn_state          = 1;
        $invoice_no        = null;
        $products          = $_POST['product_id'] ?? [];
        $quantities        = $_POST['qty'] ?? [];

        $errors = [];

        /* ================= USER SESSION ================= */
        if (!$operator) {
            $errors[] = 'Invalid session. Please login again.';
        }

        /* ================= REQUIRED FIELDS ================= */
        if (empty($delivery_note_no)) {
            $errors[] = 'Delivery note number is required.';
        }

        if ((int)$customer <= 0) {
            $errors[] = 'Customer is required.';
        }

        if (empty($delivery_date)) {
            $errors[] = 'Delivery date is required.';
        } elseif (strtotime($delivery_date) > time()) {
            $errors[] = 'Delivery date cannot be in the future.';
        }

        /* ================= PRODUCTS ================= */
        if (!is_array($products) || !is_array($quantities) || count($products) === 0) {
            $errors[] = 'At least one delivery item is required.';
        }

        if (count($products) !== count($quantities)) {
            $errors[] = 'Invalid delivery items submitted.';
        }

        /* ================= DUPLICATE PRODUCT CHECK ================= */
        $uniqueProducts = array_filter($products);
        if (count($uniqueProducts) !== count(array_unique($uniqueProducts))) {
            $errors[] = 'Duplicate products detected in delivery note.';
        }

        /* ================= PER-ITEM VALIDATION ================= */
        foreach ($products as $index => $product_id) {

            $qty = $quantities[$index] ?? 0;

            if ((int)$product_id <= 0) {
                $errors[] = 'Invalid product selected.';
                break;
            }

            if (!is_numeric($qty) || $qty <= 0) {
                $errors[] = 'Quantity must be greater than zero.';
                break;
            }

            // Check stock availability
            $opening_stock = $stock_model->getBranchQuantity($product_id, $location);
            $closing_stock = $opening_stock - $qty;

            if ($opening_stock === null) {
                $errors[] = 'Invalid product selected.';
                break;
            }

            if ($qty > $opening_stock) {
                $errors[] = "Insufficient stock for product ID {$product_id}.";
                break;
            }
        }

        /* ================= DN UNIQUENESS ================= */
        if ($sales_model->deliveryNoteExists($delivery_note_no)) {
            $delivery_note_no = getNextDeliveryNoteNumber($location, 'branch');
        }

    
        /* ================= FINAL FAIL ================= */
        if (!empty($errors)) {
            $_SESSION['notification'] = [
                'type'    => 'error',
                'title'   => 'Validation Error',
                'message' => implode('<br>', $errors)
            ];

            header('Location: ../new_delivery_note.php?dn_number=' . urlencode($delivery_note_no));
            exit();
        }

        try {
             $sales_model->link->beginTransaction();

             /* ================= CREATE DELIVERY NOTE ================= */
             //$dn_number, $reference_no, $customer, $location, $dn_date, $datecaptured, $operator, $dn_state,
             $create = $sales_model->createDeliveryNote($delivery_note_no, $reference_no, $customer, $location, $delivery_date, $datecaptured, $operator, $delivered_by, $dn_state, $invoice_no);

             if (!$create) {
                 throw new Exception('Failed to create delivery note.');
             }

             /* ================= UPDATE STOCK & CREATE DN ITEMS ================= */
             foreach ($products as $index => $product_id) {
                 $qty = $quantities[$index];

                $opening_stock = $stock_model->getBranchQuantity($product_id, $location);
                $closing_stock = $opening_stock - $qty;

                 // Update stock
                 $updateStock = $stock_model->updateBranchQuantity($product_id, $location, $closing_stock);
                 if (!$updateStock) {
                     throw new Exception("Failed to update stock for product ID {$product_id}.");
                 }
    
                 $data = [
                    'sales_number'   => $delivery_note_no,
                    'product_code'   => $product_id,
                    'type'           => 'delivery_note',
                    'department'     => $location,
                    'opening_stock'  => $opening_stock,
                    'quantity'       => $qty,
                    'closing_stock'  => $closing_stock,
                    'selling_price'  => 0,
                    'subtotal'       => 0,
                    'vat_charge'     => 0,
                    'tax'            => 0,
                    'total'          => 0,
                    'date_captured'  => $datecaptured,
                    'operator'       => $operator,
                    'status'         => 1
                 ];
                 // Create DN item
                 $createItem = $sales_model->insertSalesOrder($data);
                 if (!$createItem) {
                     throw new Exception("Failed to create delivery note item for product ID {$product_id}.");
                 }
             }

             $sales_model->link->commit();

             $_SESSION['notification'] = [
                 'type'    => 'success',
                 'title'   => 'Success',
                 'message' => 'Delivery note created successfully.'
             ];

             header('Location: ../customer_delivery_note.php?dn_number=' . urlencode($delivery_note_no));
             exit();

        } catch (Exception $e) {
            $sales_model->link->rollBack();

            $_SESSION['notification'] = [
                'type'    => 'error',
                'title'   => 'Error',
                'message' => $e->getMessage()
            ];

            header('Location: ../new_delivery_note.php?dn_number=' . urlencode($delivery_note_no));
            exit();
        }


    }

