<?php

include_once 'config.php';

class banking_model
{
    public $link;

    public function __construct()
    {
        $db_connection = new dbConnection();
        $this->link = $db_connection->connect();
    }

    public function getAccounts(){
        $sql = 'SELECT * FROM bank_accounts WHERE account_state = 1';
        $smtp = $this->link->prepare($sql);
        $smtp->execute();
        $data = $smtp->fetchAll(PDO::FETCH_ASSOC);

        return $data;
    }

    public function getTotalCashSales($location) {
        $sql = "SELECT 
                    SUM(amount_paid) as total_cash_sales 
                FROM 
                    payments 
                WHERE 
                    payment_state = 1 
                    AND payment_method = 'Cash' 
                    AND location = :location";

        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':location', $location, PDO::PARAM_INT);
        $stmt->execute();

        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row['total_cash_sales'] ?? 0;
    }

    public function getTotalCashDeposited($location){
        $sql = "SELECT 
                    SUM(amount_deposited) as total_deposited
                FROM
                    cash_deposits
                WHERE
                    deposit_state = 1
                    AND deposit_location = :location";

        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':location', $location, PDO::PARAM_INT);
        $stmt->execute();

        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row['total_deposited'] ?? 0;
    }

    public function getLastDeposits($location, $limit) {
    // Ensure limit is integer to prevent SQL injection
        $limit = (int)$limit;

        $sql = "SELECT cd.*,
                    ba.account_name AS account_name,
                    ba.account_number,
                    b.branch_name as location
                FROM 
                    cash_deposits cd
                LEFT JOIN 
                    bank_accounts ba ON cd.account = ba.account_id
                LEFT JOIN 
                    branches b ON cd.deposit_location = b.branch_id
                WHERE 
                    deposit_location = :location 
                ORDER BY 
                    date_deposited DESC, deposit_id DESC 
                LIMIT $limit";

        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':location', $location, PDO::PARAM_STR);
        $stmt->execute();

        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function insertDeposit($data){
        $sql = "INSERT INTO cash_deposits 
            (account, deposit_location, deposited_by, date_deposited, reference_number, description, amount_deposited, operator, datecaptured, deposit_state)
            VALUES
            (:account, :deposit_location, :deposited_by, :date_deposited, :reference_number, :description, :amount_deposited, :operator, :datecaptured, :deposit_state)
        ";

        $stmt = $this->link->prepare($sql);

        // Bind values
        $stmt->bindParam(':account', $data['account_id'], PDO::PARAM_INT);
        $stmt->bindParam(':deposit_location', $data['location'], PDO::PARAM_INT);
        $stmt->bindParam(':deposited_by', $data['deposited_by'], PDO::PARAM_STR);
        $stmt->bindParam(':date_deposited', $data['deposit_date'], PDO::PARAM_STR);
        $stmt->bindParam(':reference_number', $data['reference'], PDO::PARAM_STR);
        $stmt->bindParam(':description', $data['description'], PDO::PARAM_STR);
        $stmt->bindParam(':amount_deposited', $data['amount'], PDO::PARAM_STR);
        $stmt->bindParam(':operator', $data['operator'], PDO::PARAM_INT);
        $stmt->bindParam(':datecaptured', $data['datecaptured'], PDO::PARAM_STR);
        $stmt->bindValue(':deposit_state', 1, PDO::PARAM_INT);

        if($stmt->execute()){
            // Return last inserted ID
            return $this->link->lastInsertId();
        }

        return false;
    }

    public function getTotalDeposited($location) {
        $sql = "SELECT SUM(amount_deposited) as total_deposited FROM cash_deposits WHERE deposit_location = :location";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':location', $location, PDO::PARAM_INT);
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row['total_deposited'] ?? 0;
    }

    public function getDepositById($deposit_id) {
        $sql = "SELECT cd.*,
                    ba.account_name AS account_name,
                    ba.account_number,
                    b.branch_name as location
                FROM 
                    cash_deposits cd
                LEFT JOIN 
                    bank_accounts ba ON cd.account = ba.account_id
                LEFT JOIN 
                    branches b ON cd.deposit_location = b.branch_id
                WHERE 
                    cd.deposit_id = :deposit_id
                LIMIT 1";
                
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':deposit_id', $deposit_id, PDO::PARAM_INT);
        $stmt->execute();
        $deposit = $stmt->fetch(PDO::FETCH_ASSOC);
        return $deposit ?: false;
    }

    public function getRecordedExpenses($location) {
        $sql = "SELECT SUM(amount_paid) as total_expenses FROM recorded_expenses WHERE expense_location = :location AND expense_state = 2";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':location', $location, PDO::PARAM_INT);
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row['total_expenses'] ?? 0;
    }

    public function getCashPaymentSummary($location) {
        $sql = "SELECT 
                    COUNT(*) as count,
                    SUM(amount_paid) as total
                FROM 
                    payments
                WHERE 
                    payment_method = 'Cash'
                        AND location = :location";

        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':location', $location, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

        public function getExpenseTransactions($location) {
            $sql = "SELECT *,
                        COALESCE(emp.fullname, 'Default Account') as recorded_by 
                    FROM 
                        recorded_expenses re
                    LEFT JOIN 
                        employees emp ON re.operator = emp.employee_code
                    WHERE 
                        expense_location = :location 
                        AND expense_state = 2 
                    ORDER BY 
                        expense_date DESC";
            $stmt = $this->link->prepare($sql);
            $stmt->bindParam(':location', $location, PDO::PARAM_INT);
            $stmt->execute();
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        }

        public function referenceExists($reference) {
            $sql = "SELECT COUNT(*)
                FROM 
                    cash_deposits 
                WHERE 
                    reference_number = :reference";

            $stmt = $this->link->prepare($sql);
            $stmt->bindParam(':reference', $reference);
            $stmt->execute();

            return $stmt->fetchColumn() > 0;
        }

        public function getCancelledPayments($location) {
            $sql = "SELECT 
                        sale_id 
                    FROM 
                        payments 
                    WHERE 
                        payment_state = 0 
                        AND location = :location
                    ORDER BY 
                        payment_date DESC";
            $stmt = $this->link->prepare($sql);
            $stmt->bindParam(':location', $location, PDO::PARAM_INT);
            $stmt->execute();
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        }

        public function insertExpense($data) {
            $sql = "INSERT INTO recorded_expenses 
                (reference_number, expense_description, expense_date, expense_location, amount_paid, datecaptured, operator, expense_state)
                VALUES
                (:reference_number, :expense_description, :expense_date, :expense_location, :amount_paid, :datecaptured, :operator, :expense_state)";

            $stmt = $this->link->prepare($sql);

            // Bind values
            $stmt->bindParam(':reference_number', $data['reference_number'], PDO::PARAM_STR);
            $stmt->bindParam(':expense_date', $data['expense_date'], PDO::PARAM_STR);
            $stmt->bindParam(':amount_paid', $data['amount_paid'], PDO::PARAM_STR);
            $stmt->bindParam(':expense_description', $data['expense_description'], PDO::PARAM_STR);
            $stmt->bindParam(':expense_location', $data['location'], PDO::PARAM_INT);
            $stmt->bindParam(':operator', $data['operator'], PDO::PARAM_INT);
            $stmt->bindParam(':datecaptured', $data['datecaptured'], PDO::PARAM_STR);
            $stmt->bindValue(':expense_state', 1, PDO::PARAM_INT); // requested expense

            if ($stmt->execute()) {
                return $this->link->lastInsertId(); // return the new expense ID
            } else {
                return false; // insert failed
            }
        }

        public function getExpenseDetails($expense_id) {
            $sql = "SELECT re.*,
                        COALESCE(emp.fullname, 'Default Account') as recorded_by 
                    FROM 
                        recorded_expenses re
                    LEFT JOIN 
                        employees emp ON re.operator = emp.employee_code
                    WHERE 
                        re.id = :expense_id 
                    LIMIT 1";
            $stmt = $this->link->prepare($sql);
            $stmt->bindParam(':expense_id', $expense_id, PDO::PARAM_INT);
            $stmt->execute();
            return $stmt->fetch(PDO::FETCH_ASSOC);
        }

        public function getExpensesList($location, $month = null, $year = null) {

            $sql = "SELECT 
                        e.*, 
                        COALESCE(u.fullname, 'Default Account') AS operator_name
                    FROM 
                        recorded_expenses e
                    LEFT JOIN 
                        employees u ON e.operator = u.employee_code
                    WHERE 
                        e.expense_location = :location";

                    // If month & year provided, filter
                    if ($month && $year) {
                        $sql .= " AND MONTH(e.expense_date) = :month
                                AND YEAR(e.expense_date) = :year";
                    }

            $sql .= " ORDER BY e.expense_state DESC, e.expense_date DESC";

            $stmt = $this->link->prepare($sql);
            $stmt->bindValue(':location', $location);

            if ($month && $year) {
                $stmt->bindValue(':month', $month);
                $stmt->bindValue(':year', $year);
            }

            $stmt->execute();
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        }

        public function updateExpense($id, $reference_number, $expense_date, $amount_paid, $description) {
            $sql = "UPDATE 
                        recorded_expenses 
                    SET 
                        reference_number = :reference_number, 
                        expense_date = :expense_date, 
                        expense_description = :description,
                        amount_paid = :amount_paid
                    WHERE 
                        id = :id";

            $stmt = $this->link->prepare($sql);
            $stmt->bindParam(':reference_number', $reference_number, PDO::PARAM_STR);
            $stmt->bindParam(':expense_date', $expense_date, PDO::PARAM_STR);
            $stmt->bindParam(':description', $description, PDO::PARAM_STR);
            $stmt->bindParam(':amount_paid', $amount_paid, PDO::PARAM_STR);
            $stmt->bindParam(':id', $id, PDO::PARAM_INT);

            return $stmt->execute();
        }

        public function deleteExpense($id){

            // Only allow delete if state = 1 (Waiting Approval)
            $stmt = $this->link->prepare("DELETE FROM recorded_expenses WHERE id = :id AND expense_state = 1");

            $stmt->bindValue(':id', $id);
            return $stmt->execute();
        }
    
}