<?php

include_once 'config.php';

class employees_model
{
    public $link;

    public function __construct()
    {
        $db_connection = new dbConnection();
        $this->link = $db_connection->connect();
    }


    
    public function getUserByEmployeeCode(int $employee_code): ?array {
        $stmt = $this->link->prepare("SELECT * FROM employees WHERE employee_code = :employee_code LIMIT 1");
        $stmt->bindParam(':employee_code', $employee_code, PDO::PARAM_INT);
        $stmt->execute();

        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        return $user ?: null;
    }

    
    public function getTodayBirthdayEmployees($date_today) {
        $sql = "SELECT 
                    emp.fullname, 
                    emp.employee_code, 
                    emp.date_of_birth, 
                    emp.start_date,
                    pp.postname as position
                FROM 
                    employees emp
                LEFT JOIN
                    posts pp ON emp.post = pp.id  
                WHERE DATE_FORMAT(date_of_birth, '%m-%d') = :today 
                AND employee_state = 1";
        
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':today', $date_today, PDO::PARAM_STR);
        $stmt->execute();

        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }


    public function getTodaysBirthdays() {
        $today = date('m-d');
        $sql = "SELECT fullname, employee_code 
            FROM 
                employees 
            WHERE 
                DATE_FORMAT(date_of_birth, '%m-%d') = :today 
                AND employee_state = 1";

        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':today', $today, PDO::PARAM_STR);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }



    


    public function getIndividualLeaveData($employee_code) {
        $sql = "SELECT 
                e.employee_code,
                e.fullname,
                (YEAR(CURDATE()) - YEAR(e.start_date) + 1) AS years_worked,
                (YEAR(CURDATE()) - YEAR(e.start_date) + 1) * 18 AS total_days,
                COALESCE(SUM(DATEDIFF(lr.approved_end_date, lr.approved_start_date) + 1), 0) AS total_taken,
                ((YEAR(CURDATE()) - YEAR(e.start_date) + 1) * 18 -
                COALESCE(SUM(DATEDIFF(lr.approved_end_date, lr.approved_start_date) + 1), 0)) AS total_left
            FROM 
                employees e
            LEFT JOIN 
                leave_requests lr 
                ON lr.employee = e.employee_code
                AND lr.leaave_state = 1
                AND lr.leave_type = 'Annual Leave'
            WHERE
                e.employee_code = :employee_code
            GROUP BY 
                e.employee_code, e.fullname, e.start_date
            LIMIT 1";

        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':employee_code', $employee_code, PDO::PARAM_STR);
        $stmt->execute();
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    public function newLeavecalculation($start, $end) {
        $sql = "SELECT 
                    e.employee_code,
                    e.fullname,
                    e.start_date,
                    COALESCE(SUM(DATEDIFF(lr.approved_end_date, lr.approved_start_date) + 1), 0) AS total_taken
                FROM 
                    employees e
                LEFT JOIN 
                    leave_requests lr 
                    ON lr.employee = e.employee_code
                    AND lr.leaave_state = 1
                    AND lr.leave_type = 'Annual Leave'
                WHERE
                    e.employee_state = 1
                GROUP BY 
                    e.employee_code, e.fullname, e.start_date
                ORDER BY 
                    e.start_date ASC";

        $stmt = $this->link->prepare($sql);
        $stmt->execute();
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

        $result = [];
        foreach ($rows as $row) {
            $startDate = new DateTime($row['start_date']);
            $now = new DateTime();

            // Calculate probation end date
            $probationEnd = (clone $startDate)->modify('+3 months');

            if ($probationEnd > $now) {
                $accruedDays = 0; // Still on probation
            } else {
                // Months after probation
                $monthsWorked = ($now->diff($probationEnd)->y * 12) + $now->diff($probationEnd)->m;
                $accruedDays = round($monthsWorked * 1.5, 1); // 1.5 leave days per month
            }

            $totalTaken = (float) $row['total_taken'];
            $totalLeft = max(0, $accruedDays - $totalTaken);

            $result[] = [
                'employee_code' => $row['employee_code'],
                'fullname' => $row['fullname'],
                'start_date' => $row['start_date'],
                'total_taken' => $totalTaken,
                'total_days' => $accruedDays,
                'total_left' => $totalLeft
            ];
        }

        return $result;
    }


    public function getleaveSummaryData($employee_code) {
    // Step 1: Fetch employee info + approved leave days
        $sql = "SELECT 
                    e.employee_code,
                    e.fullname,
                    e.start_date,
                    COALESCE(SUM(DATEDIFF(lr.approved_end_date, lr.approved_start_date) + 1), 0) AS total_approved
                FROM 
                    employees e
                LEFT JOIN 
                    leave_requests lr 
                    ON lr.employee = e.employee_code 
                    AND lr.leaave_state = 1
                    AND lr.leave_type = 'Annual Leave'
                WHERE 
                    e.employee_code = :employee_code
                GROUP BY 
                    e.employee_code, e.fullname, e.start_date";

        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':employee_code', $employee_code, PDO::PARAM_STR);
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$row) return null;

        // Step 2: Calculate months worked after probation
        $startDate = new DateTime($row['start_date']);
        $now = new DateTime();

        $probationEnd = (clone $startDate)->modify('+3 months');

        if ($probationEnd > $now) {
            $monthsWorked = 0;
        } else {
            $interval = $now->diff($probationEnd);
            $monthsWorked = ($interval->y * 12) + $interval->m;
        }

        $accrued = round($monthsWorked * 1.5, 1);
        $taken = (float) $row['total_approved'];
        $left = max(0, $accrued - $taken);

        // Calculate total full calendar years for reference
        $yearsWorked = $startDate->diff($now)->y + 1;

        return [
            'employee_code'    => $row['employee_code'],
            'fullname'         => $row['fullname'],
            'start_date'       => $row['start_date'],
            'years_worked'     => $yearsWorked,
            'total_allocated'  => $accrued,
            'total_approved'   => $taken,
            'total_left'       => $left
        ];
    }


    public function saveLeaveRequest($leave_id, $leave_type, $employee, $start_date, $end_date, $requested_numdays, $datecaptured, $operator, $leave_state) {
        $sql = "INSERT INTO leave_requests (leave_id, leave_type, employee, start_date, end_date, requested_numdays, datecaptured, operator, leaave_state) VALUES (:leave_id, :leave_type, :employee, :start_date, :end_date, :requested_numdays, :datecaptured, :operator, :leave_state)";

        $stmt = $this->link->prepare($sql);

        $stmt->bindParam(':leave_id', $leave_id);
        $stmt->bindParam(':leave_type', $leave_type);
        $stmt->bindParam(':employee', $employee);
        $stmt->bindParam(':start_date', $start_date);
        $stmt->bindParam(':end_date', $end_date);
        $stmt->bindParam(':requested_numdays', $requested_numdays);
        $stmt->bindParam(':datecaptured', $datecaptured);
        $stmt->bindParam(':operator', $operator);
        $stmt->bindParam(':leave_state', $leave_state);

        try {
            $stmt->execute();
            return $this->link->lastInsertId();            
        } catch (PDOException $e) {
            
        }
    }

    

    public function leaveDetails($leave_id) {
        $sql = "SELECT
                    leave_name,
                    leave_type,
                    leave_id
                FROM 
                    leave_allocations
                WHERE
                    leave_id = :leave_id
        ";

        try {
            $stmt = $this->link->prepare($sql);
            $stmt->bindParam(':leave_id', $leave_id, PDO::PARAM_INT);
            $stmt->execute();
            
            // Fetch a single row as associative array
            return $stmt->fetch(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            // Optional: Log the error
            // error_log('leaveDetails error: ' . $e->getMessage());
            return false;
        }
    }


    public function getLeaves(){
        $sql = "SELECT 
            leave_id,
            leave_name,
            leave_type
            FROM
                leave_allocations
            ORDER BY 
                leave_name ASC
            ";
        $smtp = $this->link->prepare($sql);
        $smtp->execute();
        $data = $smtp->fetchAll(PDO::FETCH_ASSOC);
        return $data;
    }

    public function disapproveLeaveRequest($requestId, $reason, $leave_state, $disapprovedBy, $disapprovedDate){
        $sql = "UPDATE leave_requests 
                SET 
                    leaave_state = :leave_state, 
                    reason = :reason,
                    disapproved_by = :disapprovedBy, 
                    disapproved_date = :disapprovedDate 
                WHERE 
                    id = :requestId";

        $stmt = $this->link->prepare($sql);
        return $stmt->execute([':leave_state' => $leave_state, ':reason' => $reason, ':disapprovedBy' => $disapprovedBy, ':disapprovedDate' => $disapprovedDate, ':requestId' => $requestId]);
    }


    public function getLeaveSlipData($leave_id) {
    // SQL query to get the leave slip details
    $sql = "SELECT 
                lr.id,
                er.fullname,
                lr.employee,
                lr.start_date,
                la.leave_type,
                lr.end_date,
                lr.requested_numdays,
                lr.approved_start_date,
                lr.approved_end_date,
                lr.approved_numdays,
                lr.leaave_state,
                e.fullname as operator,
                lr.disapproved_date,
                lr.datecaptured,
                lr.reason,
                COALESCE(emp.fullname,'Default Account') as approver,
                COALESCE(em.fullname,'Default Account') as disapprover
            FROM 
                leave_requests lr
            LEFT JOIN
                employees emp ON lr.approved_by = emp.employee_code
            LEFT JOIN
                employees em ON lr.disapproved_by = em.employee_code
            LEFT JOIN
                employees e ON lr.operator = e.employee_code       
            LEFT JOIN
                employees er ON lr.employee = er.employee_code            
            LEFT JOIN
                leave_allocations la ON lr.leave_id = la.leave_id
            WHERE 
                lr.id = :leave_id";  // Replace `leave_table` with the actual table name

    // Prepare the SQL statement
    $stmt = $this->link->prepare($sql);
    
    // Bind the leave_id parameter
    $stmt->bindParam(':leave_id', $leave_id, PDO::PARAM_INT);
    
    // Execute the query
    $stmt->execute();

    // Fetch the result as an associative array
    $result = $stmt->fetch(PDO::FETCH_ASSOC);

    // Check if the result exists
    if ($result) {
        return $result;
    } else {
        return null;  // Return null if no record found
    }
}


    


    public function getLeaveDetails($employee_code) {
        $sql = "SELECT 
                    lr.id,
                    la.leave_name as leave_type,
                    lr.start_date,
                    lr.end_date,
                    lr.leaave_state,
                    lr.datecaptured,
                    COALESCE(emp.fullname,'Default Account') as operator,
                    CASE 
                        WHEN lr.leaave_state = 1 THEN 'Approved Leave'
                        WHEN lr.leaave_state = 0 THEN 'Requested Leave'
                        WHEN lr.leaave_state = 3 THEN 'Disapproved Leave'
                        ELSE 'Unknown Leave State'
                    END as leave_status,    
                    DATEDIFF(lr.end_date, lr.start_date) + 1 AS total_days
                FROM 
                    leave_requests lr
                LEFT JOIN
                    leave_allocations la ON lr.leave_id = la.leave_id
                LEFT JOIN
                    employees emp ON lr.operator = emp.employee_code
                WHERE
                    lr.employee = :employee_code
                    AND lr.leave_type = 'Annual Leave'
                ORDER BY
                    lr.start_date DESC";

        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':employee_code', $employee_code, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }


    

    public function emailExistsForOtherEmployee($email, $currentEmployeeCode) {
        $query = "SELECT COUNT(*) as total FROM employees WHERE email_address = :email AND employee_code != :code";
        $stmt = $this->link->prepare($query);
        $stmt->bindParam(':email', $email, PDO::PARAM_STR);
        $stmt->bindParam(':code', $currentEmployeeCode, PDO::PARAM_INT);
        $stmt->execute();
        $result = $stmt->fetch(PDO::FETCH_ASSOC);

        return $result['total'] > 0;
    }

    public function nationalIDExistsForOtherEmployee($national_id, $currentEmployeeCode) {
        $query = "SELECT COUNT(*) as total FROM employees WHERE national_id_number = :national_id AND employee_code != :code";
        $stmt = $this->link->prepare($query);
        $stmt->bindParam(':national_id', $national_id, PDO::PARAM_STR);
        $stmt->bindParam(':code', $currentEmployeeCode, PDO::PARAM_INT);
        $stmt->execute();
        $result = $stmt->fetch(PDO::FETCH_ASSOC);

        return $result['total'] > 0;
    }

    public function updateEmployeeDetails(array $data): bool {
        $sql = "UPDATE employees SET 
                    national_id_number = :national_id_number,
                    fullname = :fullname,
                    phone_number = :phone_number,
                    email_address = :email_address,
                    gender = :gender,
                    date_of_birth = :date_of_birth,
                    current_location = :current_location,
                    nationality = :nationality,
                    home_village = :home_village,
                    home_district = :home_district
                WHERE employee_code = :employee_code";

        $stmt = $this->link->prepare($sql);

        // Bind parameters
        $stmt->bindParam(':national_id_number', $data['national_id_number']);
        $stmt->bindParam(':fullname', $data['fullname']);
        $stmt->bindParam(':phone_number', $data['phone_number']);
        $stmt->bindParam(':email_address', $data['email_address']);
        $stmt->bindParam(':gender', $data['gender']);
        $stmt->bindParam(':date_of_birth', $data['date_of_birth']);
        $stmt->bindParam(':current_location', $data['current_location']);
        $stmt->bindParam(':nationality', $data['nationality']);
        $stmt->bindParam(':home_village', $data['home_village']);
        $stmt->bindParam(':home_district', $data['home_district']);
        $stmt->bindParam(':employee_code', $data['employee_code']);

        // Execute and return success/failure
        return $stmt->execute();
    }




    public function getEmployeeDetails($code) {
        $stmt = $this->link->prepare("SELECT 
                emp.national_id_number,
                emp.employee_code,
                emp.fullname,
                emp.phone_number,
                emp.email_address as email,
                emp.gender,
                emp.date_of_birth,
                emp.current_location,
                emp.home_village,
                emp.home_district,
                emp.nationality AS nationality_id,
                emp.post,
                emp.picture as profile_picture,
                emp.start_date,
                CASE
                    WHEN emp.employee_state = 1 THEN 'Active Employee'
                    ELSE 'Suspended'
                END as employee_status,
                emp.contract_type,
                emp.gross_salary,
                emp.end_date,
                emp.datecaptured,
                emp.operator AS operator_code,
                emp.picture,
                emp.employee_state,
                ll.location_name,
                pp.postname,
                dd.department_name,
                cc.nationality,
                d.district AS district_name,
                COALESCE(em.fullname, 'Default System Administrator') AS operator_name
    
            FROM 
                employees emp 
            LEFT JOIN 
                locations ll ON emp.current_location = ll.loc_id 
            LEFT JOIN 
                posts pp ON emp.post = pp.id
            LEFT JOIN 
                departments dd ON pp.department = dd.id
            LEFT JOIN 
                countries cc ON emp.nationality = cc.id
            LEFT JOIN 
                districts d ON ll.loc_dist = d.id
            LEFT JOIN 
                employees em ON emp.operator = em.employee_code 
            WHERE 
                emp.employee_code = ?");
        
        $stmt->execute([$code]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    public function valueExists($column, $value, $employee_code){
        // ✅ Whitelist allowed columns
        $allowed_columns = [
            'phone_number',
            'email_address',
            'national_id_number'
        ];

        if (!in_array($column, $allowed_columns)) {
            throw new Exception("Invalid column supplied");
        }

        // ✅ Column injected AFTER whitelist
        $sql = "SELECT 
                    employee_code
                FROM 
                    employees
                WHERE 
                    {$column} = :value
                    AND employee_code != :employee_code
                LIMIT 1";

        $stmt = $this->link->prepare($sql);
        $stmt->bindValue(':value', $value);
        $stmt->bindValue(':employee_code', $employee_code);
        $stmt->execute();

        return $stmt->rowCount() > 0;
    }
    public function updateEmployee(array $data){
        $sql = "UPDATE employees SET fullname = :fullname, gender = :gender, national_id_number = :nid, phone_number = :phone, email_address = :email, date_of_birth = :dob, nationality = :nationality, home_village = :village, home_district = :district, current_location = :location WHERE employee_code = :employee_code";

        $stmt = $this->link->prepare($sql);

        return $stmt->execute([
            ':fullname'       => $data['fullname'],
            ':gender'         => $data['gender'],
            ':nid'            => $data['national_id_number'],
            ':phone'          => $data['phone_number'],
            ':email'          => $data['email'],
            ':dob'            => $data['date_of_birth'],
            ':nationality'    => $data['nationality'],
            ':village'        => $data['home_village'],
            ':district'       => $data['district_id'],
            ':location'       => $data['location_id'],
            ':employee_code'  => $data['employee_code']
        ]);
    }

    public function updateProfilePicture($employee_code, $filename){
        $sql = "UPDATE employees SET picture = :picture WHERE employee_code = :employee";
        $stmt = $this->link->prepare($sql);

        return $stmt->execute([
            ':picture' => $filename,
            ':employee' => $employee_code
        ]);
    }  
    
    public function getEmployeeLeaves($employee_code){
        $sql = "SELECT
                    emp.fullname as employee_name,
                    la.leave_name as leave_type,
                    lr.start_date,
                    lr.end_date,
                    em.fullname as disapproved_by,
                    e.fullname as approved_by,
                    CASE
                        WHEN lr.leaave_state = 2 THEN  approved_numdays
                        WHEN lr.leaave_state = 1 THEN  requested_numdays
                        WHEN lr.leaave_state = 0 THEN  requested_numdays
                    END as days,
                    lr.approved_start_date,
                    lr.approved_end_date,
                    lr.leaave_state as state,
                    CASE
                        WHEN lr.leaave_state = 2 THEN 'Approved Leave'
                        WHEN lr.leaave_state = 1 THEN 'Waiting Approval'
                        WHEN lr.leaave_state = 0 THEN 'Disapproved Leave'
                    END as status
                FROM 
                    leave_requests lr
                LEFT JOIN
                    employees emp ON lr.employee = emp.employee_code
                LEFT JOIN
                    employees em ON lr.disapproved_by = em.employee_code
                LEFT JOIN
                    employees e ON lr.approved_by = e.employee_code
                LEFT JOIN
                    leave_allocations la ON lr.leave_id = la.leave_id
                WHERE
                    lr.employee =:employee
                ORDER BY 
                    lr.id DESC
                ";
        $smtp = $this->link->prepare($sql);
        $smtp->bindParam(":employee", $employee_code, PDO::PARAM_INT);
        $smtp->execute();
        $data = $smtp->fetchAll(PDO::FETCH_ASSOC);
        return $data;
    }

    public function getLeaveData($employee_code) {
        $sql = "SELECT 
                    e.employee_code,
                    e.fullname,
                    la.leave_name as leave_type, 
                    e.start_date as work_start_date,

                    CASE
                        WHEN lr.leaave_state = 2 THEN ee.fullname
                        WHEN lr.leaave_state = 1 THEN emp.fullname
                        WHEN lr.leaave_state = 0 THEN em.fullname
                    END as operator_name,
                    
                    CASE
                        WHEN lr.leaave_state = 2 THEN lr.approved_start_date
                        ELSE lr.start_date
                    END as start_date,
                    
                    CASE
                        WHEN lr.leaave_state = 2 THEN lr.approved_end_date
                        ELSE lr.end_date
                    END as end_date,
                    
                    CASE
                        WHEN lr.leaave_state = 2 THEN lr.approved_numdays
                        ELSE lr.requested_numdays
                    END as days,

                    CASE
                        WHEN lr.leaave_state = 2 THEN 'Approved Leave'
                        WHEN lr.leaave_state = 1 THEN 'Waiting Approval'
                        WHEN lr.leaave_state = 0 THEN 'Disapproved Leave'
                    END as status,

                    lr.leaave_state as state,
                    SUM(CASE WHEN lr.leaave_state = 2 THEN lr.approved_numdays ELSE 0 END) AS total_taken,
                    COUNT(lr.id) AS leave_requests_count

                FROM employees e
                LEFT JOIN leave_requests lr 
                    ON lr.employee = e.employee_code
                    AND lr.leave_type = 'Annual Leave'
                LEFT JOIN leave_allocations la ON lr.leave_id = la.leave_id
                LEFT JOIN employees emp ON lr.operator = emp.employee_code
                LEFT JOIN employees em ON lr.disapproved_by = em.employee_code
                LEFT JOIN employees ee ON lr.approved_by = ee.employee_code
                WHERE lr.employee = :employee_code
                GROUP BY e.employee_code, e.fullname, e.start_date
                ORDER BY lr.id DESC
";

        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':employee_code', $employee_code, PDO::PARAM_INT);
        $stmt->execute();
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

        $result = [];

        foreach ($rows as $row) {
            $startDate = new DateTime($row['work_start_date']);
            $now = new DateTime();

            // Probation ends 3 months after start date
            $probationEnd = (clone $startDate)->modify('+3 months');

            // Leave accrual only starts after probation
            if ($probationEnd > $now) {
                $monthsWorked = 0;
            } else {
                $monthsWorked = ($now->diff($probationEnd)->y * 12) + $now->diff($probationEnd)->m;
            }

            $accruedDays = round($monthsWorked * 1.5, 1); // 1.5 days per full month
            $totalTaken = (int) $row['total_taken'];
            $totalLeft = max(0, $accruedDays - $totalTaken);

            $yearsWorked = $startDate->diff($now)->y + 1;

            $result[] = [
                'employee_code' => $row['employee_code'],
                'fullname' => $row['fullname'],
                'start_date' => $row['start_date'],
                'end_date' => $row['end_date'],
                'years_worked' => $yearsWorked,
                'total_days' => $accruedDays,
                'total_taken' => $totalTaken,
                'total_left' => $totalLeft,
                'leave_type' => $row['leave_type'],
                'days'       => $row['days'],
                'status'     => $row['status'],
                'state'      => $row['state'],
                'operator_name'      => $row['operator_name'],
                'leave_requests_count' => $row['leave_requests_count']
            ];
        }

        return $result;
    }


}