<?php
    session_start();
    require_once '../vendor/autoload.php'; 

    use PhpOffice\PhpSpreadsheet\IOFactory;
    include_once '../model/stock_model.php';

    $stock_model = new stock_model();

    if ($_SERVER['REQUEST_METHOD'] === 'POST') {

        //FETCH Company products
        if (isset($_POST['action']) && $_POST['action'] === 'fetch_company_products') {
            header('Content-Type: application/json');

            $company_id = $_POST['company_id'] ?? '';

            $products = $stock_model->getProductsByCompany($company_id);

            echo json_encode(['products' => $products]);
            exit();
        }

        // === CHECK FOR DUPLICATES ===
        if (isset($_POST['check_duplicates'])) {
            header('Content-Type: application/json');

            $product_code = trim($_POST['product_code'] ?? '');
            $product_name = trim($_POST['product_name'] ?? '');

            $response = [
                'code_exists' => $stock_model->productCodeExists($product_code),
                'name_exists' => $stock_model->productNameExists($product_name),
                'success' => true
            ];

            echo json_encode($response);
            exit;
        }

        // === CHECK IF PRODUCT NAME OR CODE ALREADY EXISTS FOR DIFFERENT ID ===
        if (isset($_POST['action']) && $_POST['action'] === 'check_product') {
            $product_id = $_POST['product_id'];
            $product_code = trim($_POST['product_code']);
            $product_name = trim($_POST['product_name']);

            $code_exists = $stock_model->productCodeExistsAlready($product_code, $product_id);
            $name_exists = $stock_model->productNameExistsAlready($product_name, $product_id);

            echo json_encode([
                'code_exists' => $code_exists,
                'name_exists' => $name_exists
            ]);
            exit;
        }

        // === ADD PRODUCT ===
        if (isset($_POST['addProduct'])) {
            $product_code = $_POST['product_code'] ?? '';
            $product_name = $_POST['product_name'] ?? '';
            $min_order_level = $_POST['overall_min_level'] ?? '';
            $category = $_POST['category'] ?? '';
            $unit = $_POST['unit'] ?? '';
            $selling_price = isset($_POST['selling_price']) ? floatval($_POST['selling_price']) : 0;


            if (empty($product_code) || empty($product_name) || empty($selling_price) || empty($min_order_level) || empty($category) || empty($unit)) {
                $_SESSION['notification'] = [
                    'type' => 'error',
                    'title' => 'FAILED',
                    'message' => 'Please fill in all required fields.'
                ];
                header("Location: ../manage_products.php");
                exit();
            }

            $product_id = $stock_model->addProduct($product_code, $product_name, $selling_price, $min_order_level, $category, $unit, $_SESSION['sess_employeecode'], 1);

            if ($product_id) {
                $_SESSION['notification'] = [
                    'type' => 'success',
                    'title' => 'Product Added',
                    'message' => 'The product has been added successfully.'
                ];
                header("Location: ../product_details.php?product_id=" . urlencode($product_id));
                exit();
            } else {
                $_SESSION['notification'] = [
                    'type' => 'error',
                    'title' => 'Save Failed',
                    'message' => 'There was an error while adding the product.'
                ];
                header("Location: ../manage_products.php");
                exit();
            }

        }

        // === EDIT PRODUCT ===
        if (isset($_POST['editproduct'])) {
            $product_id = isset($_POST['product_id']) ? intval($_POST['product_id']) : 0;
            $product_code = isset($_POST['product_code']) ? trim($_POST['product_code']) : '';
            $product_name = isset($_POST['product_name']) ? trim($_POST['product_name']) : '';
            $min_order_level = isset($_POST['minimum_level']) ? intval($_POST['minimum_level']) : 0;

            if (isset($_POST['selling_price']) && is_numeric($_POST['selling_price'])) {
                $selling_price = floatval($_POST['selling_price']);
            } else {
                $selling_price = 0.00; // Default or log error
            }

            $category = isset($_POST['category']) ? trim($_POST['category']) : '';
            $unit = isset($_POST['unit']) ? trim($_POST['unit']) : '';
            $product_state = isset($_POST['product_state']) ? trim($_POST['product_state']) : '';


            if (empty($product_code) || empty($product_name) || empty($selling_price) || empty($min_order_level) || empty($category) || empty($unit) || !isset($product_state)) {
                $_SESSION['notification'] = [
                    'type' => 'error',
                    'title' => 'Update Failed',
                    'message' => 'Please fill in all required fields.'
                ];
                header("Location: ../product_details.php?product_id=" . urlencode($product_id));
                exit();
            }

            $result = $stock_model->updateProduct($product_id, $product_code, $selling_price, $product_name, $min_order_level, $category, $unit, $product_state);

            if ($result) {
                $_SESSION['notification'] = [
                    'type' => 'success',
                    'title' => 'Product Updated',
                    'message' => 'The product has been updated successfully.'
                ];
                header("Location: ../product_details.php?product_id=" . urlencode($product_id));
                exit();
            } else {
                $_SESSION['notification'] = [
                    'type' => 'error',
                    'title' => 'Update Failed',
                    'message' => 'There was an error while updating the product.'
                ];
                header("Location: ../product_details.php?product_id=" . urlencode($product_id));
                exit();
            }
        }

        
        // === IMPORT PRODUCTS FROM EXCEL FILE ===
        if (isset($_POST['action']) && $_POST['action'] === 'import_from_file') {
            require_once '../vendor/autoload.php';

            if (!isset($_FILES['import_file']) || $_FILES['import_file']['error'] !== UPLOAD_ERR_OK) {
                $_SESSION['notification'] = [
                    'type' => 'error',
                    'title' => 'Upload Failed',
                    'message' => 'Failed to upload Excel file.'
                ];
                header("Location: ../manage_products.php");
                exit();
            }

            $fileTmpPath = $_FILES['import_file']['tmp_name'];

            try {
                $spreadsheet = IOFactory::load($fileTmpPath);
                $sheet = $spreadsheet->getActiveSheet();
                $rows = $sheet->toArray(null, true, true, true); // A-E

                $products = [];
                $skippedProducts = [];

                // Pre-fetch valid category and unit IDs for reference
                $validCategoryIDs = array_column($stock_model->getCategories(), 'category_id');
                $validUnitIDs = array_column($stock_model->getUnits(), 'unit_id');

                foreach ($rows as $index => $row) {
                    if ($index === 1) continue;

                    $product_code = trim($row['A']);
                    $product_name = trim($row['B']);
                    list($category_id) = explode(' - ', trim($row['C']));
                    list($unit_id) = explode(' - ', trim($row['D']));
                    $min_order_level = (int) trim($row['E']);
                    $selling_price = floatval(trim($row['F']));



                    // Basic required check
                    if (!$product_code || !$product_name || !$category_id || !$unit_id || !$min_order_level || !$selling_price) {
                        continue;
                    }

                    // ID Format check
                    if (!ctype_digit($category_id) || !ctype_digit($unit_id)) {
                        $skippedProducts[] = [
                            'row' => $index,
                            'code' => $product_code,
                            'reason' => 'Invalid category/unit format'
                        ];
                        continue;
                    }

                    // ID Existence check
                    if (!in_array((int)$category_id, $validCategoryIDs)) {
                        $skippedProducts[] = [
                            'row' => $index,
                            'code' => $product_code,
                            'reason' => "Category ID '{$category_id}' not found"
                        ];
                        continue;
                    }

                    if (!in_array((int)$unit_id, $validUnitIDs)) {
                        $skippedProducts[] = [
                            'row' => $index,
                            'code' => $product_code,
                            'reason' => "Unit ID '{$unit_id}' not found"
                        ];
                        continue;
                    }

                    // Duplicate Check
                    if ($stock_model->productCodeExists($product_code)) {
                        $skippedProducts[] = [
                            'row' => $index,
                            'code' => $product_code,
                            'reason' => 'Product code already exists'
                        ];
                        continue;
                    }

                    if ($stock_model->productNameExists($product_name)) {
                        $skippedProducts[] = [
                            'row' => $index,
                            'code' => $product_code,
                            'reason' => 'Product name already exists'
                        ];
                        continue;
                    }

                    $products[] = [
                        'product_code' => $product_code,
                        'product_name' => $product_name,
                        'selling_price' => $selling_price,
                        'category' => (int) $category_id,
                        'unit' => (int) $unit_id,
                        'min_order_level' => $min_order_level
                    ];
                }

                // INSERT VALID PRODUCTS
                foreach ($products as $prod) {
                    $stock_model->addProduct(
                        $prod['product_code'],
                        $prod['product_name'],
                        $prod['selling_price'],
                        $prod['min_order_level'],
                        $prod['category'],
                        $prod['unit'],
                        $_SESSION['sess_employeecode'],
                        1
                    );
                }

                // Final message
                $imported = count($products);
                $skipped = count($skippedProducts);

                $_SESSION['notification'] = [
                    'type' => 'success',
                    'title' => 'Import Summary',
                    'message' => "{$imported} product(s) imported successfully. {$skipped} skipped."
                ];

                // Optional: Store skipped products for frontend display
                $_SESSION['skipped_products'] = $skippedProducts;

                header("Location: ../manage_products.php");
                exit();

            } catch (Exception $e) {
                $_SESSION['notification'] = [
                    'type' => 'error',
                    'title' => 'Parsing Error',
                    'message' => 'Unable to read Excel file: ' . $e->getMessage()
                ];
                header("Location: ../manage_products.php");
                exit();
            }
        }



        // === UNKNOWN ACTION ===
        $_SESSION['notification'] = [
            'type' => 'error',
            'title' => 'Save Failed',
            'message' => 'Unverified User Action'
        ];
        header("Location: ../manage_products.php");
        exit();
    } else {
        $_SESSION['notification'] = [
            'type' => 'error',
            'title' => 'Save Failed',
            'message' => 'Unverified Server Request Method'
        ];
        header("Location: ../manage_products.php");
        exit();
    }
    ?>  
