<?php
    session_start();
    date_default_timezone_set('Africa/Blantyre');

    include_once '../model/stock_model.php';
    include_once '../model/transfers_model.php';

    $stock_model = new stock_model();
    $transfers_model = new transfers_model();

    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
        $action = $_POST['action'] ?? '';

        switch ($action){
            case 'initiate_transfer':
                initiate_transfer($stock_model, $transfers_model);
            break;

            case 'update_quantity':
                update_quantity($stock_model, $transfers_model);
            break;

            case 'update_date':
                update_transfer_date($transfers_model);
            break;

            case 'delete_item':
                delete_transfer_item($transfers_model, $stock_model);
            break;

            case 'finalize_transfer':
                finaliseTransfer($transfers_model,$stock_model);
            break;

            case 'initiate_supplier_shipment':
                initiate_supplier_shipment($stock_model, $transfers_model);
            break;

            default:
                $_SESSION['notification']=[
                    'type' => 'error',
                    'title' => 'FAILED',
                    'message' => 'Unverified Post Activity'
                ];
            header('location:../dashboard.php');
            exit();
        }

    }

    function initiate_supplier_shipment($stock_model, $transfers_model) {
        $shipment_date  = trim($_POST['shipment_date'] ?? '');
        $supplier_id    = trim($_POST['supplier_id'] ?? '');
        $receiver       = trim($_POST['receiver_id'] ?? '');
        $waybill_no = isset($_POST['waybill_no']) ? trim($_POST['waybill_no']) : null;

        if ($waybill_no === '') {
            $waybill_no = null;
        }
        $reference_no   = trim($_POST['reference_no'] ?? '');
        $transport_mode = $_POST['transport_mode'] ?? [];
        $operator       = $_SESSION['sess_employeecode'] ?? 0;

        /* ===================== VALIDATION ===================== */
        if (empty($shipment_date)) { notifyAndRedirect('Shipment Date Missing', 'Please select the shipment date.'); }
        if (empty($supplier_id)) { notifyAndRedirect('Supplier Not Selected', 'Please select a supplier.'); }
        if (empty($receiver)) { notifyAndRedirect('Receiving Company Missing', 'Please select the receiving company.'); }
        if (empty($reference_no)) { notifyAndRedirect('Reference Number Missing', 'Please enter the reference number (PO / Delivery Note).'); }
        if (empty($transport_mode)) { notifyAndRedirect('Transport Mode Missing', 'Please select at least one transport mode.'); }
        if (empty($operator)) { notifyAndRedirect('Session Error', 'Your session has expired. Please login again.', '../login.php'); }
        if (!isValidDate($shipment_date)) { notifyAndRedirect('Invalid Date', 'Please select a valid shipment date.'); }

        /* ===================== DUPLICATE CHECKS ===================== */
        if (!empty($waybill_no) && $transfers_model->waybillExists($waybill_no)) {
            notifyAndRedirect('Duplicate Waybill', 'This waybill number has already been used.');
        }
        if ($transfers_model->referenceExists($reference_no)) {
            notifyAndRedirect('Duplicate Reference', 'This reference/document number has already been used.');
        }

        /* ===================== CREATE SHIPMENT ===================== */
        $transport_mode_str = implode(', ', $transport_mode);
        $items = $_POST['item_id'] ?? [];
        $quantities = $_POST['quantity'] ?? [];
        $prices = $_POST['order_price'] ?? [];
        $num_items = count($items);

        $shipment_id = $transfers_model->createSupplierShipment([
            'shipment_date'  => $shipment_date,
            'supplier_id'    => $supplier_id,
            'receiver_id'    => $receiver,
            'waybill_no'     => $waybill_no,
            'reference_no'   => $reference_no,
            'transport_mode' => $transport_mode_str,
            'operator'       => $operator,
            'num_items'      => $num_items,
            'currency'       => $_POST['currency'] ?? 'USD',
            'total_cost'     => 0,
            'datecaptured'   => date('Y-m-d H:i'),
            'operator'       => $_SESSION['sess_employeecode']
        ]);

        if (!$shipment_id) {
            notifyAndRedirect('Failed', 'Failed to initiate shipment. Please try again.');
        }

        /* ===================== ADD ITEMS ===================== */
            $items = $_POST['item_id'] ?? [];        // product IDs
            $quantities = $_POST['qty'] ?? [];       // quantities per item
            $prices = $_POST['cost_price'] ?? [];    // cost per item

            $transfers_model->addShipmentItems(
                $shipment_id, 
                $supplier_id, 
                $receiver, 
                $items, 
                $quantities, 
                $prices, 
                $operator
            );
                            
        $_SESSION['notification'] = [
            'type' => 'success',
            'title' => 'Shipment Initiated',
            'message' => 'Supplier shipment processed successfully.'
        ];
        header("Location: ../shipment_document.php?shipment_id=$shipment_id");
        exit;
    }

    /* Helper function to show notification and redirect */
    function notifyAndRedirect($title, $message, $url = '../ship_procurement.php') {
        $_SESSION['notification'] = [
            'type' => 'error',
            'title' => $title,
            'message' => $message
        ];
        header("Location: $url");
        exit;
    }


    function finaliseTransfer($transfers_model, $stock_model){
        date_default_timezone_set('Africa/Blantyre'); // ✅ Set timezone
        $transfer_id = trim($_POST['transfer_id'] ?? '');
        $operator = $_SESSION['sess_employeecode'] ?? 0;


        $transfer_details = $transfers_model->getFirstTransferItemDetails($transfer_id);

        if (!$transfer_details) {
            $_SESSION['notification'] = [
                'type' => 'error',
                'title' => 'Error',
                'message' => 'Unable to finalize: transfer details not found.'
            ];
            header("Location: ../initiate_transfer.php?order_number=$transfer_id");
            exit;
        }

        // Add this before calling finalizeStockTransfer()
        if ($stock_model->orderNumberExists($transfer_id)) {
            $_SESSION['notification'] = [
                'type' => 'warning',
                'title' => 'Already Finalized',
                'message' => 'This transfer has already been finalized. No duplicate entries allowed.'
            ];
            header("Location: ../delivery_note.php?order_number=$transfer_id");
            exit;
        }


        $saved = $transfers_model->finalizeStockTransfer(['transfer_id' => $transfer_id, 'transfer_from'   => $transfer_details['transfer_from'], 'transfer_to'     => $transfer_details['transfer_to'], 'transfer_date'   => $transfer_details['transfer_date'], 'operator' => $operator, 'transfer_state'  => 1, 'datecaptured' => date('Y-m-d H:i:s'), 'driver' => trim($_POST['driver'] ?? '')]);

        if ($saved) {
            $_SESSION['notification'] = [
                'type' => 'success',
                'title' => 'Success',
                'message' => 'Stock transfer has been finalized successfully.'
            ];
        header("Location: ../transfer_note.php?order_number=$transfer_id");
        exit;
        } else {
            $_SESSION['notification'] = [
                'type' => 'error',
                'title' => 'Error',
                'message' => 'Failed to finalize the transfer.'
            ];
        header("Location: ../initiate_transfer.php?order_number=$transfer_id");
        exit;
        }

       
    }

    function delete_transfer_item($transfers_model, $stock_model){
        $item_id = intval($_POST['item_id'] ?? 0);
        $transfer_id = trim($_POST['transfer_id'] ?? '');
        $quantity = floatval($_POST['quantity'] ?? 0);
        $company_id = trim($_POST['company_id'] ?? 0);
        $product_id = intval($_POST['product_id'] ?? 0);

        if ($item_id === 0 || empty($transfer_id)) {
            $_SESSION['notification'] = [
                'type' => 'error',
                'title' => 'Failed!',
                'message' => 'Invalid item or transfer ID.'
            ];
            header("Location: ../initiate_transfer.php?order_number=$transfer_id");
            exit;
        }


        $company_stock_data = $stock_model->getCompanyProductStock($company_id, $product_id);
        $current_quantity = floatval($company_stock_data['quantity'] ?? 0);
        $new_quantity = $current_quantity + $quantity;

        $deleted = $transfers_model->deleteTransferItemById($item_id);
        // Restore stock
        $stock_model->updateSourceQuantity($company_id, $product_id, $new_quantity);

        if ($deleted) {
            $_SESSION['notification'] = [
                'type' => 'success',
                'title' => 'Item Removed!',
                'message' => 'The transfer item was successfully removed.'
            ];
        } else {
            $_SESSION['notification'] = [
                'type' => 'error',
                'title' => 'Failed!',
                'message' => 'Could not delete the item. Please try again.'
            ];
        }

        header("Location: ../initiate_transfer.php?order_number=$transfer_id");
        exit;
    }


    function initiate_transfer($stock_model, $transfers_model){
        date_default_timezone_set('Africa/Blantyre'); // ✅ Set timezone

        $order_number = trim($_POST['order_number'] ?? '');
        $company_id   = trim($_POST['company_id'] ?? '');
        $transfer_date = trim($_POST['transfer_date'] ?? '');
        $product_id        = filter_var($_POST['product_id'] ?? '', FILTER_VALIDATE_INT);
        $quantity          = filter_var($_POST['quantity'] ?? '', FILTER_VALIDATE_FLOAT);
        $remaining_quantity = filter_var($_POST['remaining'] ?? '', FILTER_VALIDATE_FLOAT);
        $destination_id    = filter_var($_POST['destination'] ?? '', FILTER_VALIDATE_INT);
        $operator = $_SESSION['sess_employeecode'] ?? 0;
        $transfer_state = 1;

        // Timestamp
        $datecaptured = date('Y-m-d H:i:s');

    
        $company_stock_data = $stock_model->getCompanyProductStock($company_id, $product_id);
        $current_quantity = floatval($company_stock_data['quantity'] ?? 0);

        // Step 1: Field validation
        if (empty($company_id) || $product_id === 0 || $destination_id === 0 || $quantity <= 0) {
            $_SESSION['notification'] = [
                'type' => 'error',
                'title' => 'Missing Fields!',
                'message' => 'Please fill in all required fields correctly.'
            ];
            header("Location: ../initiate_transfer.php?order_number=$order_number");
            exit;
        }

        if($operator === 0 || empty($operator)){
             $_SESSION['notification'] = [
                'type' => 'error',
                'title' => 'User Failure!',
                'message' => 'Please login to verify your identity.'
            ];
            header("Location: ../../index.php ");
            exit;
        }

        if (!isValidDate($transfer_date)) {
            $_SESSION['notification'] = [
                'type' => 'error',
                'title' => 'Invalid Date',
                'message' => 'Please select a valid transfer date.'
            ];
            header("Location: ../initiate_transfer.php?order_number=$order_number");
            exit;
        }


        // Step 2: Check product & quantity
        if ($current_quantity <= 0) {
            $_SESSION['notification'] = [
                'type' => 'error',
                'title' => 'Out of Stock!',
                'message' => 'The selected product has no stock in this company.'
            ];
            header("Location: ../initiate_transfer.php?order_number=$order_number");
            exit;
        }

        // Step 3: Check if requested quantity exceeds available stock
        if ($quantity > $current_quantity) {
            $_SESSION['notification'] = [
                'type' => 'error',
                'title' => 'Quantity Exceeded!',
                'message' => 'You have requested more than the available stock (' . number_format($current_quantity, 0) . ').'
            ];
            header("Location: ../initiate_transfer.php?order_number=$order_number");
            exit;
        }

        // Step 4: Get total pending transfer quantity for this product and company
        $pending_quantity = $transfers_model->getPendingQuantityForProduct($company_id, $product_id);

        // Step 5: Ensure total requested + pending does not exceed available stock
        $total_requested = $quantity + $pending_quantity;

        if ($total_requested > $remaining_quantity) {
            $_SESSION['notification'] = [
                'type' => 'error',
                'title' => 'Stock Limit Exceeded',
                'message' => 'The total requested quantity (' . number_format($total_requested, 2) . ') exceeds the available stock (' . number_format($remaining_quantity, 2) . ').'
            ];
            header("Location: ../initiate_transfer.php?order_number=$order_number");
            exit;
        }

        // Step 6: Ensure product is registered at destination
        if (!$stock_model->productExistsAtDestination($product_id, $destination_id)) {
            $_SESSION['notification'] = [
                'type' => 'error',
                'title' => 'Invalid Location',
                'message' => 'The selected product is not attached to the selected location. Please check first.'
            ];
            header("Location: ../initiate_transfer.php?order_number=$order_number");
            exit;
        }

        if (strtotime($transfer_date) > time()) {
            $_SESSION['notification'] = [
                'type' => 'error',
                'title' => 'Invalid Date',
                'message' => 'You cannot select a future transfer date.'
            ];
            header("Location: ../initiate_transfer.php?order_number=$order_number");
            exit;
        }

        // Step 7: Prevent duplicate product in same order
        if ($transfers_model->isProductAlreadyInTransfer($order_number, $product_id)) {
            $_SESSION['notification'] = [
                'type' => 'warning',
                'title' => 'Already Added!',
                'message' => 'This product is already in the transfer list. Please update its quantity instead.'
            ];
            header("Location: ../initiate_transfer.php?order_number=$order_number");
            exit;
        }


        $receiver_stock = $stock_model->getProductQuantityAtDestination($product_id, $destination_id);
        $receiver_opening_stock = floatval($receiver_stock['quantity'] ?? 0);

        $sender_closing_stock = $remaining_quantity - $quantity;
        $receiver_closing_stock = $receiver_opening_stock + $quantity;

        // Prepare data for saving
        $data = [
            'transfer_id'             => $order_number,
            'product_id'              => $product_id,
            'transfer_from'           => $company_id,
            'transfer_to'             => $destination_id,
            'sender_opening_stock'    => $current_quantity,
            'receiver_opening_stock'  => $receiver_opening_stock,
            'qty_initiated'           => $quantity,
            'original_qty_initiated'  => $quantity,
            'sender_closing_stock'    => $sender_closing_stock,
            'receiver_closing_stock'  => $receiver_closing_stock,
            'operator'                => $operator,
            'transfer_date'           => $transfer_date,
            'transfer_state'          => $transfer_state,
            'datecaptured'            => $datecaptured
        ];


        $result = $transfers_model->transferStock($data);
        
        //update sender stock since items have been allocated for transfer
        $stock_model->updateSourceQuantity($company_id, $product_id, $sender_closing_stock);

        if ($result) {
            $_SESSION['notification'] = [
                'type' => 'success',
                'title' => 'Success!',
                'message' => 'Item successfully added to transfer list.'
            ];
        } else {
            $_SESSION['notification'] = [
                'type' => 'error',
                'title' => 'Failed!',
                'message' => 'Could not save transfer item. Please try again.'
            ];
        }

        header("Location: ../initiate_transfer.php?order_number=$order_number");
        exit;



    }

    function update_quantity($stock_model, $transfers_model) {
        $item_id       = intval($_POST['item_id'] ?? 0);
        $transfer_id   = trim($_POST['transfer_id'] ?? '');
        $new_quantity  = floatval($_POST['new_quantity'] ?? 0);

        if ($item_id === 0 || empty($transfer_id) || $new_quantity <= 0) {
            $_SESSION['notification'] = [
                'type' => 'error',
                'title' => 'Invalid Data',
                'message' => 'Some important data is missing, Try again.'
            ];
            header("Location: ../initiate_transfer.php?order_number=$transfer_id");
            exit;
        }

        // Fetch item details
        $item = $transfers_model->getItemDetailsById($item_id);

        if (!$item) {
            $_SESSION['notification'] = [
                'type' => 'error',
                'title' => 'Not Found',
                'message' => 'Transfer item not found.'
            ];
            header("Location: ../initiate_transfer.php?order_number=$transfer_id");
            exit;
        }

        $product_id   = $item['product_id'];
        $company_id   = $item['transfer_from'];
        $destination  = $item['transfer_to'];
        $old_qty      = floatval($_POST['old_quantity'] ?? 0);
        $original_quantity = floatval($item['original_qty_initiated'] ?? 0);

        if($new_quantity == $original_quantity){
            $_SESSION['notification'] = [
                'type' => 'warning',
                'title' => 'No Changes',
                'message' => 'The new quantity is the same as the previously entered amount.'
            ];
            header("Location: ../initiate_transfer.php?order_number=$transfer_id");
            exit;
        }
        // Get current stock at sender
        $company_stock_data = $stock_model->getCompanyProductStock($company_id, $product_id);
        $sender_current_stock = floatval($company_stock_data['quantity'] ?? 0);

        $sender_opening_stock = $sender_current_stock + $old_qty;

        // Get stock at receiver
        $receiver_stock_data = $stock_model->getProductQuantityAtDestination($product_id, $destination);
        $receiver_stock = floatval($receiver_stock_data['quantity'] ?? 0);
          
        // Recalculate stock
        $sender_closing_stock   = ($sender_current_stock + $old_qty) - $new_quantity;
        $receiver_closing_stock = $receiver_stock + $new_quantity;
        
        //Update quantity i product transfer table
        $stock_model->updateSourceQuantity($company_id, $product_id, $sender_closing_stock);

       $updated = $transfers_model->updateTransferQuantity($item_id, $new_quantity, $sender_opening_stock, $receiver_stock, $sender_closing_stock, $receiver_closing_stock);

       $_SESSION['notification'] = $updated
            ? ['type' => 'success', 'title' => 'Updated!', 'message' => 'Quantity updated successfully.']
            : ['type' => 'error', 'title' => 'Failed!', 'message' => 'Failed to update quantity.'];

        header("Location: ../initiate_transfer.php?order_number=$transfer_id");
        exit;
    }

    function update_transfer_date($transfers_model) {
        $item_id      = intval($_POST['item_id'] ?? 0);
        $transfer_id  = trim($_POST['transfer_id'] ?? '');
        $new_date = isset($_POST['new_date']) ? trim($_POST['new_date']) : '';
        $new_date = date('Y-m-d', strtotime($new_date)); // Normalize format


        $transfer_item = $transfers_model->getItemDetailsById($item_id);
        $original_date = $transfer_item['transfer_date'] ?? '';


        // Validate
        if (empty($transfer_id) || !isValidDate($new_date)) {
            $_SESSION['notification'] = [
                'type' => 'error',
                'title' => 'Invalid Data',
                'message' => 'Please enter a valid date.'
            ];
            header("Location: ../initiate_transfer.php?order_number=$transfer_id");
            exit;
        }


        if ($new_date === $original_date) {
            $_SESSION['notification'] = [
                'type' => 'warning',
                'title' => 'No Changes',
                'message' => 'The selected date is the same as the current transfer date.'
            ];
            header("Location: ../initiate_transfer.php?order_number=$transfer_id");
            exit;
        }


        // Prevent future dates
        if (strtotime($new_date) > time()) {
            $_SESSION['notification'] = [
                'type' => 'error',
                'title' => 'Invalid Date',
                'message' => 'You cannot select a future transfer date.'
            ];
            header("Location: ../initiate_transfer.php?order_number=$transfer_id");
            exit;
        }

        $updated = $transfers_model->updateTransferDate($transfer_id, $new_date);

        $_SESSION['notification'] = $updated
            ? ['type' => 'success', 'title' => 'Updated!', 'message' => 'Transfer date updated.']
            : ['type' => 'error', 'title' => 'Failed!', 'message' => 'Could not update transfer date.'];

        header("Location: ../initiate_transfer.php?order_number=$transfer_id");
        exit;
    }



    function isValidDate($date, $format = 'Y-m-d') {
        $d = DateTime::createFromFormat($format, $date);
        return $d && $d->format($format) === $date;     
    }