<?php
    require 'vendor/autoload.php';

    require_once 'model/stock_model.php';

    $stock_model = new stock_model();

    use PhpOffice\PhpSpreadsheet\Spreadsheet;
    use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
    use PhpOffice\PhpSpreadsheet\Cell\DataValidation;

    // Fetch from DB
    $categories = $stock_model->getCategories();
    $units = $stock_model->getUnits();

    // Init Spreadsheet
    $spreadsheet = new Spreadsheet();
    $sheet = $spreadsheet->getActiveSheet();

    // Headers
    $headers = ['product code', 'product name', 'category', 'unit', 'minimum order level', 'selling price'];
    $col = 'A';
    foreach ($headers as $header) {
        $sheet->setCellValue($col . '1', strtoupper($header));
        $col++;
    }

    // Set auto widths
    foreach (range('A', 'F') as $columnID) {
        $sheet->getColumnDimension($columnID)->setAutoSize(true);
    }

    // Define named lists for categories and units on a hidden sheet
    $hiddenSheet = $spreadsheet->createSheet();
    $spreadsheet->setActiveSheetIndex(1);
    $hiddenSheet->setTitle('dropdowns');

    $row = 1;
    foreach ($categories as $cat) {
        $hiddenSheet->setCellValue("B{$row}", $cat['category_id'] . ' - ' . $cat['categoryname']); // <-- change here
        $row++;
    }


    $row = 1;
    foreach ($units as $unit) {
        $hiddenSheet->setCellValue("E{$row}", $unit['unit_id'] . ' - ' . $unit['unitname']); // <-- change here
        $row++;
    }


    // Hide the dropdown sheet
    $hiddenSheet->setSheetState(\PhpOffice\PhpSpreadsheet\Worksheet\Worksheet::SHEETSTATE_HIDDEN);

    // Return to first sheet
    $spreadsheet->setActiveSheetIndex(0);

    // Apply category dropdown (show name)
    for ($i = 2; $i <= 100; $i++) {
        $validation = $sheet->getCell("C{$i}")->getDataValidation();
        $validation->setType(DataValidation::TYPE_LIST);
        $validation->setAllowBlank(false);
        $validation->setShowDropDown(true);
        $validation->setFormula1('=dropdowns!$B$1:$B$' . count($categories)); // <-- show NAMES
    }

    // Apply unit dropdown (show name)
    for ($i = 2; $i <= 100; $i++) {
        $validation = $sheet->getCell("D{$i}")->getDataValidation();
        $validation->setType(DataValidation::TYPE_LIST);
        $validation->setAllowBlank(false);
        $validation->setShowDropDown(true);
        $validation->setFormula1('=dropdowns!$E$1:$E$' . count($units)); // <-- show NAMES
    }


    // Validate min_order_level (integer between 5-100)
    for ($i = 2; $i <= 100; $i++) {
        $valid = $sheet->getCell("E{$i}")->getDataValidation();
        $valid->setType(DataValidation::TYPE_WHOLE);
        $valid->setOperator(DataValidation::OPERATOR_BETWEEN);
        $valid->setFormula1(5);
        $valid->setFormula2(100);
        $valid->setAllowBlank(false);
        $valid->setShowErrorMessage(true); // <-- Important
        $valid->setErrorTitle('Invalid Entry');
        $valid->setError('Min Order Level must be a whole number between 5 and 100.');
    }



    // Prevent duplicate product_code using Excel formula
    for ($i = 2; $i <= 100; $i++) {
        $validation = $sheet->getCell("A{$i}")->getDataValidation();
        $validation->setType(DataValidation::TYPE_CUSTOM);
        $validation->setAllowBlank(false);
        $validation->setShowErrorMessage(true);
        $validation->setErrorTitle('Duplicate Entry');
        $validation->setError('Product Code must be unique.');
        $validation->setFormula1("=COUNTIF(A:A,A{$i})=1");
    }
    
    // ✅ New: Validate selling_price (decimal ≥ 0.01)
    for ($i = 2; $i <= 100; $i++) {
        $valid = $sheet->getCell("F{$i}")->getDataValidation();
        $valid->setType(DataValidation::TYPE_DECIMAL);
        $valid->setOperator(DataValidation::OPERATOR_GREATERTHANOREQUAL);
        $valid->setFormula1(100.00);
        $valid->setAllowBlank(false);
        $valid->setShowErrorMessage(true);
        $valid->setErrorTitle('Invalid Price');
        $valid->setError('Selling Price must be a number greater than or equal to 100.00.');
    }

    // Download Excel
    header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
    header('Content-Disposition: attachment; filename="product_upload_template.xlsx"');
    header('Cache-Control: max-age=0');

    $writer = new Xlsx($spreadsheet);
    $writer->save("php://output");
    exit;

    // --- Fetching functions ---
    function getCategories($db) {
        $stmt = $db->query("SELECT category_id, categoryname FROM product_categories");
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    function getUnits($db) {
        $stmt = $db->query("SELECT unit_id, unit_name FROM units");
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
