<?php

use PhpOffice\PhpSpreadsheet\Calculation\TextData\Concatenate;

session_start();
date_default_timezone_set('Africa/Blantyre');

include_once '../model/stock_model.php';
include_once '../model/sales_model.php';
include_once '../model/common_model.php';
include_once '../model/payments_model.php';
include_once '../model/customers_model.php';

$stock_model  = new stock_model();
$payments_model = new payments_model();
$sales_model  = new sales_model();
$common_model = new common_model();
$customers_model = new customers_model();

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {

    $action = $_POST['action'];

    switch ($action) {

        case 'save_invoice':
            // Pass the existing PDO connection found in sales_model->link
            saveInvoice($stock_model, $sales_model, $common_model, $sales_model->link);
        break;

        case 'fetch_active_invoices':
            fetchActiveInvoices($sales_model);
        break;

        case 'get_scrap_amount':
            fetchScrapAmount($sales_model);
            break;

        case 'save_cash_sale':
            // Pass the existing PDO connection found in sales_model->link
            saveCashSale($stock_model, $sales_model, $common_model, $payments_model, $customers_model, $sales_model->link);
        break;

        case 'check_transid':
            checkTransactionID($sales_model);
            break;

        case 'request_dn_cancel':
            deliveryNoteCancelRequest($sales_model);
            break;
        
        case 'check_pending_delivery_notes':
            checkPendingDeliveryNotes($sales_model);
            break;

        case 'get_delivery_note_items':
            getDeliveryNoteItems($sales_model);
            break;
        case 'get_delivery_note_details':
            deliveryNoteData($sales_model);
            break;
        
        case 'save_invoice_dn':
            saveInvoiceFromDN($stock_model, $sales_model, $common_model, $sales_model->link);
            break;
        default:
            echo json_encode(['status' => 'error', 'message' => 'Invalid action']);
        break;
    }
}

function saveInvoice($stock_model, $sales_model, $common_model, $db){
    try {
        $db->beginTransaction();

        $invoice_number = $_POST['invoice_number'] ?? null;
        $customer_id    = $_POST['customer_id'] ?? null;
        $items          = $_POST['items'] ?? [];
        $discount       = floatval($_POST['discount'] ?? 0);
        $invoice_date   = $_POST['invoice_date'] ?? date('Y-m-d');
        $warehouse_id   = $_POST['warehouse_id'] ?? 1;
        $debug          = intval($_POST['debug'] ?? 0);
        $vat_applicable = filter_var($_POST['vat_applicable'] ?? false, FILTER_VALIDATE_BOOLEAN);
        $lpo_number     = $_POST['lpoNumber'];

        if (!$customer_id) {
            echo json_encode(['status'=>'error','message'=>'Missing customer details']);
            exit;
        }
        
        if (!$items || !is_array($items) || count($items) == 0) {
            echo json_encode(['status'=>'error','message'=>'Select at least one item for the invoice']);
            exit;
        }
        
        if (!$invoice_number || trim($invoice_number) === '') {
            echo json_encode(['status'=>'error','message'=>'Invalid invoice number']);
            exit;
        }

        if ($discount < 0) {
            echo json_encode(['status'=>'error','message'=>'Invalid discount amount']);
            exit;
        }

        if (!strtotime($invoice_date)) {
            echo json_encode(['status'=>'error','message'=>'Invalid invoice date']);
            exit;
        }

        if (!$warehouse_id) {
            echo json_encode(['status'=>'error','message'=>'Invalid warehouse selection']);
            exit;
        }

        if (!$lpo_number) {
            echo json_encode(['status'=>'error','message'=>'Please Enter LPO Number for this Invoice']);
            exit;
        }

        // -------------------------------------------------------
        // CHECK IF INVOICE NUMBER IS ALREADY USED
        // IF USED, GENERATE NEW ONE
        // -------------------------------------------------------
        if ($sales_model->invoiceNumberExists($invoice_number)) {
            $invoice_number = getNextInvoiceNumber($warehouse_id, 'warehouse');
        }

        if($sales_model->LpoNumberExists($lpo_number)){
            echo json_encode(['status'=>'error', 'message'=>'LPO Number used for a different Invoice, Please Cross check LPO Number']);
            exit;
        }

        // VAT RATE
        $companyDetails = $common_model->getCompanyDetails();
        $VAT_RATE = floatval($companyDetails['vat_percentage']); // e.g. 16.5
        
        $running_total = 0;
        $subtotal      = 0;
        $total_tax     = 0;

        $computed_items = [];

        // ---------------------------------------------------
        // LOOP ITEMS
        // ---------------------------------------------------
        foreach ($items as $item) {

            $product_code  = $item['product_code'];
            $price_inc_vat = floatval($item['price']); // ALWAYS VAT-inclusive from UI
            $qty           = intval($item['qty']);

            if ($qty <= 0) {
                $db->rollBack();
                echo json_encode(['status'=>'error','message'=>'Invalid quantity']);
                exit;
            }

            $opening_stock = $stock_model->getWarehouseQuantity($product_code, $warehouse_id);
            $closing_stock = $opening_stock - $qty;

            if ($vat_applicable) {
                $price_ex_vat = $price_inc_vat / (1 + $VAT_RATE);
            } else {
                $price_ex_vat = $price_inc_vat;
            }


            $line_subtotal = $price_ex_vat * $qty;                 // EX VAT
            $line_tax      = $vat_applicable ? ($line_subtotal * $VAT_RATE) : 0;
            $line_total    = $line_subtotal + $line_tax;           // INC VAT


             // Price * Qty (VAT inclusive)
            
            $running_total += $line_total;
            $subtotal      += $line_subtotal;
            $total_tax     += $line_tax;

            $computed_items[] = [
                'product_code'  => $product_code,
                'qty'           => $qty,
                'price'         => round($price_ex_vat, 2), // ✅ STORE EX VAT PRICE
                'line_total'    => round($line_total, 2),
                'line_subtotal' => round($line_subtotal, 2),
                'line_tax'      => round($line_tax, 2),
                'opening_stock' => $opening_stock,
                'closing_stock' => $closing_stock,
                'warehouse_id'  => $warehouse_id
            ];
        }

        // FINAL TOTAL
        $total_bill = $running_total - $discount; // Subtract discount from total

        // Due date +21 days
        $due_date = date('Y-m-d', strtotime($invoice_date . ' +21 days'));

        // ---------------------------------------------------
        // DEBUG MODE – DO NOT SAVE ANYTHING
        // ---------------------------------------------------
        if ($debug === 1) {

            $db->rollBack(); // ensure nothing commits

            echo json_encode([
                'status'         => 'preview',
                'message'        => 'Preview only – no data saved.',
                'invoice_number' => $invoice_number,
                'invoice_date'   => $invoice_date,
                'due_date'       => $due_date,
                'computed_items' => $computed_items,
                'subtotal'       => round($subtotal, 2),
                'total_tax'      => round($total_tax, 2),
                'discount'       => round($discount, 2),
                'total_bill'     => round($total_bill, 2),
                'Vat Select'     => $vat_applicable
            ]);
            exit;
        }

        // ---------------------------------------------------
        // SAVE MODE — INSERT ITEMS FIRST
        // ---------------------------------------------------
        foreach ($computed_items as $ci) {

            $row = [
                'sales_number'  => $invoice_number,
                'product_code'  => $ci['product_code'],
                'type'          => 'Customer Order',
                'department'    => $warehouse_id,
                'opening_stock' => $ci['opening_stock'],
                'quantity'      => $ci['qty'],
                'closing_stock' => $ci['closing_stock'],
                'selling_price' => $ci['price'],
                'subtotal'      => round($ci['line_subtotal'], 2),
                'vat_charge'    => $VAT_RATE,
                'tax'           => round($ci['line_tax'], 2),
                'total'         => round($ci['line_total'], 2),
                'date_captured' => date('Y-m-d H:i:s'),
                'operator'      => $_SESSION['sess_employeecode'] ?? 0,
                'status'        => 1
            ];

            // Insert item
            if (!$sales_model->insertSalesOrder($row)) {
                $db->rollBack();
                echo json_encode(['status' => 'error', 'message' => 'Failed to save item: '.$ci['product_code']]);
                exit;
            }

            // Update warehouse stock
            $sql = "UPDATE 
                        warehouse_products 
                    SET 
                        quantity = :closing_stock 
                    WHERE 
                        product_id = :product_code 
                        AND warehouse_id = :warehouse_id";

            $stmt = $db->prepare($sql);
            $updated = $stmt->execute([
                ':closing_stock' => $ci['closing_stock'],
                ':product_code'  => $ci['product_code'],
                ':warehouse_id'  => $warehouse_id
            ]);

            if (!$updated) {
                $db->rollBack();
                echo json_encode(['status' => 'error', 'message' => 'Failed to update stock for: '.$ci['product_code']]);
                exit;
            }
        }

        // ---------------------------------------------------
        // INSERT INTO saved_orders
        // ---------------------------------------------------
        $orderData = [
            'order_number'   => $invoice_number,
            'customer'       => $customer_id,
            'subtotal'       => round($subtotal, 2),
            'total_tax'      => round($total_tax, 2),
            'discount'       => round($discount, 2),
            'total_bill'     => round($total_bill, 2),
            'warehouse_id'   => $warehouse_id,
            'orderdate'      => $invoice_date,
            'due_date'       => $due_date,
            'datecaptured'   => date('Y-m-d H:i:s'),
            'date_cancelled' => null,
            'operator'       => $_SESSION['sess_employeecode'] ?? 0,
            'order_state'    => 0,
            'lpo_number'     => $lpo_number
        ];

        $saved_order_id = $sales_model->insertSavedOrder($orderData);

        if (!$saved_order_id) {
            $db->rollBack();
            echo json_encode(['status'=>'error','message'=>'Failed to save invoice summary']);
            exit;
        }

        $db->commit();

        echo json_encode([
            'status'     => 'success',
            'invoice_no' => $saved_order_id,
            'invoice_id' => $invoice_number
        ]);

    } catch (Exception $e) {

        $db->rollBack();
        echo json_encode(['status'=>'error','message'=>$e->getMessage()]);
    }
}

function saveInvoiceFromDN($stock_model, $sales_model, $common_model, $db){
    try {
        $db->beginTransaction();

        $invoice_number = $_POST['invoice_number'] ?? null;
        $dn_number      = $_POST['dn_number'];
        $customer_id    = $_POST['customer_id'] ?? null;
        $items          = $_POST['items'] ?? [];
        $discount       = floatval($_POST['discount'] ?? 0);
        $invoice_date   = $_POST['invoice_date'] ?? date('Y-m-d');
        $warehouse_id   = $_POST['warehouse_id'] ?? 1;
        $debug          = intval($_POST['debug'] ?? 0);
        $vat_applicable = filter_var($_POST['vat_applicable'] ?? false, FILTER_VALIDATE_BOOLEAN);
        $lpo_number     = $_POST['lpo_number'];
        $dn_state       = 2;
        $date_appoved   = date('Y-m-d H:i:s');

        if (!$customer_id) {
            echo json_encode(['status'=>'error','message'=>'Missing customer details']);
            exit;
        }
        
        if (!$items || !is_array($items) || count($items) == 0) {
            echo json_encode(['status'=>'error','message'=>'Select at least one item for the invoice']);
            exit;
        }
        
        if (!$invoice_number || trim($invoice_number) === '') {
            echo json_encode(['status'=>'error','message'=>'Invalid invoice number']);
            exit;
        }

        if ($discount < 0) {
            echo json_encode(['status'=>'error','message'=>'Invalid discount amount']);
            exit;
        }

        if (!strtotime($invoice_date)) {
            echo json_encode(['status'=>'error','message'=>'Invalid invoice date']);
            exit;
        }

        if (!$warehouse_id) {
            echo json_encode(['status'=>'error','message'=>'Invalid warehouse selection']);
            exit;
        }

        if (!$lpo_number) {
            echo json_encode(['status'=>'error','message'=>'Please Enter LPO Number for this Invoice']);
            exit;
        }

        // -------------------------------------------------------
        // CHECK IF INVOICE NUMBER IS ALREADY USED
        // IF USED, GENERATE NEW ONE
        // -------------------------------------------------------
        if ($sales_model->invoiceNumberExists($invoice_number)) {
            $invoice_number = getNextInvoiceNumber($warehouse_id, 'warehouse');
        }

        if($sales_model->LpoNumberExists($lpo_number)){
            echo json_encode(['status'=>'error', 'message'=>'LPO Number used for a different Invoice, Please Cross check LPO Number']);
            exit;
        }

        // VAT RATE
        $companyDetails = $common_model->getCompanyDetails();
        $VAT_RATE = floatval($companyDetails['vat_percentage']); // e.g. 16.5
        
        $running_total = 0;
        $subtotal      = 0;
        $total_tax     = 0;

        $computed_items = [];

        // ---------------------------------------------------
        // LOOP ITEMS
        // ---------------------------------------------------

        $update_sales_order = $sales_model->updateSalesOrder($dn_number, $invoice_number);
        if (!$update_sales_order) {
            $db->rollBack();
            echo json_encode(['status'=>'error','message'=>'Failed to save update invoice items details']);
            exit;
        }
        foreach ($items as $item) {

            $product_code  = intval($item['product_code']);
            $price_inc_vat = floatval($item['price']); // ALWAYS VAT-inclusive from UI
            $qty           = intval($item['qty']);
            $line_id       = intval($item['item_id']);

            if ($qty <= 0) {
                $db->rollBack();
                echo json_encode(['status'=>'error','message'=>'Invalid quantity']);
                exit;
            }

            $opening_stock = $stock_model->getWarehouseQuantity($product_code, $warehouse_id);
            $closing_stock = $opening_stock - $qty;

            if ($vat_applicable) {
                $price_ex_vat = $price_inc_vat / (1 + $VAT_RATE);
            } else {
                $price_ex_vat = $price_inc_vat;
            }


            $line_subtotal = $price_ex_vat * $qty;                 // EX VAT
            $line_tax      = $vat_applicable ? ($line_subtotal * $VAT_RATE) : 0;
            $line_total    = $line_subtotal + $line_tax;           // INC VAT


             // Price * Qty (VAT inclusive)
            
            $running_total += $line_total;
            $subtotal      += $line_subtotal;
            $total_tax     += $line_tax;

            $computed_items[] = [
                'product_code'  => $product_code,
                'qty'           => $qty,
                'price'         => round($price_ex_vat, 2), // ✅ STORE EX VAT PRICE
                'line_total'    => round($line_total, 2),
                'line_subtotal' => round($line_subtotal, 2),
                'line_tax'      => round($line_tax, 2),
                'opening_stock' => $opening_stock,
                'closing_stock' => $closing_stock,
                'warehouse_id'     => $warehouse_id
            ];

            // --------------------------------------------
            // UPDATE SALES ORDER ITEM (SET SELLING VALUES)
            // --------------------------------------------
            $updateItem = $sales_model->updateSalesOrderItemFromInvoice($line_id, $product_code,
                [
                    'selling_price' => round($price_ex_vat, 2),
                    'line_subtotal' => round($line_subtotal, 2),
                    'tax_amount'    => round($line_tax, 2),
                    'line_total'    => round($line_total, 2)
                ]
            );

            if ($updateItem <=0) {
                $db->rollBack();
                echo json_encode([
                    'status'=>'error',
                    'message'=>'Failed to update sales order item values'
                ]);
                exit;
            }

        }

        // FINAL TOTAL
        $total_bill = $running_total - $discount; // Subtract discount from total

        // Due date +21 days
        $due_date = date('Y-m-d', strtotime($invoice_date . ' +21 days'));
        
        // ---------------------------------------------------
        // DEBUG MODE – DO NOT SAVE ANYTHING
        // ---------------------------------------------------
        if ($debug === 1) {

            $db->rollBack(); // ensure nothing commits

            echo json_encode([
                'status'         => 'preview',
                'message'        => 'Preview only – no data saved.',
                'invoice_number' => $invoice_number,
                'invoice_date'   => $invoice_date,
                'due_date'       => $due_date,
                'computed_items' => $computed_items,
                'subtotal'       => round($subtotal, 2),
                'total_tax'      => round($total_tax, 2),
                'discount'       => round($discount, 2),
                'total_bill'     => round($total_bill, 2),
                'Vat Select'     => $vat_applicable
            ]);
            exit;
        }
        /* ---------------------------------------------------
                     INSERT INTO saved_orders
        ----------------------------------------------------*/
        $orderData = [
            'order_number'   => $invoice_number,
            'customer'       => $customer_id,
            'subtotal'       => round($subtotal, 2),
            'total_tax'      => round($total_tax, 2),
            'discount'       => round($discount, 2),
            'total_bill'     => round($total_bill, 2),
            'warehouse_id'   => $warehouse_id,
            'orderdate'      => $invoice_date,
            'due_date'       => $due_date,
            'datecaptured'   => date('Y-m-d H:i:s'),
            'date_cancelled' => null,
            'operator'       => $_SESSION['sess_employeecode'] ?? 0,
            'order_state'    => 0,
            'lpo_number'     => $lpo_number
        ];

        $saved_order_id = $sales_model->insertSavedOrder($orderData);

        if (!$saved_order_id) {
            $db->rollBack();
            echo json_encode(['status'=>'error','message'=>'Failed to save invoice summary']);
            exit;
        }

        $updateDN = $sales_model->updatedeliveryNote($dn_number, $invoice_number, $_SESSION['sess_employeecode'], $date_appoved, $dn_state);
        if (!$updateDN) {
            $db->rollBack();
            echo json_encode(['status'=>'error','message'=>'Failed to save update deivery note data']);
            exit;
        }

        
        $db->commit();

        echo json_encode([
            'status'     => 'success',
            'invoice_no' => $saved_order_id,
            'invoice_id' => $invoice_number
        ]);
        
        }
    catch (Exception $e) {

        $db->rollBack();
        echo json_encode(['status'=>'error','message'=>$e->getMessage()]);
    }
}

function deliveryNoteData($sales_model){

    $dn_number = trim($_POST['dn_id'] ?? '');

    if ($dn_number === '') {
        echo json_encode([
            'success' => false,
            'message' => 'Invalid delivery note number'
        ]);
        return;
    }

    // Get header
    $delivery_note = $sales_model->getDeliveryNoteDetails($dn_number);

    // Get items
    $items = $sales_model->getInvoiceItems($dn_number);

    if (!$delivery_note) {
        echo json_encode([
            'success' => false,
            'message' => 'Delivery note not found'
        ]);
        return;
    }

    echo json_encode([
        'success' => true,
        'delivery_note' => $delivery_note,
        'items' => $items
    ]);
}



function getDeliveryNoteItems($sales_model){
    $dn_number = $_POST['dn_number'] ?? '';
    if (empty($dn_number)) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Invalid DN number.'
        ]);
        exit;
    }

    $items = $sales_model->getInvoiceItems($dn_number);

    echo json_encode([
        'status' => 'success',
        'items' => $items
    ]);
    exit;
}

function checkPendingDeliveryNotes($sales_model){
    $customer_id = intval($_POST['customer_id'] ?? 0);
    if ($customer_id <= 0) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Invalid customer.'
        ]);
        exit;
    }

    $delivery_notes = $sales_model->getPendingDeliveryNotesByCustomer($customer_id);
    $count = count($delivery_notes);

    if ($count === 0) {
        echo json_encode([
            'status' => 'none'
        ]);
    } elseif ($count === 1) {
        echo json_encode([
            'status' => 'single',
            'delivery_note' => $delivery_notes[0]
        ]);
    } else {
        echo json_encode([
            'status' => 'multiple',
            'delivery_notes' => $delivery_notes
        ]);
    }
}   

function deliveryNoteCancelRequest($sales_model){
    $dn_number = $_POST['dn_number'] ?? '';
    $state = 3; // 3 = Cancellation Requested
    
    if (empty($dn_number)) {
        echo json_encode([
            'success' => false,
            'message' => 'Invalid DN number.'
        ]);
        exit;
    }

    $updated = $sales_model->requestCancelDeliveryNote($dn_number, $state);

    if ($updated) {
        echo json_encode([
            'success' => true
        ]);
    } else {
        echo json_encode([
            'success' => false,
            'message' => 'Update failed.'
        ]);
    }

    exit;
}

function fetchScrapAmount($sales_model){
    $setting_value = $_POST['value'];
    $amount = $sales_model->getScrapDiscount($setting_value);

    if ($amount > 0) {
        echo json_encode([
            'status' => 'success',
            'amount' => $amount
        ]);
    } else {
        echo json_encode([
            'status' => 'info',
            'message' => 'No scrap balance available'
        ]);
    }

    exit;
}

function checkTransactionID($sales_model){
    $transaction_id = $_POST['transaction_id'];

    $exists = $sales_model->transactionIdExists($transaction_id);

    echo json_encode([
        "exists" => $exists
    ]);
    exit;
}


function fetchActiveInvoices($sales_model){
    $cid = intval($_POST['customer_id']);
    $invoices = $sales_model->getActiveInvoicesByCustomer($cid);
    echo json_encode($invoices);
}

function saveCashSale($stock_model, $sales_model, $common_model, $payments_model, $customers_model, $db){
    try {
        $db->beginTransaction();

        $apply_vat        = isset($_POST['apply_vat']) ? (int) $_POST['apply_vat'] : 0; 
        $cash_sale_number = $_POST['cash_sale_number'] ?? null;
        $check            = $_POST['customer_id'] ?? null;
        $items            = $_POST['items'] ?? [];
        $discount         = floatval($_POST['discount'] ?? 0);
        $sale_date        = $_POST['cash_sale_date'] ?? date('Y-m-d');
        $warehouse_id     = $_POST['warehouse_id'] ?? 1;
        $payments         = $_POST['payments'] ?? [];
        $debug            = intval($_POST['debug'] ?? 0);
        $walkin_name      = trim($_POST['walkin_name'] ?? '');
        $entered_number   = trim($_POST['walkin_phone'] ?? '');
 
        $scrap_collected  = isset($_POST['scrap_collected']) ? (int) $_POST['scrap_collected'] : 0;
        $scrap_quantity   = intval($_POST['scrap_quantity'] ?? 0);
        $scrap_amount     = $sales_model->getScrapDiscount('scrap_collection_fee');



        if (!is_array($payments) || count($payments) === 0) {
            echo json_encode(['status'=>'error','message'=>'At least one payment method is required']);
            exit;
        }


        if (!$check) {
            echo json_encode(['status'=>'error','message'=>'Missing customer details']);
            exit;
        }

        if (!$items || !is_array($items) || count($items) == 0) {
            echo json_encode(['status'=>'error','message'=>'Select at least one item for the cash sale']);
            exit;
        }
        if (!$cash_sale_number || trim($cash_sale_number) === '') {
            echo json_encode(['status'=>'error','message'=>'Invalid cash sale number']);
            exit;
        }

        if ($discount < 0) {
            echo json_encode(['status'=>'error','message'=>'Invalid discount amount']);
            exit;
        }

        if (!strtotime($sale_date)) {
            echo json_encode(['status'=>'error','message'=>'Invalid sale date']);
            exit;
        }

        if (!$warehouse_id) {
            echo json_encode(['status'=>'error','message'=>'Invalid warehouse selection']);
            exit;
        }

        if($check =='Walk-In'){
            $customer_id = $customers_model->getNextCustomerId(); 
        }else{
            $customer_id = $check;
        }

        $total_paid = 0;

        foreach ($payments as $p) {

            if (!isset($p['method'], $p['amount'])) {
                echo json_encode(['status'=>'error','message'=>'Invalid payment data']);
                exit;
            }

            if ($p['amount'] <= 0) {
                echo json_encode(['status'=>'error','message'=>'Invalid payment amount']);
                exit;
            }

            // Only non-cash requires transaction ID
            if ($p['method'] !== 'Cash') {

                if (empty($p['transaction_id'])) {
                    echo json_encode([
                        'status'=>'error',
                        'message'=>'Transaction ID required for '.$p['method']
                    ]);
                    exit;
                }

                // Uniqueness check
                if ($sales_model->transactionIdExists($p['transaction_id'])) {
                    echo json_encode([
                        'status'=>'error',
                        'message'=>'Transaction ID '.$p['transaction_id'].' has already been used'
                    ]);
                    exit;
                }
            }

            $total_paid += floatval($p['amount']);
        }


        

        if ($scrap_collected === 1 && $scrap_quantity < 1) {
            echo json_encode([
                'status'  => 'error',
                'message' => 'Invalid scrap quantity'
            ]);
            exit;
        }


        // ---------------------------------------------------
        // HANDLE WALK-IN CUSTOMER
        // ---------------------------------------------------
        if ($check === 'Walk-In') {

            if ($walkin_name === '' || $entered_number === '') {
                echo json_encode([
                    'status'  => 'error',
                    'message' => 'Walk-in customer name and phone are required.'
                ]);
                exit;
            }

            // Normalize phone (spaces only)
            $walkin_phone_filtered = preg_replace('/\s+/', '', $entered_number);
            $walkin_phone = '+265' . $walkin_phone_filtered;

            // Check existing customer by phone
            $existingCustomer = $customers_model->getCustomerByPhone($walkin_phone);

            if ($existingCustomer) {

                /**
                 * Phone exists → name MUST match
                 * Using LIKE comparison (case-insensitive)
                 */
                if (stripos($existingCustomer['customer_name'], $walkin_name) !== false
                    || stripos($walkin_name, $existingCustomer['customer_name']) !== false
                ) {

                    // ✅ Same person → reuse customer_id
                    $customer_id = $existingCustomer['customer_id'];

                } else {

                    // ❌ Phone exists but name differs
                    echo json_encode([
                        'status'  => 'error',
                        'message' => 'This phone number already exists for a different customer. 
                                    Please verify phone number.'
                    ]);
                    exit;
                }

            } else {

                // New walk-in customer
                $customerData = [
                    'customer_id'    => $customer_id,
                    'customer_name'  => $walkin_name,
                    'phone'          => $walkin_phone,
                    'registered_loc' => $warehouse_id,
                    'customer_type'  => 'Walk-In',
                    'customer_state' => 1,
                    'date_created'   => date('Y-m-d H:i:s'),
                    'operator'       => $_SESSION['sess_employeecode'] ?? 0
                ];

                $result = $customers_model->insertCustomer($customerData);

                if ($result['status'] === false) {
                    $db->rollBack();
                    echo json_encode([
                        'status'  => 'error',
                        'message' => $result['error'],
                        'db_error'=> $result['error']
                    ]);
                    exit;
                }

            }
        }



        
        // -------------------------------------------------------
        // CHECK IF CASH SALE NUMBER IS ALREADY USED
        // IF USED, GENERATE NEW ONE
        // -------------------------------------------------------

        if ($sales_model->cashSaleNumberExists($cash_sale_number)) {
            $cash_sale_number = getNextCashSaleNumber($warehouse_id, 'warehouse');
        }
        // VAT RATE
        $companyDetails = $common_model->getCompanyDetails();
        $VAT_PERCENT = floatval($companyDetails['vat_percentage']); // e.g. 0.165 VAT IS ALREADY IN DECIMAL FORMAT
        $VAT_RATE     = $VAT_PERCENT;

        $running_total = 0;
        $subtotal      = 0;
        $total_tax     = 0;

        $computed_items = [];
        // ---------------------------------------------------
        // LOOP ITEMS
        // ---------------------------------------------------
        foreach ($items as $item) {

            $product_code    = $item['product_code'];
            $price_inclusive = floatval($item['price']); // UI price
            $qty             = intval($item['qty']);

            if ($qty <= 0) {
                $db->rollBack();
                echo json_encode(['status'=>'error','message'=>'Invalid quantity']);
                exit;
            }

            if ($apply_vat === 1) {
                // VAT INCLUDED price → extract VAT
                $price_exclusive = round($price_inclusive / (1 + $VAT_RATE), 2);
                $line_subtotal   = round($price_exclusive * $qty, 2);
                $line_total      = round($price_inclusive * $qty, 2);
                $line_tax        = round($line_total - $line_subtotal, 2);
            } else {
                // VAT NOT APPLIED
                $price_exclusive = $price_inclusive;
                $line_subtotal   = round($price_exclusive * $qty, 2);
                $line_total      = $line_subtotal;
                $line_tax        = 0;
            }

            $opening_stock = $stock_model->getWarehouseQuantity($product_code, $warehouse_id);
            $closing_stock = $opening_stock - $qty;

            $running_total += $line_total;
            $subtotal      += $line_subtotal;
            $total_tax     += $line_tax;

            $computed_items[] = [
                'product_code'  => $product_code,
                'qty'           => $qty,
                'price'         => $price_exclusive, // ✅ STORE VAT-EXCLUSIVE PRICE
                'line_total'    => $line_total,
                'line_subtotal' => $line_subtotal,
                'line_tax'      => $line_tax,
                'opening_stock' => $opening_stock,
                'closing_stock' => $closing_stock,
                'warehouse_id'  => $warehouse_id
            ];
        }

        // FINAL TOTAL
        $total_bill = $running_total - $discount;

       if ($total_paid > $total_bill) {
            echo json_encode([
                'status'=>'error',
                'message'=>'Total payment exceeds sale amount'
            ]);
            exit;
        }

        $balance = round($total_bill - $total_paid, 2);

        /*-----------------------------------------------
        DEBUG MODE – DO NOT SAVE ANYTHING
        -------------------------------------------------*/

        // ---------------------------------------------------
        // PREPARE SCRAP PREVIEW (NO INSERT)
        // ---------------------------------------------------
        $scrap_preview = null;

        if ($scrap_collected === 1) {

            if (!$scrap_amount || $scrap_amount <= 0) {
                $db->rollBack();
                echo json_encode([
                    'status'  => 'error',
                    'message' => 'Invalid scrap configuration amount'
                ]);
                exit;
            }

            $scrap_preview = [
                'transaction_number'   => $cash_sale_number,
                'scrap_collected'      => $scrap_collected,
                'scrap_quantity'       => $scrap_quantity,
                'scrap_amount'         => floatval($scrap_amount),
                'total_scrap_discount' => round($scrap_amount * $scrap_quantity, 2)
            ];
        }


        if ($debug === 1) {

            $db->rollBack();

            echo json_encode([
                'status'           => 'preview',
                'message'          => 'Preview only – no data saved.',
                'customer_id'      => $customer_id,
                'cash_sale_number' => $cash_sale_number,
                'cash_sale_date'   => $sale_date,
                'computed_items'   => $computed_items,
                'subtotal'         => round($subtotal, 2),
                'total_tax'        => round($total_tax, 2),
                'discount'         => round($discount, 2),
                'total_bill'       => round($total_bill, 2),
                'payments'         => $payments,    
                'scrap'            => $scrap_preview   // 👈 ADD THIS
            ]);
            exit;
        }


           
        // ---------------------------------------------------
        // SAVE MODE — INSERT ITEMS FIRST
        // ---------------------------------------------------
        foreach ($computed_items as $ci) {

            $row = [
                'sales_number'  => $cash_sale_number,
                'product_code'  => $ci['product_code'],
                'type'          => 'Cash Sale',
                'department'    => $warehouse_id,
                'opening_stock' => $ci['opening_stock'],
                'quantity'      => $ci['qty'],
                'closing_stock' => $ci['closing_stock'],
                'selling_price' => $ci['price'],
                'subtotal'      => round($ci['line_subtotal'], 2),
                'vat_charge'    => $apply_vat ? $VAT_PERCENT : 0,
                'tax'           => round($ci['line_tax'], 2),
                'total'         => round($ci['line_total'], 2),
                'date_captured' => date('Y-m-d H:i:s'),
                'operator'      => $_SESSION['sess_employeecode'] ?? 0,
                'status'        => 1
            ];


            // Insert item
            if (!$sales_model->insertSalesOrder($row)) {
                $db->rollBack();
                echo json_encode(['status' => 'error', 'message' => 'Failed to save item: '.$ci['product_code']]);
                exit;
            }

            // Update warehouse stock
            $sql = "UPDATE 
                            warehouse_products 
                    SET 
                        quantity = :closing_stock 
                    WHERE 
                        product_id = :product_code 
                        AND warehouse_id = :warehouse_id";
            $stmt = $db->prepare($sql);
            $updated = $stmt->execute([
                ':closing_stock' => $ci['closing_stock'],
                ':product_code'  => $ci['product_code'],
                ':warehouse_id'  => $warehouse_id
            ]);
            
            if (!$updated) {
                $db->rollBack();
                echo json_encode(['status' => 'error', 'message' => 'Failed to update stock for: '.$ci['product_code']]);
                exit;
            }

        }

        // -------------------------------------
        // Generate receipt number (same as invoice payments)
        // -------------------------------------
         
        

        // ---------------------------------------------------
        // INSERT INTO payments
        foreach ($payments as $p) {

            

            $paymentData = [
                'tx_id'          => uniqid("TX"),
                'sale_id'        => $cash_sale_number,
                'customer'       => $customer_id,
                'location'       => $warehouse_id,
                'receipt_number' => $sales_model->getNextDocumentNumber('receipt', $warehouse_id, 'warehouse'),
                'payment_method' => $p['method'],
                'payment_type'   => 'Cash Sale',
                'transaction_id' => $p['transaction_id'] ?? null,
                'payment_date'   => $sale_date,
                'discount'       => $discount,
                'amount_paid'    => round($p['amount'], 2),
                'balance'        => $balance, // same balance on all rows
                'datecaptured'   => date('Y-m-d H:i:s'),
                'operator'       => $_SESSION['sess_employeecode'] ?? 0,
                'payment_state' => 1
            ];

            if (!$sales_model->insertCashSalePayment($paymentData)) {
                $db->rollBack();
                echo json_encode(['status'=>'error','message'=>'Failed to save payment']);
                exit;
            }
        }


            // ---------------------------------------------------
            // SAVE SCRAP COLLECTION (IF APPLICABLE)
            // ---------------------------------------------------
            if ($scrap_collected === 1) {

                if (!$scrap_amount || $scrap_amount <= 0) {
                    $db->rollBack();
                    echo json_encode([
                        'status'  => 'error',
                        'message' => 'Invalid scrap configuration amount'
                    ]);
                    exit;
                }

                $scrapData = [
                    'transaction_number' => $cash_sale_number,
                    'customer_id'      => $customer_id,
                    'quantity'         => $scrap_quantity,
                    'amount'           => floatval($scrap_amount),
                    'total_scrap_disc' => floatval($scrap_amount * $scrap_quantity),
                    'location'         => $warehouse_id,
                    'date_collected'   => $sale_date,
                    'date_captured'    => date('Y-m-d H:i:s'),
                    'operator'         => $_SESSION['sess_employeecode'] ?? 0
                ];

                if (!$sales_model->insertScrapCollection($scrapData)) {
                    $db->rollBack();
                    echo json_encode([
                        'status'  => 'error',
                        'message' => 'Failed to save scrap collection'
                    ]);
                    exit;
                }
            }
    
        $db->commit();

        echo json_encode([
            'status'            => 'success',
            'cash_sale_id'      => $cash_sale_number
        ]);

    } catch (Exception $e) {

        $db->rollBack();
        echo json_encode(['status'=>'error','message'=>$e->getMessage()]);
    }
}








?>
