<?php

include_once 'config.php';

class documents_model
{
    public $link;

    public function __construct()
    {
        $db_connection = new dbConnection();
        $this->link = $db_connection->connect();
    }

    public function getDocumentData($type, $reference) {

        switch ($type) {
            case 'cash_sale':
                return $this->getCashSale($reference);

            case 'invoice':
                return $this->getInvoice($reference);

            case 'quotation':
                return $this->getQuotation($reference);

            case 'customer_delivery_note':
                return $this->getCustomerDeliveryNote($reference);

            default:
                throw new Exception('Unsupported document type');
        }
    }

    /* =======================
       CASH SALE
    ======================== */
    private function getCashSale($reference) {

        $sql = " SELECT 
                    p.sale_id            AS document_number,
                    p.payment_date       AS document_date,
                    p.amount_paid        AS amount,
                    c.customer_name,
                    c.contact_number     AS phone,
                    c.email
                FROM 
                    payments p
                LEFT JOIN 
                    customers c ON c.customer_id = p.customer
                WHERE 
                    p.sale_id = ?
                    AND p.payment_type = 'Cash Sale'
                LIMIT 1";

        $stmt = $this->link->prepare($sql);
        $stmt->execute([$reference]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$row) {
            throw new Exception('Cash sale not found');
        }

        return $this->normalize($row, 'cash_sale');
    }

    /* =======================
       INVOICE
    ======================== */
    private function getInvoice($reference) {

        $sql = "SELECT 
                    so.order_number      AS document_number,
                    so.orderdate         AS document_date,
                    so.total_bill        AS amount,
                    c.customer_name,
                    c.contact_number     AS phone,
                    c.email
                FROM 
                    saved_orders so
                LEFT JOIN 
                    customers c ON c.customer_id = so.customer
                WHERE 
                    so.order_number = ?
                LIMIT 1
        ";

        $stmt = $this->link->prepare($sql);
        $stmt->execute([$reference]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$row) {
            throw new Exception('Invoice not found');
        }

        return $this->normalize($row, 'invoice');
    }

    /* =======================
       QUOTATION
    ======================== */
    private function getQuotation($reference) {

        $sql = "SELECT 
                    q.quote_number       AS document_number,
                    q.quote_date         AS document_date,
                    q.total_amount       AS amount,
                    c.customer_name,
                    c.contact_number     AS phone,
                    c.email
                FROM 
                    quotations q
                LEFT JOIN 
                    customers c ON c.customer_id = q.customer
                WHERE 
                    q.quote_number = ?
                LIMIT 1";

        $stmt = $this->link->prepare($sql);
        $stmt->execute([$reference]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$row) {
            throw new Exception('Quotation not found');
        }

        return $this->normalize($row, 'quotation');
    }

    /* =======================
       CUSTOMER DELIVERY NOTE
    ======================== */
    private function getCustomerDeliveryNote($reference) {

        $sql = "SELECT 
                    dn.dn_number AS document_number,
                    dn.dn_date        AS document_date,
                    0         AS amount,
                    c.customer_name,
                    c.contact_number        AS phone,
                    c.email
                FROM 
                    delivery_notes dn
                LEFT JOIN 
                    customers c ON c.customer_id = dn.customer
                WHERE 
                    dn.dn_number = ?
                LIMIT 1";

        $stmt = $this->link->prepare($sql);
        $stmt->execute([$reference]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$row) {
            throw new Exception('Customer delivery note not found');
        }

        return $this->normalize($row, 'customer_delivery_note');
    }

    /* =======================
       NORMALIZER (CRITICAL)
    ======================== */
    private function normalize($row, $type) {

        return [
            'document_type'   => $type,
            'document_number' => $row['document_number'],
            'document_date'   => $row['document_date'],
            'amount'          => (float) $row['amount'],

            'customer' => [
                'name'  => $row['customer_name'] ?? 'Walk-in Customer',
                'phone' => $row['phone'] ?? null,
                'email' => $row['email'] ?? null
            ],

            'links' => [
                'view' => $this->getViewUrl($type, $row['document_number'])
            ]
        ];
    }

    private function getViewUrl($type, $ref) {

        switch ($type) {
            case 'cash_sale':
                return "cash_sale_receipt.php?id={$ref}";

            case 'invoice':
                return "customer_invoice.php?invoice_number={$ref}";

            case 'quotation':
                return "quotation_view.php?quote_number={$ref}";

            default:
                return '#';
        }
    }
}