<?php

include_once 'config.php';

class transfer_model
{
    public $link;

    public function __construct()
    {
        $db_connection = new dbConnection();
        $this->link = $db_connection->connect();
    }

    // Get last TN number from database
    public function getLastTNNumber() {
        $stmt = $this->link->query("SELECT transfer_id FROM stock_transfers_final ORDER BY id DESC LIMIT 1");
        return $stmt->fetchColumn();
    }

    /**
     * Check if a document number already exists either in the reserved numbers table
     * or in the finalized stock transfer table.
     *
     * @param string $tn_number
     * @return bool  true if number exists, false otherwise
     */
    public function documentNumberExists(string $tn_number): bool{
        // 1) Check in document_numbers table
        $sql = "SELECT 
                COUNT(*) 
            FROM 
                document_numbers 
            WHERE 
                number = :tn";
        $stmt = $this->link->prepare($sql);
        $stmt->execute([':tn' => $tn_number]);
        $existsReserved = $stmt->fetchColumn() > 0;

        if ($existsReserved) {
            return true;
        }

        

        // 2) Check in stock_transfers_final table
        $sql = "SELECT 
                    COUNT(*) 
                FROM 
                    stock_transfers_final 
                WHERE 
                    transfer_id = :tn";
        $stmt = $this->link->prepare($sql);
        $stmt->execute([':tn' => $tn_number]);
        $existsFinalized = $stmt->fetchColumn() > 0;

        return $existsFinalized;
    }



    public function checkdocumentNumber($tn_number){
        $sql = "SELECT COUNT(*) document_numbers WHERE number = :tn_number";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':tn_number', $tn_number);
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row && intval($row['document_numbers']) > 0;
    }

    public function savedocumentNumber($tn_number, $purpose, $operator, $date){
        try {
            $sql = "INSERT INTO document_numbers (number, purpose, operator, date_captured) 
                    VALUES (:number, :purpose, :operator, :datecaptured)";

            $stmt = $this->link->prepare($sql);
            $stmt->bindParam(':number', $tn_number);
            $stmt->bindParam(':purpose', $purpose);
            $stmt->bindParam(':operator', $operator);
            $stmt->bindParam(':datecaptured', $date);

            return $stmt->execute();
        } catch (PDOException $e) {
            // Optional: log or handle error
            error_log("Save Document Number Error: " . $e->getMessage());
            return false;
        }
    }

    public function getPendingQuantityForProduct($warehouse, $product) {
        $sql = "SELECT SUM(qty_initiated) AS pending_qty FROM stock_transfer_items WHERE transfer_from = :warehouse AND product_id = :product AND transfer_state = 1";

        $stmt = $this->link->prepare($sql);
        $stmt->execute([':warehouse' => $warehouse, ':product' => $product]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        return floatval($result['pending_qty'] ?? 0);
    }

    public function isProductAlreadyInTransfer($transfer_id, $product_id) {
        $sql = "SELECT COUNT(*) as count FROM stock_transfer_items 
                WHERE transfer_id = :transfer_id AND product_id = :product_id";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':transfer_id', $transfer_id);
        $stmt->bindParam(':product_id', $product_id);
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row && intval($row['count']) > 0;
    }

    public function transferStock(array $data) {
        try {
            $sql = "INSERT INTO stock_transfer_items (transfer_id, product_id, transfer_from, transfer_to, sender_opening_stock, receiver_opening_stock, qty_initiated, original_qty_initiated, sender_closing_stock, receiver_closing_stock, operator, transfer_date, transfer_state, datecaptured) VALUES (:transfer_id, :product_id, :transfer_from, :transfer_to, :sender_opening_stock, :receiver_opening_stock, :qty_initiated, :original_qty_initiated, :sender_closing_stock, :receiver_closing_stock, :operator, :transfer_date, :transfer_state, :datecaptured)";

            $stmt = $this->link->prepare($sql);

            $stmt->bindParam(':transfer_id',             $data['transfer_id']);
            $stmt->bindParam(':product_id',              $data['product_id'], PDO::PARAM_INT);
            $stmt->bindParam(':transfer_from',           $data['transfer_from']);
            $stmt->bindParam(':transfer_to',             $data['transfer_to'], PDO::PARAM_INT);
            $stmt->bindParam(':sender_opening_stock',    $data['sender_opening_stock']);
            $stmt->bindParam(':receiver_opening_stock',  $data['receiver_opening_stock']);
            $stmt->bindParam(':qty_initiated',           $data['qty_initiated']);
            $stmt->bindParam(':original_qty_initiated',  $data['original_qty_initiated']);
            $stmt->bindParam(':sender_closing_stock',    $data['sender_closing_stock']);
            $stmt->bindParam(':receiver_closing_stock',  $data['receiver_closing_stock']);
            $stmt->bindParam(':operator',                $data['operator'], PDO::PARAM_INT);
            $stmt->bindParam(':transfer_date',           $data['transfer_date']);
            $stmt->bindParam(':transfer_state',          $data['transfer_state'], PDO::PARAM_INT);
            $stmt->bindParam(':datecaptured',            $data['datecaptured']);

            return $stmt->execute();
        } catch (PDOException $e) {
            // Optional: log or handle error
            error_log("TransferStock Error: " . $e->getMessage());
            return false;
        }

    }

    public function buildItemsTableHTML($tn_number) {
        $items = $this->getTransferItems($tn_number);

        if (!$items || count($items) == 0) {
            return '<tr><td colspan="6" class="text-center text-muted">No items yet. Add items to begin.</td></tr>';
        }

        $html = '';

        foreach ($items as $row) {
            $id = $row['id'];
            $product_id = $row['product_id'];
            $warehouse_id = $row['transfer_from'];
            $qty = $row['qty_initiated'];

            $html .= '
                <tr>

                    <td class="text-center">
                        <button class="btn btn-sm btn-danger deleteItemBtn" 
                                data-id="'.$id.'" 
                                data-action="delete-item"
                                data-product_id="'.$product_id.'"
                                data-warehouse_id="'.$warehouse_id.'"
                                data-qty="'.$qty.'"
                                data-item-id="'.$id.'">
                            <i class="fas fa-trash"></i>
                        </button>
                    </td>


                    <!-- DATE: clickable link opens modal -->
                    <td>
                        <a href="#" class="editDateLink" data-id="' . $id . '" data-toggle="modal" data-target="#editDateModal_' . $id . '">
                            ' . htmlspecialchars($row['transfer_date']) . '
                        </a>
                    </td>

                    <!-- PRODUCT NAME -->
                    <td>' . htmlspecialchars($row['product_name']) . '</td>

                    <!-- FROM LOCATION -->
                    <td>' . htmlspecialchars($row['location_from']) . '</td>

                    <!-- DESTINATION -->
                    <td>' . htmlspecialchars($row['location_to']) . '</td>

                    <!-- QTY: clickable -->
                    <td style="text-align: center;">
                        <a href="#" class="editQtyLink" data-id="' . $id . '" data-toggle="modal" data-target="#editQtyModal_' . $id . '">
                            ' . number_format($row['qty_initiated'], 2) . '
                        </a>
                    </td>

                </tr>
            ';
        }

        return $html;
    }

    public function buildItemModalsHTML($tn_number){
        $items = $this->getTransferItems($tn_number);
        $html = '';

        foreach ($items as $row) {
            $id = $row['id'];
            $product = $row['product_id'];
            $old_quantity = $row['qty_initiated'];
            $warehouse = $row['transfer_from'];

            // Edit Quantity Modal
            $html .= '
            <div class="modal fade" id="editQtyModal_' . $id . '" tabindex="-1">
                <div class="modal-dialog">
                    <div class="modal-content">

                        <div class="modal-header">
                            <h5 class="modal-title">Edit Quantity</h5>
                            <button type="button" class="close" data-dismiss="modal">&times;</button>
                        </div>

                        <div class="modal-body">
                            <input type="number" min="1" class="form-control editQtyInput"
                                data-id="' . $id . '" value="' . $row['qty_initiated'] . '">
                            <input type="hidden" data-warehouse="'. $row['transfer_from'].'" value="'.  $row['transfer_from'] .'" readonly>
                            <input type="hidden" data-product="'. $row['product_id'].'" value="'.  $row['product_id'] .'" readonly>
                            <input type="hidden" data-old_qty="'. $row['qty_initiated'].'" value="'.  $row['qty_initiated'] .'" readonly>
                        </div>

                        <div class="modal-footer">
                            <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                            <button type="button" class="btn btn-primary saveQtyBtn" data-product="'.$product.'" data-old_qty="'.$old_quantity.'" data-warehouse="'.$warehouse.'" data-id="' . $id . '">Save</button>
                        </div>

                    </div>
                </div>
            </div>';


            // Edit Date Modal
            $html .= '
            <div class="modal fade" id="editDateModal_' . $id . '" tabindex="-1">
                <div class="modal-dialog">
                    <div class="modal-content">

                        <div class="modal-header">
                            <h5 class="modal-title">Edit Transfer Date</h5>
                            <button type="button" class="close" data-dismiss="modal">&times;</button>
                        </div>

                        <div class="modal-body">
                            <input type="date" max="' . date('Y-m-d') . '" class="form-control editDateInput"
                                data-id="' . $id . '" value="' . $row['transfer_date'] . '">
                        </div>

                        <div class="modal-footer">
                            <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                            <button type="button" class="btn btn-primary saveDateBtn" data-product="'.$product.'" data-id="' . $id . '">Save</button>
                        </div>

                    </div>
                </div>
            </div>';

        }

        return $html;
    }


    public function getTransferItems($tn_number) {
        try {
            $sql = "SELECT 
                        sti.*, 
                        p.product_name, 
                        w1.warehouse_name AS location_from,
                        COALESCE(w2.warehouse_name,b.branch_name) AS location_to
                    FROM 
                        stock_transfer_items sti
                    LEFT JOIN 
                        products p ON sti.product_id = p.product_id
                    LEFT JOIN 
                        warehouses w1 ON sti.transfer_from = w1.warehouse_id
                    LEFT JOIN 
                        warehouses w2 ON sti.transfer_to = w2.warehouse_id
                    LEFT JOIN 
                        branches b ON sti.transfer_to = b.branch_id
                    WHERE 
                        sti.transfer_id = :tn_number";

            $stmt = $this->link->prepare($sql);
            $stmt->bindParam(':tn_number', $tn_number);
            $stmt->execute();
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            error_log("Get Transfer Items Error: " . $e->getMessage());
            return [];
        }
    }

    public function countTransferItems($tn_number) {
        try {
            $sql = "SELECT COUNT(*) as cnt FROM stock_transfer_items WHERE transfer_id = :tn_number";
            $stmt = $this->link->prepare($sql);
            $stmt->bindParam(':tn_number', $tn_number);
            $stmt->execute();
            $row = $stmt->fetch(PDO::FETCH_ASSOC);
            return intval($row['cnt'] ?? 0);
        } catch (PDOException $e) {
            error_log("Count Transfer Items Error: " . $e->getMessage());
            return 0;
        }
    }

    public function getTransferLocations($tn_number) {
        try {
            $sql = "SELECT 
                        sti.transfer_from,
                        w.warehouse_name as from_name,
                        w2.warehouse_id as to_wh_id, 
                        w2.warehouse_name as to_wh_name,
                        b.branch_id as to_branch_id, 
                        b.branch_name as to_branch_name
                    FROM 
                        stock_transfer_items sti
                    LEFT JOIN 
                        warehouses w ON sti.transfer_from = w.warehouse_id
                    LEFT JOIN 
                        warehouses w2 ON sti.transfer_to = w2.warehouse_id
                    LEFT JOIN 
                        branches b ON sti.transfer_to = b.branch_id
                    WHERE 
                        sti.transfer_id = :tn_number
                    LIMIT 1"; // take first item
            $stmt = $this->link->prepare($sql);
            $stmt->bindParam(':tn_number', $tn_number);
            $stmt->execute();
            $row = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$row) return [];

            // Determine destination type
            if ($row['to_wh_id']) {
                $destType = 'warehouse';
                $destId = $row['to_wh_id'];
                $destName = $row['to_wh_name'];
            } else {
                $destType = 'branch';
                $destId = $row['to_branch_id'];
                $destName = $row['to_branch_name'];
            }

            return [
                'from_warehouse_id' => $row['transfer_from'],
                'from_warehouse_name' => $row['from_name'],
                'destination_type' => $destType,
                'destination_id' => $destId,
                'destination_name' => $destName
            ];
        } catch (PDOException $e) {
            error_log("getTransferLocations Error: " . $e->getMessage());
            return [];
        }
    }

    public function updateTransferDate($tn_number, $new_date) {
        try {
            $sql = "UPDATE stock_transfer_items SET transfer_date = :new_date WHERE transfer_id = :tn_number";
            $stmt = $this->link->prepare($sql);
            $stmt->bindParam(':new_date', $new_date);
            $stmt->bindParam(':tn_number', $tn_number);
            return $stmt->execute();
        } catch (PDOException $e) {
            error_log("Update Transfer Date Error: " . $e->getMessage());
            return false;
        }
    }

    public function updateItemQuantity($item_id, $new_quantity) {
        try {
            $stmt = $this->link->prepare("UPDATE stock_transfer_items SET qty_initiated = :qty, original_qty_initiated =:qty WHERE id = :id");
            $stmt->bindParam(':qty', $new_quantity, PDO::PARAM_INT);
            $stmt->bindParam(':id', $item_id, PDO::PARAM_INT);
            return $stmt->execute();
        } catch (PDOException $e) {
            error_log("Update Quantity Error: " . $e->getMessage());
            return false;
        }
    }

    public function deleteTransferItem($item_id) {
        try {
            $sql = "DELETE FROM stock_transfer_items WHERE id = :id";
            $stmt = $this->link->prepare($sql);
            $stmt->bindParam(':id', $item_id, PDO::PARAM_INT);
            return $stmt->execute();
        } catch (PDOException $e) {
            error_log("Delete Transfer Item Error: " . $e->getMessage());
            return false;
        }
    }


    public function deleteDocumentNumber($documentNumber){
        $sql = "DELETE FROM document_numbers WHERE number = :doc";
    

        $stmt = $this->link->prepare($sql);
        $params = [':doc' => $documentNumber];
        return $stmt->execute($params);
    }

    public function finalizeStockTransfer($data) {
        $sql = "INSERT INTO stock_transfers_final (transfer_id, transfer_from, transfer_to, transfer_date, datecaptured, operator, transfer_state, driver) 
                VALUES ( :transfer_id, :transfer_from, :transfer_to, :transfer_date, :datecaptured, :operator, :transfer_state, :driver)";

        $stmt = $this->link->prepare($sql);
        return $stmt->execute([
            ':transfer_id'    => $data['transfer_id'],
            ':transfer_from'  => $data['transfer_from'],
            ':transfer_to'    => $data['transfer_to'],
            ':transfer_date'  => $data['transfer_date'],
            ':datecaptured'   => $data['datecaptured'],
            ':operator'       => $data['operator'],
            ':transfer_state' => $data['transfer_state'],
            ':driver'         => $data['driver']
        ]);
    }

    public function getFirstTransferItemDetails($transfer_id) {
        $sql = "SELECT 
                transfer_from, 
                transfer_to, 
                transfer_date,
                sti.transfer_id,
                COALESCE(wh.warehouse_name,br.branch_name) as destination_name

            FROM 
                stock_transfer_items sti
            LEFT JOIN
                warehouses wh ON sti.transfer_to = wh.warehouse_id
            LEFT JOIN
                branches br ON sti.transfer_to = br.branch_id
            WHERE 
                sti.transfer_id = :transfer_id
            ORDER BY 
                sti.id ASC 
            LIMIT 1";
        $stmt = $this->link->prepare($sql);
        $stmt->execute([':transfer_id' => $transfer_id]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    public function orderNumberExists($orderNumber) {
        $query = "SELECT transfer_id FROM stock_transfers_final WHERE transfer_id = :order_number";

        $stmt = $this->link->prepare($query);
        $stmt->execute(['order_number' => $orderNumber]);

        return $stmt->rowCount() > 0;
    }
           
}