<?php

include_once 'config.php';

class sales_model
{
    public $link;

    public function __construct()
    {
        $db_connection = new dbConnection();
        $this->link = $db_connection->connect();
    }

    private function getGroupKeyAndLabel(DateTime $dt, string $mode): array {
        switch ($mode) {

            case 'hour':
                // Key by date and hour
                return [
                    $dt->format('Y-m-d H'),
                    $dt->format('H:00') // display as "14:00", "15:00"
                ];

            case 'week':
                // Key by year-week
                $weekNum = (int)$dt->format('W'); // ISO week number
                return [
                    $dt->format('o-W'),        // key: 2026-01
                    'Week ' . $weekNum          // display as "Week 1", "Week 2"
                ];

            case 'month':
                // Key by year-month
                return [
                    $dt->format('Y-m'),       // key: 2026-01
                    $dt->format('M Y')        // display as "Jan 2026", "Feb 2026"
                ];

            default: // day
                // Key by date
                return [
                    $dt->format('Y-m-d'),     // key: 2026-01-12
                    $dt->format('d M')        // display as "12 Jan", "13 Jan"
                ];
        }
    }


    

    public function buildSalesPaymentsChart(array $sales, array $payments, $start_date, $end_date){
        $daysDiff = (strtotime($end_date) - strtotime($start_date)) / 86400 + 1;

        if ($daysDiff <= 1) {
            $mode = 'hour';
        } elseif ($daysDiff <= 31) {
            $mode = 'day';
        } elseif ($daysDiff <= 60) {
            $mode = 'week';
        } else {
            $mode = 'month';
        }

        $map = [];

        foreach ($sales as $row) {
            $dt = new DateTime($row['date']);
            [$key, $label] = $this->getGroupKeyAndLabel($dt, $mode);

            $map[$key]['label'] = $label;
            $map[$key]['sales'] = ($map[$key]['sales'] ?? 0) + (float)$row['amount'];
        }

        foreach ($payments as $row) {
            $dt = new DateTime($row['date']);
            [$key, $label] = $this->getGroupKeyAndLabel($dt, $mode);

            $map[$key]['label'] = $label;
            $map[$key]['payments'] = ($map[$key]['payments'] ?? 0) + (float)$row['amount'];
        }

        ksort($map);

        $chart = [];
        foreach ($map as $row) {
            $chart[] = [
                'label'    => $row['label'],
                'sales'    => $row['sales'] ?? 0,
                'payments' => $row['payments'] ?? 0
            ];
        }

        return ['data' => $chart];
    }







    public function getSales($location, $start_date = null, $end_date = null){

        if($start_date == null && $end_date == null){
            $start_date = date('Y-m-01');
            $end_date = date('Y-m-t');
        }
        $sql = "SELECT
                    so.id,
                    TIMESTAMP(cash_totals.payment_date, TIME(cash_totals.datecaptured)) AS date,
                    cash_totals.sale_id AS reference,
                    'Cash Sale' AS sale_type,
                    c.customer_name AS customer,
                    NULL AS payment_method,
                    b.branch_name AS location,
                    so.total AS amount,
                    cash_totals.total_paid AS paid,
                    cash_totals.total_discount AS discount,
                    so.total - (cash_totals.total_paid + cash_totals.total_discount) AS balance,
                    CASE 
                        WHEN so.total - (cash_totals.total_paid + cash_totals.total_discount) <= 0 THEN 'Paid'
                        WHEN cash_totals.total_paid > 0 THEN 'Partial'
                        ELSE 'Unpaid'
                    END AS status,
                    cash_totals.sale_id AS action_id,
                    'cash' AS source
                FROM
                    (SELECT 
                        sale_id,
                        MAX(payment_date) AS payment_date,
                        SUM(amount_paid) AS total_paid,
                        MAX(discount) AS total_discount,
                        MAX(datecaptured) AS datecaptured,
                        MAX(location) AS location
                    FROM payments
                    WHERE 
                        payment_type = 'Cash Sale'
                        AND payment_state = 1
                        AND location = :location
                    GROUP BY sale_id
                    ) cash_totals
                LEFT JOIN (
                    SELECT
                        sales_number,
                        MAX(id) AS id,
                        SUM(total) AS total
                    FROM sales_order
                    GROUP BY sales_number
                ) so ON so.sales_number = cash_totals.sale_id

                LEFT JOIN branches b 
                    ON cash_totals.location = b.branch_id
                LEFT JOIN customers c 
                    ON c.customer_id = (
                        SELECT customer 
                        FROM payments p2 
                        WHERE p2.sale_id = cash_totals.sale_id 
                        LIMIT 1
                    )
                WHERE
                    payment_date >= :start_date
                    AND payment_date <= :end_date

            UNION ALL

                /* ======= INVOICE SALES ======= */
                SELECT
                    so.id, 
                    TIMESTAMP(so.orderdate, TIME(so.datecaptured)) AS date,
                    so.order_number AS reference,
                    'Customer Invoice' AS sale_type,
                    c.customer_name AS customer,
                    NULL AS payment_method,
                    b.branch_name AS location,
                    so.total_bill AS amount,
                    IFNULL(ip.total_paid, 0) AS paid,
                    so.discount,
                    so.total_bill - IFNULL(ip.total_paid, 0) AS balance,
                    CASE
                        WHEN so.order_state = 2 THEN 'Paid'
                        WHEN so.order_state = 1 THEN 'Partial'
                        ELSE 'Unpaid'
                    END AS status,
                    so.id AS action_id,
                    'invoice' AS source
                FROM saved_orders so
                LEFT JOIN customers c 
                    ON so.customer = c.customer_id
                LEFT JOIN branches b 
                    ON so.invoice_location = b.branch_id
                LEFT JOIN (
                    SELECT 
                        invoice_number,
                        SUM(amount_paid) AS total_paid
                    FROM invoice_payments
                    GROUP BY invoice_number
                ) ip ON ip.invoice_number = so.order_number
                WHERE 
                    so.invoice_location = :location
                    AND so.order_state IN (0,1,2)
                    AND so.orderdate >= :start_date
                    AND so.orderdate <= :end_date

            ORDER BY date DESC";

        $stmt = $this->link->prepare($sql);
        $stmt->execute(['location' => $location, 'start_date' => $start_date, 'end_date' => $end_date]);

        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }





    public function getReceiptData($id) {
        $sql = "SELECT p.*, 
                c.customer_name,
                p.payment_state,
                p.sale_id
            FROM 
                payments p 
            LEFT JOIN
                customers c ON p.customer = c.customer_id
            WHERE p.tx_id = :id";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':id', $id, PDO::PARAM_INT);
    
        if ($stmt->execute()) {
            return $stmt->fetch(PDO::FETCH_ASSOC); // return single row as associative array
        } else {
            return false;
        }
    }

    public function getReceiptItems($sales_number) {
        $sql = "SELECT *,
                p.product_name as description,
                so.selling_price,
                so.total,
                so.subtotal
            FROM 
                sales_order so
            LEFT JOIN
                products p ON p.product_id = so.product_code
            WHERE 
                sales_number = :sales_number";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':sales_number', $sales_number, PDO::PARAM_STR);
    
        if ($stmt->execute()) {
            return $stmt->fetchAll(PDO::FETCH_ASSOC); // return all matching rows
        } else {
            return false;
        }
    }

    public function getCustomerDetails($customer_id) {
        $query = "SELECT 
                    cc.customer_id,
                    cc.customer_name,
                    cc.customer_address,
                    COALESCE(emp.fullname,'Super User') as operator,
                    cc.customer_state,
                    cc.email,
                    cc.contact_number as phone_number,
                    cc.gender,
                    ll.location_name,
                    cc.datecreated,
                    c.country_name,
                    c.nationality,
                    cc.nationality as current_nationality,
                    cc.location,
                    cc.whatsapp_number,
                    cc.type_customer,
                    cc.nationality as nationality_id,
                    c.nationality as nationality_name,
                    CASE
                        WHEN cc.customer_state = 1 THEN 'Active Customer'
                        WHEN cc.customer_state = 0  THEN 'Suspended Customer'
                    END as customer_status
                FROM 
                    customers cc
                LEFT JOIN
                    employees emp ON cc.operator = emp.employee_code
                LEFT JOIN
                    locations ll ON cc.location = ll.loc_id 
                LEFT JOIN
                    countries c ON cc.nationality = c.id
                WHERE 
                    cc.customer_id = :customer_id
                    ";

        $stmt = $this->link->prepare($query);
        $stmt->bindParam(':customer_id', $customer_id, PDO::PARAM_STR);
        $stmt->execute();
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    public function getNextDocumentNumber($type, $location_id, $location_type){
        $year = date('Y');

        // Document type mapping
        $map = [
            'cash_sale'     => ['table' => 'payments', 'column' => 'sale_id', 'location_column' => 'location', 'prefix' => 'C'],
            'invoice'       => ['table' => 'saved_orders', 'column' => 'order_number', 'location_column' => 'invoice_location', 'prefix' => 'I'],
            'receipt'       => ['table' => 'payments', 'column' => 'receipt_number', 'location_column' => 'location', 'prefix' => 'R'],
            'transfer_note' => ['table' => 'stock_transfers_final', 'column' => 'transfer_id', 'location_column' => 'transfer_from', 'prefix' => 'TN'],
            'quotation'     => ['table' => 'quotations', 'column' => 'quote_number', 'location_column' => 'quote_from', 'prefix' => 'Q'],
            'delivery_note' => ['table' => 'delivery_notes', 'column' => 'dn_number', 'location_column' => 'shop_location', 'prefix' => 'DN']
        ];

        if(!isset($map[$type])){
            throw new Exception("Invalid document type: $type");
        }

        $table = $map[$type]['table'];
        $column = $map[$type]['column'];
        $location_column = $map[$type]['location_column'];
        $prefix = $map[$type]['prefix'];

        // Fetch location code
        if($location_type === 'branch'){
            $stmt = $this->link->prepare("SELECT branch_code FROM branches WHERE branch_id = ? LIMIT 1");
        } else if($location_type === 'warehouse'){
            $stmt = $this->link->prepare("SELECT warehouse_code FROM warehouses WHERE warehouse_id = ? LIMIT 1");
        } else {
            throw new Exception("Invalid location type: $location_type");
        }

        $stmt->execute([$location_id]);
        $location_code = $stmt->fetchColumn();

        if(!$location_code){
            throw new Exception("Code not found for location ID: $location_id");
        }

        // Base pattern: PREFIX + YEAR + LOCATION_CODE
        $basePattern = $prefix . $year . $location_code;

        // Fetch last sequence for this type and location
        $sql = "SELECT MAX(CAST(SUBSTRING($column, LENGTH(?) + 1) AS UNSIGNED)) AS last_seq 
                FROM $table 
                WHERE $column LIKE ? 
                AND $location_column = ?";
        $stmt = $this->link->prepare($sql);
        $stmt->execute([$basePattern, $basePattern . '%', $location_id]);

        $lastSeq = (int)($stmt->fetchColumn() ?? 0);
        $nextSeq = str_pad($lastSeq + 1, 2, '0', STR_PAD_LEFT);

        return $basePattern . $nextSeq;
    }

    public function deliveryNoteExists($dn_number){
        $sql = "SELECT 1 FROM delivery_notes WHERE dn_number = ? LIMIT 1";
        $stmt = $this->link->prepare($sql);
        $stmt->execute([$dn_number]);

        return $stmt->fetchColumn() ? true : false;
    }

    public function createDeliveryNote($dn_number, $reference_no, $customer, $location, $dn_date, $datecaptured, $operator,  $delivered_by, $dn_state, $invoice_no = null) {
        $sql = "INSERT INTO delivery_notes 
                (dn_number, reference, customer, invoice_number, shop_location, dn_date, datecaptured, operator, delivered_by, dn_state)
                VALUES
                (:dn_number, :reference_no, :customer, :invoice_number, :shop_location, :delivery_date, :datecaptured, :operator, :delivered_by, :dn_state)";
        
        $stmt = $this->link->prepare($sql);
        
        $stmt->bindParam(':dn_number', $dn_number);
        $stmt->bindParam(':reference_no', $reference_no);
        $stmt->bindParam(':customer', $customer);
        $stmt->bindParam(':shop_location', $location);
        $stmt->bindParam(':delivery_date', $dn_date);
        $stmt->bindParam(':datecaptured', $datecaptured);
        $stmt->bindParam(':operator', $operator);
        $stmt->bindParam(':delivered_by', $delivered_by);
        $stmt->bindParam(':dn_state', $dn_state);
        $stmt->bindParam(':invoice_number', $invoice_no);

        return $stmt->execute();

    }

    public function getDeliveryNoteDetails($dn_number) {
        $sql = "SELECT dn.*, 
                    c.customer_name,
                    c.contact_number as customer_phone,
                    c.email,
                    w.warehouse_name as location_name,
                    ss.vat_charge as vat_percentage
                FROM 
                    delivery_notes dn
                LEFT JOIN
                    customers c ON dn.customer = c.customer_id
                LEFT JOIN
                    warehouses w ON dn.shop_location = w.warehouse_id
                LEFT JOIN
                    sales_order ss ON dn.dn_number = ss.sales_number
                WHERE 
                    dn.dn_number = :dn_number";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':dn_number', $dn_number, PDO::PARAM_STR);
    
        if ($stmt->execute()) {
            return $stmt->fetch(PDO::FETCH_ASSOC); // return single row as associative array
        } else {
            return false;
        }
    }

    public function getCustomerDeliveryNotes($location){
        $sql = "SELECT dn.*, 
                    c.customer_name,
                    w.warehouse_name as location_name,
                    -- Dynamic count of products
                    (SELECT COUNT(*) 
                    FROM sales_order dni 
                    WHERE dni.sales_number = COALESCE(dn.invoice_number, dn.dn_number)
                    ) AS number_of_products,
                    -- Dynamic sum of quantity
                    (SELECT SUM(quantity) 
                    FROM sales_order dni 
                    WHERE dni.sales_number = COALESCE(dn.invoice_number, dn.dn_number)
                    ) AS total_quantity,
                    CASE 
                        WHEN dn.dn_state = 0 THEN 'Cancelled'
                        WHEN dn.dn_state = 1 THEN 'Pending Invoice'
                        WHEN dn.dn_state = 2 THEN 'Invoiced'
                        WHEN dn.dn_state = 3 THEN 'Cancellation Requested'
                        ELSE 'Unknown'
                    END AS dn_status,
                    'customer_delivery_note' as sale_type,
                    CASE 
                        WHEN dn_state = 1 THEN 'Pending Invoice Creation'
                        WHEN dn_state = 2 THEN dn.invoice_number
                        WHEN dn_state = 0 THEN 'Cancelled Delivery Note'
                        ELSE 'N/A'
                    END AS invoice_ref
                FROM 
                    delivery_notes dn
                LEFT JOIN
                    customers c ON dn.customer = c.customer_id
                LEFT JOIN
                    warehouses w ON dn.shop_location = w.warehouse_id
                WHERE 
                    dn.shop_location = :location
                ORDER BY 
                    dn.datecaptured DESC";

        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':location', $location, PDO::PARAM_STR);

        if ($stmt->execute()) {
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } else {
            return false;
        }
    }

    public function getPendingDeliveryNotesByCustomer($customer_id){
        $sql = "SELECT 
                    dn_number,
                    dn_date
                FROM 
                    delivery_notes
                WHERE 
                    customer = ?
                    AND dn_state = 1";

        $stmt = $this->link->prepare($sql);
        $stmt->execute([$customer_id]);

        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function updateSalesOrderItemFromInvoice($line_id, $product_code, $data){

        $sql = "UPDATE 
                    sales_order
                SET 
                    selling_price = :selling_price,
                    subtotal      = :line_subtotal,
                    tax           = :tax_amount,
                    total         = :line_total
                WHERE 
                    id = :dn_number
                    AND product_code = :product_code";

        $stmt = $this->link->prepare($sql);

        $stmt->execute([
            ':selling_price' => $data['selling_price'],
            ':line_subtotal' => $data['line_subtotal'],
            ':tax_amount'    => $data['tax_amount'],
            ':line_total'    => $data['line_total'],
            ':dn_number'     => $line_id,
            ':product_code'  => $product_code
        ]);

        return $stmt->rowCount(); // ✅ Now this runs
    }

    public function getCashPayments($location, $from_date, $to_date) {
        $sql = "SELECT 
                    TIMESTAMP(DATE(p.payment_date), TIME(datecaptured)) AS payment_date, 
                    COALESCE(
                        p.sale_id,
                        GROUP_CONCAT(DISTINCT ip.invoice_number ORDER BY ip.invoice_number SEPARATOR ', ')
                    ) AS sale_id,
                    p.receipt_number,
                    CASE
                        WHEN p.payment_type = 'Order Payment' THEN 'Customer Invoice Payment'
                        WHEN p.payment_type = 'Cash Sale' THEN 'Cash Sale'
                        ELSE p.payment_type
                    END AS transaction_type,
                    c.customer_name,    
                    p.amount_paid
                FROM 
                    payments p
                LEFT JOIN 
                    customers c ON c.customer_id = p.customer
                LEFT JOIN
                    invoice_payments ip 
                        ON p.id = ip.payment_id 
                        AND ip.payment_state = 1
                WHERE 
                    p.location = :location
                    AND p.payment_method = 'Cash'
                    AND DATE(p.payment_date) BETWEEN :from_date AND :to_date
                GROUP BY 
                    p.id
                ORDER BY 
                    p.payment_date DESC";

        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':location', $location);
        $stmt->bindParam(':from_date', $from_date);
        $stmt->bindParam(':to_date', $to_date);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }






    public function fetchInvoices($location, $states = []) {
        $whereState = "";
        if (!empty($states)) {
            // Generate placeholders like :s0, :s1, :s2
            $placeholders = implode(',', array_fill(0, count($states), '?'));
            $whereState = "AND so.order_state IN ($placeholders)";
        }

        $sql = "SELECT 
                    so.order_number,
                    c.customer_name,
                    so.orderdate AS invoice_date,
                    so.total_bill,
                    IFNULL(SUM(ip.amount_paid), 0) AS paid_amount,
                    so.total_bill - IFNULL(SUM(ip.amount_paid), 0) AS outstanding,
                    so.order_state
                FROM 
                    saved_orders so
                LEFT JOIN
                    customers c ON so.customer = c.customer_id
                LEFT JOIN
                    invoice_payments ip ON so.order_number = ip.invoice_number
                WHERE
                    so.invoice_location = ?
                    $whereState
                GROUP BY
                    so.order_number
                ORDER BY
                    so.orderdate DESC";

        $stmt = $this->link->prepare($sql);

        // Bind operator first
        $params = [$location];

        // Bind each requested state
        foreach ($states as $state) {
            $params[] = $state;
        }

        $stmt->execute($params);

        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getScrapDiscount($setting){
        $sql = "SELECT 
                    value as  scrap_amount
                FROM 
                    system_settings
                WHERE
                    setting=:value";

        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':value', $setting);
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);

        return floatval($row['scrap_amount']);
    }





    public function getInvoiceData($invoice_number) {
        $sql = "SELECT so.*, 
                so.lpo_number,
                c.customer_name,
                c.contact_number as customer_phone,
                c.email,
                so.order_state,
                so.discount,
                so.total_bill,
                so.orderdate as invoice_date,
                ss.vat_charge as vat_percentage
            FROM 
                saved_orders so 
            LEFT JOIN
                sales_order ss ON so.order_number = ss.sales_number
            LEFT JOIN
                customers c ON so.customer = c.customer_id
            WHERE 
                so.order_number = :invoice_number";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':invoice_number', $invoice_number, PDO::PARAM_STR);
    
        if ($stmt->execute()) {
            return $stmt->fetch(PDO::FETCH_ASSOC); // return single row as associative array
        } else {
            return false;
        }
    }
    
    public function getCashSaleData($cash_sale_number){
    $sql = "SELECT 
                py.sale_id,
                py.customer,
                py.discount,
                py.payment_state,
                py.payment_date AS cash_sale_date,
                py.discount AS cash_sale_discount,
                py.amount_paid AS cash_sale_amount_paid,

                c.customer_name,
                c.contact_number AS customer_phone,
                c.email,

                so_totals.vat_charge AS vat_percentage,
                so_totals.subtotal,
                so_totals.total_tax,
                so_totals.total_b4_discount,
                (so_totals.total_b4_discount - py.discount) AS total

            FROM payments py
            LEFT JOIN customers c 
                ON py.customer = c.customer_id
            LEFT JOIN (
                SELECT 
                    sales_number,
                    vat_charge,
                    SUM(subtotal) AS subtotal,
                    SUM(tax) AS total_tax,
                    SUM(total) AS total_b4_discount
                FROM sales_order
                WHERE sales_number = :cash_sale_number
                GROUP BY sales_number
            ) so_totals 
                ON py.sale_id = so_totals.sales_number

            WHERE py.sale_id = :cash_sale_number
            ORDER BY py.payment_date ASC
            LIMIT 1";

    $stmt = $this->link->prepare($sql);
    $stmt->bindParam(':cash_sale_number', $cash_sale_number, PDO::PARAM_STR);

    if ($stmt->execute()) {
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    return false;
}
public function getCashSalePayments($cash_sale_number){
    $sql = "SELECT
                payment_method,
                amount_paid,
                payment_date
            FROM payments
            WHERE sale_id = :cash_sale_number
            ORDER BY payment_date ASC";

    $stmt = $this->link->prepare($sql);
    $stmt->bindParam(':cash_sale_number', $cash_sale_number, PDO::PARAM_STR);

    if ($stmt->execute()) {
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    return [];
}



    public function getInvoiceItems($invoice_number) {
        $sql = "SELECT *,
                p.product_name as description,
                so.selling_price,
                so.quantity as qty,
                so.total,
                so.subtotal,
                p.selling_price as p_price,
                so.tax,
                cat.categoryname as categoryname
            FROM 
                sales_order so
            LEFT JOIN
                products p ON p.product_id = so.product_code
            LEFT JOIN
                product_categories cat ON p.category = cat.category_id
            WHERE 
                sales_number = :invoice_number";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':invoice_number', $invoice_number, PDO::PARAM_STR);
    
        if ($stmt->execute()) {
            return $stmt->fetchAll(PDO::FETCH_ASSOC); // return all matching rows
        } else {
            return false;
        }
    }

    public function LpoNumberExists($lpo_number){
        $sql = "SELECT COUNT(*) AS c FROM saved_orders WHERE lpo_number = ?";
        $stmt = $this->link->prepare($sql);
        $stmt->execute([$lpo_number]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return ($row['c'] > 0);
    }

    public function invoiceNumberExists($invoice_number){
        $sql = "SELECT COUNT(*) AS c FROM saved_orders WHERE order_number = ?";
        $stmt = $this->link->prepare($sql);
        $stmt->execute([$invoice_number]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return ($row['c'] > 0);
    }

    public function cashSaleNumberExists($cash_sale_number){
        $sql = "SELECT COUNT(*) AS c FROM payments WHERE sale_id = ?";
        $stmt = $this->link->prepare($sql);
        $stmt->execute([$cash_sale_number]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return ($row['c'] > 0);
    }

    public function getNextAvailableInvoiceNumber($location, $location_type) {
        // Load next document number using your existing logic
        $number = $this->getNextDocumentNumber('invoice', $location, $location_type);

        // Ensure uniqueness
        while ($this->invoiceNumberExists($number)) {
            // Extract numeric part
            $prefix = 'INV';
            $num = intval(substr($number, strlen($prefix)));

            // Increment
            $num++;

            // Regenerate padded number
            $number = $prefix . str_pad($num, 5, '0', STR_PAD_LEFT);
        }

        return $number;
    }

    public function getNextCashSaleNumber($location, $location_type) {
        // Load next document number using your existing logic
        $number = $this->getNextDocumentNumber('cash_sale', $location, $location_type);

        // Ensure uniqueness
        while ($this->cashSaleNumberExists($number)) {
            // Extract numeric part
            $prefix = 'CS';
            $num = intval(substr($number, strlen($prefix)));

            // Increment
            $num++;

            // Regenerate padded number
            $number = $prefix . str_pad($num, 5, '0', STR_PAD_LEFT);
        }

        return $number;
    }

    public function insertSavedOrder($data){
        try {
            $sql = "INSERT INTO saved_orders 
                    (order_number, lpo_number, customer, subtotal, total_tax, discount, total_bill, invoice_location, orderdate, due_date, datecaptured, operator, order_state)
                    VALUES
                    (:order_number, :lpo_number, :customer, :subtotal, :total_tax, :discount, :total_bill, :branch_id, :orderdate, :due_date, :datecaptured, :operator, :order_state)";
            
            $stmt = $this->link->prepare($sql);
            
            $stmt->bindParam(':order_number',   $data['order_number']);
            $stmt->bindParam(':lpo_number',     $data['lpo_number']);
            $stmt->bindParam(':customer',       $data['customer']);
            $stmt->bindParam(':subtotal',       $data['subtotal']);
            $stmt->bindParam(':total_tax',      $data['total_tax']);
            $stmt->bindParam(':discount',       $data['discount']);
            $stmt->bindParam(':total_bill',     $data['total_bill']);
            $stmt->bindParam(':branch_id',      $data['branch_id']);
            $stmt->bindParam(':orderdate',      $data['orderdate']);
            $stmt->bindParam(':due_date',       $data['due_date']);
            $stmt->bindParam(':datecaptured',   $data['datecaptured']);
            $stmt->bindParam(':operator',       $data['operator']);
            $stmt->bindParam(':order_state',    $data['order_state']);
            
            if ($stmt->execute()) {
                return $this->link->lastInsertId(); // Return the saved order ID
            } else {
                return false; // Failed to insert
            }
            
        } catch (PDOException $e) {
            error_log("insertSavedOrder Error: " . $e->getMessage());
            return false;
        }
    }

    public function updateDeliveryNote($dn_number, $invoice_number, $operator, $date_approved, $dn_state){
    
        $sql = "UPDATE 
                    delivery_notes 
                SET 
                    invoice_number = :invoice_number, 
                    approved_by    = :approver, 
                    date_approved  = :date_approved,
                    dn_state       = :dn_state
                WHERE 
                    dn_number    = :dn_number";

        $stmt = $this->link->prepare($sql);

        $stmt->bindParam(':invoice_number', $invoice_number);
        $stmt->bindParam(':approver', $operator);
        $stmt->bindParam(':date_approved', $date_approved);
        $stmt->bindParam(':dn_number', $dn_number);
        $stmt->bindParam(':dn_state', $dn_state);

        return $stmt->execute();
    }

    public function updateSalesOrder($dn_number, $invoice_number){
        $sql = "UPDATE 
                    sales_order 
                SET 
                    sales_number = :invoice_number 
                WHERE 
                    sales_number = :dn_number";

        $stmt = $this->link->prepare($sql);

        $stmt->bindParam(':invoice_number', $invoice_number);
        $stmt->bindParam(':dn_number', $dn_number);

        return $stmt->execute();
    }
    
    public function insertCashSalePayment($data){
        try {

            $sql = "INSERT INTO payments 
                    (tx_id, sale_id, customer, receipt_number, location, payment_method, payment_type, transction_id, payment_date, discount, amount_paid, balance, datecaptured, operator, payment_state)
                    VALUES
                    (:tx_id, :sale_id, :customer, :receipt_number, :location, :payment_method, :payment_type, :transaction_id, :payment_date, :discount, :amount_paid, :balance, :datecaptured, :operator, :payment_state)";
            
            $stmt = $this->link->prepare($sql);

            // 🔥 CORRECT PARAMETER BINDING
            $stmt->bindParam(':tx_id',          $data['tx_id']);
            $stmt->bindParam(':sale_id',        $data['sale_id']);
            $stmt->bindParam(':customer',       $data['customer']);
            $stmt->bindParam(':receipt_number', $data['receipt_number']);
            $stmt->bindParam(':location',       $data['location']); 
            $stmt->bindParam(':payment_method', $data['payment_method']);
            $stmt->bindParam(':payment_type',   $data['payment_type']);
            $stmt->bindParam(':transaction_id', $data['transaction_id']);
            $stmt->bindParam(':payment_date',   $data['payment_date']);
            $stmt->bindParam(':discount',       $data['discount']);
            $stmt->bindParam(':amount_paid',    $data['amount_paid']);
            $stmt->bindParam(':balance',        $data['balance']);
            $stmt->bindParam(':datecaptured',   $data['datecaptured']);
            $stmt->bindParam(':operator',       $data['operator']);
            $stmt->bindParam(':payment_state',  $data['payment_state']);

            if ($stmt->execute()) {
                return $this->link->lastInsertId();
            } else {
                return false;
            }

        } catch (PDOException $e) {
            error_log("insertCashSalePayment Error: " . $e->getMessage());
            return false;
        }
    }

    
    public function insertSalesOrder($data){
        try {
            $sql = "INSERT INTO sales_order
                    (sales_number, product_code, type, department, opening_stock, quantity, closing_stock, selling_price, subtotal, vat_charge, tax, total, date_captured, operator, status)
                    VALUES
                    (:sales_number, :product_code, :type, :department, :opening_stock, :quantity, :closing_stock, :selling_price, :subtotal, :vat_charge, :tax, :total, :date_captured, :operator, :status)";
            
            $stmt = $this->link->prepare($sql);

            $stmt->bindParam(':sales_number',  $data['sales_number']);
            $stmt->bindParam(':product_code',  $data['product_code']);
            $stmt->bindParam(':type',          $data['type']);
            $stmt->bindParam(':department',    $data['department']);
            $stmt->bindParam(':opening_stock', $data['opening_stock']);
            $stmt->bindParam(':quantity',      $data['quantity']);
            $stmt->bindParam(':closing_stock', $data['closing_stock']);
            $stmt->bindParam(':selling_price', $data['selling_price']);
            $stmt->bindParam(':subtotal',      $data['subtotal']);
            $stmt->bindParam(':vat_charge',    $data['vat_charge']);
            $stmt->bindParam(':tax',           $data['tax']);
            $stmt->bindParam(':total',         $data['total']);
            $stmt->bindParam(':date_captured', $data['date_captured']);
            $stmt->bindParam(':operator',      $data['operator']);
            $stmt->bindParam(':status',        $data['status']);

            if ($stmt->execute()) {
                return true; // Successfully inserted
            } else {
                return false; // Failed
            }

        } catch (PDOException $e) {
            error_log("insertSalesOrder Error: " . $e->getMessage());
            return false;
        }
    }

    public function getRecentPayments($location, $limit){
        $sql = "SELECT
                    c.customer_name,
                    p.payment_method,
                    p.transction_id as transaction_id,
                    p.amount_paid,
                    p.datecaptured as payment_date,
                    p.receipt_number,
                    CASE 
                        WHEN p.payment_type = 'Cash Sale' THEN 'Cash Sale'
                        WHEN p.payment_type = 'order Payment' THEN 'Invoice Payment'
                        ELSE p.payment_type
                    END AS payment_type
                FROM
                    payments p
                LEFT JOIN
                    customers c ON p.customer = c.customer_id
                WHERE
                    p.location = :location
                ORDER BY 
                    p.datecaptured DESC
                LIMIT :limit
                ";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':limit', $limit, PDO::PARAM_INT);
        $stmt->bindParam(':location', $location, PDO::PARAM_INT);
        $stmt->execute();
        $payments = $stmt->fetchAll(PDO::FETCH_ASSOC);
        return $payments;
    }

    public function transactionIdExists($transaction_id){
        $sql = "SELECT COUNT(*) FROM payments WHERE transction_id = :transaction_id";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':transaction_id', $transaction_id);
        $stmt->execute();
        return $stmt->fetchColumn() > 0;
    }

    public function getQuickPaymentStats($location){
        $stats = [];

        $sql1 = "SELECT COUNT(*)
                FROM (
                    SELECT
                        inv.customer,
                        inv.total_invoiced - COALESCE(pay.total_paid, 0) AS balance
                    FROM (
                        SELECT
                            customer,
                            SUM(total_bill) AS total_invoiced
                        FROM saved_orders
                        WHERE order_state IN (0,1,2)
                        AND invoice_location = :location
                        GROUP BY customer
                    ) inv
                    LEFT JOIN (
                        SELECT
                            so.customer,
                            SUM(ip.amount_paid) AS total_paid
                        FROM invoice_payments ip
                        INNER JOIN saved_orders so
                            ON so.order_number = ip.invoice_number
                        WHERE ip.payment_state = 1
                        AND so.invoice_location = :location
                        GROUP BY so.customer
                    ) pay ON pay.customer = inv.customer
                    WHERE (inv.total_invoiced - COALESCE(pay.total_paid, 0)) > 0
                ) t
                ";

                $stmt1 = $this->link->prepare($sql1);
                $stmt1->bindValue(':location', $location);
                $stmt1->execute();
                $stats['customers_with_balance'] = (int)$stmt1->fetchColumn();


        $sql2 = "SELECT
                    COALESCE(SUM(inv.total_invoiced - COALESCE(pay.total_paid, 0)), 0) AS total_outstanding,
                    COALESCE(SUM(COALESCE(pay.total_paid, 0)), 0) AS total_paid
                FROM (
                    SELECT
                        customer,
                        SUM(total_bill) AS total_invoiced
                    FROM saved_orders
                    WHERE order_state IN (0,1,2)
                    AND invoice_location = :location
                    GROUP BY customer
                ) inv
                LEFT JOIN (
                    SELECT
                        so.customer,
                        SUM(ip.amount_paid) AS total_paid
                    FROM invoice_payments ip
                    INNER JOIN saved_orders so
                        ON so.order_number = ip.invoice_number
                    WHERE ip.payment_state = 1
                    AND so.invoice_location = :location
                    GROUP BY so.customer
                ) pay ON pay.customer = inv.customer";

            $stmt2 = $this->link->prepare($sql2);
            $stmt2->bindValue(':location', $location);
            $stmt2->execute();

            $result = $stmt2->fetch(PDO::FETCH_ASSOC);

            $stats['total_outstanding'] = (float)$result['total_outstanding'];
            $stats['total_paid']        = (float)$result['total_paid'];



        $sql3 = "SELECT
                    COALESCE(SUM(ip.amount_paid), 0) AS payments_today
                FROM 
                    invoice_payments ip
                INNER JOIN 
                    saved_orders so ON so.order_number = ip.invoice_number
                WHERE
                    ip.payment_state = 1
                    AND DATE(ip.date_paid) = CURDATE()
                    AND so.invoice_location = :location";

        $stmt3 = $this->link->prepare($sql3);
        $stmt3->bindValue(':location', $location);
        $stmt3->execute();
        $stats['payments_today'] = (float)$stmt3->fetchColumn();

        return $stats;
    }


    public function getCustomersWithActiveInvoices($location_id){
        $sql = "SELECT
            c.customer_id,
            c.customer_name,
            IFNULL(o.total_bill, 0) AS total_bill,
            IFNULL(p.amount_paid, 0) AS amount_paid,
            IFNULL(o.total_bill, 0) - IFNULL(p.amount_paid, 0) AS balance
        FROM 
            customers c
            LEFT JOIN (
                SELECT 
                    customer,
                    SUM(total_bill) AS total_bill
                FROM saved_orders
                WHERE order_state IN (0,1,2)
                AND (:location_id = 7000 OR invoice_location = :location_id)
                GROUP BY customer
            ) o ON o.customer = c.customer_id
            LEFT JOIN (
                SELECT 
                    so.customer,
                    SUM(ip.amount_paid) AS amount_paid
                FROM invoice_payments ip
                INNER JOIN saved_orders so
                    ON so.order_number = ip.invoice_number
                WHERE ip.payment_state = 1
                AND (:location_id = 7000 OR so.invoice_location = :location_id)
                GROUP BY so.customer
            ) p ON p.customer = c.customer_id
            WHERE 
                IFNULL(o.total_bill, 0) - IFNULL(p.amount_paid, 0) > 0";

        $stmt = $this->link->prepare($sql);
        $stmt->bindValue(':location_id', $location_id, PDO::PARAM_INT);
        $stmt->execute();

        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }



    public function getActiveInvoicesByCustomer($cid){
        $sql = "SELECT 
                    s.order_number AS invoice_number,
                    s.orderdate AS invoice_date,
                    s.order_state,
                    CAST(s.total_bill AS DECIMAL(12,2)) AS total,
                    IFNULL(SUM(p.amount_paid),0) AS paid,
                    (s.total_bill - IFNULL(SUM(p.amount_paid),0)) AS balance
                FROM 
                    saved_orders s
                LEFT JOIN 
                    invoice_payments p 
                    ON s.order_number = p.invoice_number
                    AND p.payment_state = 1
                WHERE 
                    s.customer = :cid
                    AND s.order_state <> 3
                GROUP BY 
                    s.order_number
                HAVING 
                    balance > 0
                ORDER BY 
                    s.orderdate ASC";

        $stmt = $this->link->prepare($sql);
        $stmt->bindValue(':cid', $cid);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getTotalCustomerInvoiceBalances($cid) {
        $sql = "SELECT
                    IFNULL(SUM(so.total_bill),0) 
                    - IFNULL((
                        SELECT SUM(ip.amount_paid)
                        FROM invoice_payments ip
                        INNER JOIN saved_orders so2 
                            ON ip.invoice_number = so2.order_number
                        WHERE so2.customer = so.customer
                        AND so2.order_state IN (0,1)
                        AND ip.payment_state = 1
                    ),0) AS total_balance
                FROM saved_orders so
                WHERE so.customer = :cid
                AND so.order_state IN (0,1)";

        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':cid', $cid, PDO::PARAM_INT);
        $stmt->execute();

        $result = $stmt->fetch(PDO::FETCH_ASSOC);

        return $result['total_balance'] ?? 0;
    }

    public function getActiveInvoices($customer_id) {
        // SQL to fetch active invoices
        $sql = "SELECT 
                    so.order_number AS invoice_number,
                    so.total_bill AS total_amount,
                    IFNULL(SUM(ip.amount_paid), 0) AS paid_amount,
                    so.order_state
                FROM 
                    saved_orders so
                LEFT JOIN 
                    invoice_payments ip 
                    ON so.order_number = ip.invoice_number
                WHERE 
                    so.customer = :customer_id
                GROUP BY 
                    so.order_number
                HAVING 
                    total_amount - paid_amount > 0
                ORDER BY 
                    so.orderdate ASC";

        // Prepare the statement
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':customer_id', $customer_id, PDO::PARAM_INT);

        // Execute
        $stmt->execute();

        // Fetch all active invoices as associative array
        $invoices = $stmt->fetchAll(PDO::FETCH_ASSOC);

        return $invoices;
    }


    public function getInvoiceForPayment($invoice_number, $customer_id){
        $sql = "SELECT 
                order_number AS invoice_number,
                total_bill,
                COALESCE((SELECT SUM(ip.amount_paid) FROM invoice_payments ip WHERE invoice_number =:invoice_number AND payment_state= 1),0) as paid_amount,
                order_state
            FROM 
                saved_orders
            WHERE 
                order_number = :invoice_number
                AND customer = :customer
                AND order_state IN (0,1)";

        $stmt = $this->link->prepare($sql);
        $stmt->bindValue(':invoice_number', $invoice_number);
        $stmt->bindValue(':customer', $customer_id);
        $stmt->execute();

        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    public function insertScrapCollection(array $data){
        try {
            $sql = "INSERT INTO scrap_collection (transaction_number, customer, quantity, scrap_discount, total_scrap_discount, location, date_collected, date_captured, operator)
                    VALUES (:transaction_number, :customer, :quantity, :scrap_discount, :total_scrap_discount, :location, :date_collected, :date_captured, :operator)";

            $stmt = $this->link->prepare($sql);

            return $stmt->execute([
                ':transaction_number'    => $data['transaction_number'],
                ':customer'              => $data['customer_id'],
                ':quantity'              => $data['quantity'],
                ':scrap_discount'        => $data['amount'],
                ':total_scrap_discount'  => $data['total_scrap_disc'],
                ':location'              => $data['location'],
                ':date_collected'        => $data['date_collected'],
                ':date_captured'         => $data['date_captured'],
                ':operator'              => $data['operator']
            ]);

        } catch (PDOException $e) {
            error_log('insertScrapCollection Error: ' . $e->getMessage());
            return false;
        }
    }










           
}