<?php

include_once 'config.php';

class stock_model
{
    public $link;

    public function __construct()
    {
        $db_connection = new dbConnection();
        $this->link = $db_connection->connect();
    }

    public function beginTransaction() {
        $this->link->beginTransaction();
    }

        public function commit() {
        $this->link->commit();
    }

    public function rollback() {
        $this->link->rollBack();
    }    

    public function countTransfersin($location){
        $sql = "SELECT 
                    COUNT(*) AS requested_transfers
                FROM 
                    stock_transfers_final
                WHERE 
                    transfer_state = 1
                    AND transfer_to = :location
        ";

        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':location', $location, PDO::PARAM_INT);
        $stmt->execute();

        return (int) $stmt->fetchColumn();
    }
    
    public function countTransfersout($location){
        $sql = "SELECT 
                    COUNT(*) AS requested_transfers
                FROM 
                    stock_transfers_final
                WHERE 
                    transfer_state = 1
                    AND transfer_from = :location
        ";

        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':location', $location);
        $stmt->execute();

        return (int) $stmt->fetchColumn();
    }

    public function getStockTransfers($location, $type = 'all'){
        $sql = "SELECT 
                    COALESCE(tpc.company_name, w.warehouse_name, bh.branch_name) as source_name,
                    stf.transfer_from,
                    COALESCE(wh.warehouse_name, b.branch_name) as destination_name,
                    stf.transfer_to,
                    stf.transfer_state,
                    stf.transfer_date,
                    stf.transfer_id AS delivery_note_number,
                    COUNT(sti.product_id) AS number_of_items,
                    COALESCE(emp.fullname,'Default Account') as initiated_by,
                    CASE 
                        WHEN stf.transfer_state = 1 THEN 'Waiting Approval'
                        WHEN stf.transfer_state = 2 THEN 'Completed Transfer'
                        WHEN stf.transfer_state = 0 THEN 'Cancelled'
                    END as transfer_status,
                    CASE 
                        WHEN stf.transfer_to = :location THEN 'IN'
                        WHEN stf.transfer_from = :location THEN 'OUT'
                    END AS transfer_type
                FROM 
                    stock_transfers_final stf
                LEFT JOIN 
                    warehouses w ON stf.transfer_from = w.warehouse_id
                LEFT JOIN 
                    branches b ON stf.transfer_to = b.branch_id
                LEFT JOIN 
                    third_party_companies tpc ON stf.transfer_from = tpc.company_id
                LEFT JOIN 
                    warehouses wh ON stf.transfer_to = wh.warehouse_id
                LEFT JOIN
                    branches bh ON stf.transfer_from = bh.branch_id
                LEFT JOIN 
                    stock_transfer_items sti ON sti.transfer_id = stf.transfer_id
                LEFT JOIN 
                    employees emp ON stf.operator = emp.employee_code
                WHERE
                    ((:type = 'in' AND stf.transfer_to = :location)
                    OR (:type = 'out' AND stf.transfer_from = :location)
                    OR (:type = 'all' AND (:location IN (stf.transfer_to, stf.transfer_from))))
                GROUP BY 
                    stf.transfer_id
                ORDER BY 
                    stf.datecaptured DESC";

        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':location', $location, PDO::PARAM_INT);
        $stmt->bindParam(':type', $type, PDO::PARAM_STR);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }


    public function getDeliveryNote($dn_number) {
        $sql = "SELECT 
                    transfer_id as dn_number, 
                    stf.transfer_state,
                    bh.branch_name as delivery_location,
                    transfer_date as date, 
                    emp.fullname as initiated_by,
                    em.fullname as received_by,
                    e.fullname as declined_by,
                    emm.fullname as cancelled_by,
                    stf.operator,
                    driver as delivered_by,
                    driver_delivery_date,
                    reason,
                    stf.transfer_date as initiated_date,
                    datereceived as received_date,
                    date_declined as date_declined,
                    date_cancelled as date_cancelled
                FROM 
                    stock_transfers_final stf
                LEFT JOIN
                    employees emp ON stf.operator = emp.employee_code
                LEFT JOIN
                    employees em ON stf.received_by = em.employee_code
                LEFT JOIN
                    employees e ON stf.declined_by = e.employee_code
                LEFT JOIN
                    employees emm ON stf.cancelled_by = emm.employee_code
                LEFT JOIN
                    branches bh ON stf.transfer_to = bh.branch_id
                WHERE 
                    transfer_id = :dn_number";
        $stmt = $this->link->prepare($sql);
        $stmt->bindparam(":dn_number", $dn_number);
        $stmt->execute();
        $result = $stmt->fetch();
        

        return $result;
    }

    /**
     * Get delivery note items by DN number
     */
    public function getDeliveryNoteItems($dn_number) {
        $sql = "SELECT 
                    sti.id,
                    sti.product_id,
                    sti.transfer_from,
                    sti.transfer_to,
                    sti.qty_initiated as qty, 
                    sti.qty_initiated,
                    sti.original_qty_initiated,
                    sti.qty_accepted,
                    p.product_name as description,
                    pc.categoryname,
                    sti.reason
                FROM 
                    stock_transfer_items sti
                LEFT JOIN
                    products p ON sti.product_id = p.product_id
                LEFT JOIN
                    product_categories pc ON p.category = pc.category_id
                WHERE 
                    transfer_id = :dn_number";
        $stmt = $this->link->prepare($sql);
        
        $stmt->execute([':dn_number' => $dn_number]);
        
        $items = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        return $items;
    }

    public function getDeliveryNoteItemQty($itemId) {
        $stmt = $this->link->prepare("SELECT qty_initiated FROM stock_transfer_items WHERE id=?");
        $stmt->execute([$itemId]);
        return (int)$stmt->fetchColumn();
    }

    public function updateAdjustedStock($tn_number, $updates) {
        $this->link->beginTransaction();
        try {
            foreach($updates as $u) {
                $stmt = $this->link->prepare("UPDATE stock_transfer_items 
                    SET qty_accepted =:received_qty, reason=:reason 
                    WHERE id=:item_id
                ");
                $stmt->execute([
                    ':received_qty' => $u['received_qty'],
                    ':reason' => $u['reason'],
                    ':item_id' => $u['item_id']
                ]);
            }
            $this->link->commit();
            return true;
        } catch(Exception $e) {
            $this->link->rollBack();
            return false;
        }
    }

    public function updateTransferNoteFinal($tn_number, $state, $operator, $date, $reason){
        $sql = "UPDATE 
                    stock_transfers_final
                SET 
                    transfer_state = :state,
                    declined_by = :declined_by,
                    date_declined = :date_declined,
                    reason = :reason
                WHERE
                    transfer_id = :tn_number";

        $smtp = $this->link->prepare($sql);

        $smtp->bindParam(':state', $state, PDO::PARAM_INT);
        $smtp->bindParam(':declined_by', $operator, PDO::PARAM_INT);
        $smtp->bindParam(':date_declined', $date, PDO::PARAM_STR);
        $smtp->bindParam(':reason', $reason, PDO::PARAM_STR);
        $smtp->bindParam(':tn_number', $tn_number, PDO::PARAM_STR);

        if ($smtp->execute()) {
            return true;
        } else {
            return false;
        }
    }

    public function approveTransferNote($tn_number, $state, $operator, $date, $reason, $driver_date, $delivery_note_number = null){
        $sql = "UPDATE 
                    stock_transfers_final
                SET 
                    transfer_state = :state,
                    received_by = :received_by,
                    datereceived = :date_accepted,
                    reason = :reason,
                    driver_delivery_date =:driver_delivery_date,
                    delivery_note_number =:delivery_note_number
                WHERE
                    transfer_id = :tn_number";

        $smtp = $this->link->prepare($sql);

        $smtp->bindParam(':state', $state, PDO::PARAM_INT);
        $smtp->bindParam(':received_by', $operator, PDO::PARAM_INT);
        $smtp->bindParam(':date_accepted', $date, PDO::PARAM_STR);
        $smtp->bindParam(':reason', $reason, PDO::PARAM_STR);
        $smtp->bindParam(':tn_number', $tn_number, PDO::PARAM_STR);
        $smtp->bindParam(':driver_delivery_date', $driver_date, PDO::PARAM_STR);
        if ($delivery_note_number === null) {
            $smtp->bindValue(':delivery_note_number', null, PDO::PARAM_NULL);
        } else {
            $smtp->bindValue(':delivery_note_number', $delivery_note_number, PDO::PARAM_STR);
        }

        if ($smtp->execute()) {
            return true;
        } else {
            return false;
        }
    }

    public function declineTransferNote($tn_number, $state, $operator, $date, $reason){
        try {
            // Begin transaction
            $this->link->beginTransaction();

            // --- 1. Update stock_transfers_final ---
            $sqlFinal = "UPDATE 
                            stock_transfers_final
                        SET 
                            transfer_state = :state,
                            declined_by = :declined_by,
                            date_declined = :date_declined,
                            reason = :reason
                        WHERE
                            transfer_id = :tn_number";

            $stmtFinal = $this->link->prepare($sqlFinal);
            $stmtFinal->bindParam(':state', $state, PDO::PARAM_INT);
            $stmtFinal->bindParam(':declined_by', $operator, PDO::PARAM_INT);
            $stmtFinal->bindParam(':date_declined', $date, PDO::PARAM_STR);
            $stmtFinal->bindParam(':reason', $reason, PDO::PARAM_STR);
            $stmtFinal->bindParam(':tn_number', $tn_number, PDO::PARAM_STR);

            if (!$stmtFinal->execute()) {
                $this->link->rollBack();
                return "Failed to update stock_transfers_final";
            }

            // --- 2. Update stock_transfer_items ---
            $sqlItems = "UPDATE 
                            stock_transfer_items
                        SET 
                            transfer_state = :state,
                            declined_by = :declined_by,
                            date_declined = :date_declined,
                            reason = :reason
                        WHERE
                            transfer_id = :tn_number";

            $stmtItems = $this->link->prepare($sqlItems);
            $stmtItems->bindParam(':state', $state, PDO::PARAM_INT);
            $stmtItems->bindParam(':declined_by', $operator, PDO::PARAM_INT);
            $stmtItems->bindParam(':date_declined', $date, PDO::PARAM_STR);
            $stmtItems->bindParam(':reason', $reason, PDO::PARAM_STR);
            $stmtItems->bindParam(':tn_number', $tn_number, PDO::PARAM_STR);

            if (!$stmtItems->execute()) {
                $this->link->rollBack();
                return "Failed to update stock_transfer_items";
            }

            // If both succeeded → commit
            $this->link->commit();
            return true;

        } catch (Exception $e) {
            // Rollback on ANY error
            $this->link->rollBack();
            return "Transaction error: " . $e->getMessage();
        }
    }

    // returns the transfer_to (warehouse_id) for a given transfer number
    public function getTransferDestination($tn_number) {
        $sql = "SELECT 
                    stf.transfer_to AS branch_id,
                    b.branch_name,
                    stf.driver,
                    stf.operator
                FROM 
                    stock_transfers_final stf
                LEFT JOIN 
                    branches b ON stf.transfer_to = b.branch_id
                WHERE 
                    stf.transfer_id = :tn_number
                LIMIT 1";

        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':tn_number', $tn_number, PDO::PARAM_STR);
        $stmt->execute();
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }


    // returns current stock (quantity) for a product in a specific branch
    public function getBranchQuantity($product_id, $branch_id) {
        $sql = "SELECT quantity FROM branch_products WHERE product_id = :product_id AND branch_id = :branch_id LIMIT 1";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':product_id', $product_id, PDO::PARAM_INT);
        $stmt->bindParam(':branch_id', $branch_id, PDO::PARAM_INT); // use STR if your branch id can be non-int
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row ? floatval($row['quantity']) : 0.0;
    }

    public function getCurrentQuantity($transfer_from, $product_id) {
        // Check warehouses
        $stmt = $this->link->prepare("SELECT quantity FROM branch_products WHERE branch_id=:warehouse_id AND product_id=:product");
        $stmt->bindParam(':warehouse_id', $transfer_from);
        $stmt->bindParam(':product', $product_id, PDO::PARAM_INT);
        $stmt->execute();
        $q = $stmt->fetchColumn();
        if ($q !== false) return floatval($q);

        // Check third-party companies
        $stmt = $this->link->prepare("SELECT quantity FROM third_party_products WHERE company_id=? AND product_id=?");
        $stmt->execute([$transfer_from, $product_id]);
        $q = $stmt->fetchColumn();
        if ($q !== false) return floatval($q);

        // Check branches
        $stmt = $this->link->prepare("SELECT quantity FROM branch_products WHERE branch_id=? AND product_id=?");
        $stmt->execute([$transfer_from, $product_id]);
        $q = $stmt->fetchColumn();
        if ($q !== false) return floatval($q);

        return 0; // default if not found
    }

    public function updateTransferItem($item_id, $tn_number, $receiver_opening_stock, $qty_accepted, $receiver_closing_stock, $transfer_state, $date_accepted, $accepted_by){
        try {
            $sql = "UPDATE 
                        stock_transfer_items 
                    SET
                        receiver_opening_stock = :receiver_opening_stock,
                        qty_accepted           = :qty_accepted, 
                        receiver_closing_stock = :receiver_closing_stock,
                        transfer_state         = :transfer_state,
                        date_accepted          = :date_accepted, 
                        accepted_by            = :accepted_by
                    WHERE
                        transfer_id = :tn_number
                        AND id = :item_id";

            $stmt = $this->link->prepare($sql);

            $stmt->bindParam(':receiver_opening_stock', $receiver_opening_stock);
            $stmt->bindParam(':qty_accepted', $qty_accepted);
            $stmt->bindParam(':receiver_closing_stock', $receiver_closing_stock);
            $stmt->bindParam(':transfer_state', $transfer_state);
            $stmt->bindParam(':date_accepted', $date_accepted);
            $stmt->bindParam(':accepted_by', $accepted_by);
            $stmt->bindParam(':tn_number', $tn_number);
            $stmt->bindParam(':item_id', $item_id);

            if ($stmt->execute()) {
                return true;
            }

            return false;

        } catch (PDOException $e) {
            error_log("updateTransferItem Error: " . $e->getMessage());
            return false;
        }
    }

    public function updateSourceQuantity($transfer_from, $product_id, $new_quantity) {

        /* ---------------------------
            1. Update warehouse_products
        ----------------------------*/
        $sql = "UPDATE 
                warehouse_products 
            SET 
                quantity = :qty 
            WHERE 
                warehouse_id = :src 
                AND product_id = :pid";
        $stmt = $this->link->prepare($sql);
        $stmt->bindValue(':qty', $new_quantity);
        $stmt->bindValue(':src', $transfer_from);
        $stmt->bindValue(':pid', $product_id, PDO::PARAM_INT);
        $stmt->execute();

        if ($stmt->rowCount() > 0) {
            return true;
        }


        /* --------------------------------
            2. Update third_party_products
        ---------------------------------*/
        $sql = "UPDATE
                    third_party_products 
                SET 
                    quantity = :qty 
                WHERE 
                    company_id = :src 
                    AND product_id = :pid";
        $stmt = $this->link->prepare($sql);
        $stmt->bindValue(':qty', $new_quantity);
        $stmt->bindValue(':src', $transfer_from);
        $stmt->bindValue(':pid', $product_id, PDO::PARAM_INT);
        $stmt->execute();

        if ($stmt->rowCount() > 0) {
            return true;
        }


        /* ------------------------------
            3. Update branch_products
        -------------------------------*/
        $sql = "UPDATE
                branch_products 
            SET 
                quantity = :qty 
            WHERE 
                branch_id = :src 
                AND product_id = :pid";
        $stmt = $this->link->prepare($sql);
        $stmt->bindValue(':qty', $new_quantity);
        $stmt->bindValue(':src', $transfer_from);
        $stmt->bindValue(':pid', $product_id, PDO::PARAM_INT);
        $stmt->execute();

        if ($stmt->rowCount() > 0) {
            return true;
        }


        /* ------------------------------------
            No table matched this transfer_from
        -------------------------------------*/
        return false;
    }

    public function getProductsByLocation($type, $location_id){

        if ($type === 'warehouse') {

            $sql = "SELECT p.product_id, p.product_name, p.selling_price
                    FROM warehouse_products wp
                    JOIN products p ON p.product_id = wp.product_id
                    WHERE wp.warehouse_id = :id";

        } else {

            $sql = "SELECT p.product_id, p.product_name, p.selling_price
                    FROM branch_products bp
                    JOIN products p ON p.product_id = bp.product_id
                    WHERE bp.branch_id = :id";
        }

        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':id', $location_id, PDO::PARAM_INT);
        $stmt->execute();

        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }


    public function updateBranchQuantity($product_id, $branch_id, $new_quantity) {
        $stmt = $this->link->prepare("UPDATE branch_products SET quantity=? WHERE branch_id =? AND product_id=?");
        return $stmt->execute([$new_quantity, $branch_id, $product_id]);
    }

    public function getproducts($location){
        $sql = "SELECT
                    p.product_id,
                    bp.branch_id,
                    p.product_name,
                    p.product_code,
                    p.selling_price,
                    bp.min_level,
                    c.categoryname,
                    u.unitname,
                    bp.quantity,
                    bp.opening_stock
                FROM 
                    branch_products bp
                LEFT JOIN
                    products p ON p.product_id = bp.product_id
                LEFT JOIN
                    product_categories c ON p.category = c.category_id
                LEFT JOIN
                    product_units u ON p.unit = u.unit_id
                WHERE
                    bp.branch_id = :branch";
                    
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':branch', $location, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

        public function countProducts($location) {
        $sql = "SELECT 
                    COUNT(*) AS total 
                FROM 
                    branch_products 
                WHERE 
                    branch_id = :warehouse";

        $stmt = $this->link->prepare($sql);
        $stmt->bindValue(':warehouse', $location, PDO::PARAM_INT);
        $stmt->execute();

        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row['total'] ?? 0;
    }

    public function getstockSheet($location, $start_date, $end_date) {
        $sql = "SELECT 
                    p.product_id,
                    p.product_code,
                    p.product_name,
                    c.categoryname,
                    bp.quantity as current_qty,
                    bp.min_level,

                    -- OPENING STOCK as of start_date
                    (
                        COALESCE(bp.opening_stock, 0)

                        + COALESCE((
                            SELECT SUM(sti.qty_accepted)
                            FROM stock_transfer_items sti
                            WHERE sti.product_id = bp.product_id
                            AND sti.transfer_to = :branch
                            AND DATE(sti.date_accepted) < :start_date
                            AND sti.transfer_state = 2
                        ), 0)

                        - COALESCE((
                            SELECT SUM(stii.qty_initiated)
                            FROM stock_transfer_items stii
                            WHERE stii.product_id = bp.product_id
                            AND stii.transfer_from = :branch
                            AND DATE(stii.date_accepted) < :start_date
                            AND stii.transfer_state = 2
                        ), 0)

                        - COALESCE((
                            SELECT SUM(s.quantity)
                            FROM sales_order s
                            WHERE s.product_code = bp.product_id
                            AND s.department = :branch
                            AND DATE(s.date_captured) < :start_date
                            AND s.status = 1
                        ), 0)
                    ) as opening_stock,

                    -- Transfer In within period
                    (
                        SELECT COALESCE(SUM(sti.qty_accepted), 0)
                        FROM stock_transfer_items sti
                        WHERE sti.product_id = bp.product_id
                        AND sti.transfer_to = :branch
                        AND sti.transfer_state = 2
                        AND DATE(sti.date_accepted) BETWEEN :start_date AND :end_date
                    ) AS transfer_in,

                    -- Transfer Out within period
                    (
                        SELECT 
                            COALESCE(SUM(stii.qty_initiated), 0)
                        FROM 
                            stock_transfer_items stii
                        WHERE 
                            stii.product_id = bp.product_id
                            AND stii.transfer_from = :branch
                            AND stii.transfer_state = 2
                            AND DATE(stii.date_accepted) >= :start_date AND DATE(stii.date_accepted) <= :end_date
                    ) AS transfer_out,

                    -- SOLD during period
                    (
                        SELECT COALESCE(SUM(s.quantity), 0)
                        FROM sales_order s
                        WHERE s.product_code = bp.product_id
                        AND s.department = :branch
                        AND DATE(s.date_captured) BETWEEN :start_date AND :end_date
                        AND s.status = 1
                    ) AS sold

                FROM branch_products bp
                LEFT JOIN products p ON bp.product_id = p.product_id
                LEFT JOIN product_categories c ON c.category_id = p.category

                -- 🔥🔥 THE FIX: ONLY RETURN PRODUCTS FROM THIS LOCATION
                WHERE bp.branch_id = :branch
                        AND DATE(bp.datecaptured) <= :end_date

                ORDER BY p.product_name ASC";

        $stmt = $this->link->prepare($sql);
        $stmt->bindValue(':branch', $location);
        $stmt->bindValue(':start_date', $start_date);
        $stmt->bindValue(':end_date', $end_date);
        $stmt->execute();

        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }


    public function getProductBreakdown($branch_id, $product_id) {
        $sql = "SELECT 
                    sti.date_accepted AS date,
                    sti.transfer_date AS transfer_date,
                    'Transfer In' AS type,
                    sti.qty_accepted AS quantity,
                    CONCAT('Transfered Items From ', COALESCE(sfw.warehouse_name, b.branch_name, tpc.company_name, 'Unknown')) AS reference,
                    sti.receiver_closing_stock AS closing_stock
                FROM 
                    stock_transfer_items sti
                LEFT JOIN 
                    warehouses sfw ON sfw.warehouse_id = sti.transfer_from
                LEFT JOIN
                    branches b ON b.branch_id = sti.transfer_from
                LEFT JOIN
                    third_party_companies tpc ON tpc.company_id = sti.transfer_from
                WHERE 
                    sti.product_id = :product
                    AND sti.transfer_to = :branch
                    AND sti.transfer_state = 2

            UNION ALL

                SELECT 
                    sti.datecaptured AS date,
                    sti.transfer_date AS transfer_date,
                    'Transfer Out' AS type,
                    sti.qty_initiated AS quantity,
                    CONCAT('Transfered Items To ', IFNULL(stw.branch_name, 'Unknown')) AS reference,
                    sti.sender_closing_stock AS closing_stock
                FROM 
                    stock_transfer_items sti
                LEFT JOIN 
                    branches stw ON stw.branch_id = sti.transfer_to
                WHERE 
                    sti.product_id = :product
                    AND sti.transfer_from = :branch
                    AND sti.transfer_state IN (1,2)

            UNION ALL

                SELECT 
                    s.date_captured AS date,
                    s.date_captured as transfer_date,
                    'Sold' AS type,
                    s.quantity AS quantity,
                    CONCAT('Invoice: ', s.sales_number) AS reference,
                    s.closing_stock AS closing_stock
                FROM 
                    sales_order s
                WHERE 
                    s.product_code = :product
                    AND s.department = :branch
                    AND s.status = 1

            UNION ALL
                SELECT
                    wp.datecaptured as date,
                    wp.datecaptured as transfer_date,
                    'Opening Stock' as type,
                    opening_stock as quantity,
                    'Product Initialization' as reference,
                    wp.opening_stock AS closing_stock
                FROM
                    branch_products wp
                WHERE
                    wp.product_id = :product
                    AND wp.branch_id = :branch

            ORDER BY date DESC"; // latest first

        $stmt = $this->link->prepare($sql);
        $stmt->bindValue(':branch', $branch_id, PDO::PARAM_INT);
        $stmt->bindValue(':product', $product_id, PDO::PARAM_INT);
        $stmt->execute();

        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }


    public function getProductDetails($product_id) {
        $sql ="SELECT 
                p.product_id,
                p.product_code,
                p.product_name,
                p.selling_price,
                c.categoryname,
                u.unitname
            FROM 
                products p
            LEFT JOIN 
                product_categories c ON p.category = c.category_id
            LEFT JOIN 
                product_units u ON p.unit = u.unit_id
            WHERE 
                p.product_id = :product
                LIMIT 1
        ";

        $stmt = $this->link->prepare($sql);
        $stmt->bindValue(':product', $product_id, PDO::PARAM_INT);
        $stmt->execute();

        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    public function updateProductDetails($product_id, $warehouse_id, $min_order_level) {
        $sql = "UPDATE 
                    warehouse_products 
                SET 
                    min_level = :min_order_level
                WHERE 
                    product_id = :product_id
                    AND warehouse_id = :warehouse_id";

        $stmt = $this->link->prepare($sql);
        $stmt->bindValue(':min_order_level', $min_order_level, PDO::PARAM_INT);
        $stmt->bindValue(':product_id', $product_id, PDO::PARAM_INT);
        $stmt->bindValue(':warehouse_id', $warehouse_id, PDO::PARAM_INT);

        return $stmt->execute();
    }

    public function getProductStockSheet($branch_id, $product_id, $start_date, $end_date){
        $sql = "SELECT
                    bp.datecaptured AS date,
                    'Opening Stock' AS description,
                    0 AS opening_stock,
                    bp.opening_stock AS quantity,
                    bp.opening_stock AS closing_stock
                FROM 
                    branch_products bp
                WHERE 
                    bp.product_id = :product
                    AND bp.branch_id = :branch
                    AND bp.datecaptured BETWEEN :start AND :end
            UNION ALL
                    SELECT
                        so.date_captured as date,
                        so.type as description,
                        so.opening_stock as opening_stock,
                        so.quantity as quantity,
                        so.closing_stock as closing_stock
                    FROM 
                        sales_order so 
                    WHERE
                        so.department =:branch 
                        AND so.product_code =:product  
                        AND so.date_captured BETWEEN :start AND :end
            UNION ALL
                SELECT 
                    sti.date_accepted as date,
                    'Stock Transfer In' as description,
                    sti.receiver_opening_stock as opening_stock,
                    sti.qty_accepted as quantity,
                    sti.receiver_closing_stock as closing_stock
                FROM
                    stock_transfer_items sti
                WHERE
                    sti.transfer_to =:branch
                    AND sti.transfer_state = 2
                    AND sti.product_id =:product
                    AND sti.date_accepted BETWEEN :start AND :end
            UNION ALL
                SELECT
                    sti.datecaptured as date,
                    'Stock Transfer Out' as description,
                    sti.sender_opening_stock as opening_stock,
                    sti.qty_initiated as quantity,
                    sti.sender_closing_stock as closing_stock
                FROM
                    stock_transfer_items sti
                WHERE
                    sti.transfer_from =:branch
                    AND sti.transfer_state IN (1,2)
                    AND sti.product_id =:product
                    AND sti.datecaptured BETWEEN :start AND :end
                    

            ORDER BY date DESC
        ";

        $stmt = $this->link->prepare($sql);

        $stmt->bindValue(':branch', $branch_id, PDO::PARAM_INT);
        $stmt->bindValue(':product', $product_id, PDO::PARAM_INT);
        $stmt->bindValue(':start', $start_date);
        $stmt->bindValue(':end', $end_date);

        $stmt->execute();

        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } 
    
    public function productExistsAtDestination($product_id, $destination_id) {
        $sql = "SELECT COUNT(*) FROM (
                SELECT product_id FROM warehouse_products WHERE warehouse_id = :destination_id AND product_id = :product_id
                UNION
                SELECT product_id FROM branch_products WHERE branch_id = :destination_id AND product_id = :product_id
            ) AS combined
        ";

        $stmt = $this->link->prepare($sql);
        $stmt->execute([
            ':destination_id' => $destination_id,
            ':product_id' => $product_id
        ]);
        
        return $stmt->fetchColumn() > 0;
    }

    public function getProductQuantityAtDestination($product_id, $destination_id) {
        // First, try warehouse
        $sql = "SELECT quantity FROM warehouse_products WHERE warehouse_id = :destination_id AND product_id = :product_id";
        $stmt = $this->link->prepare($sql);
        $stmt->execute([
            'destination_id' => $destination_id,
            'product_id' => $product_id
        ]);
        $warehouseResult = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($warehouseResult) {
            return ['quantity' => floatval($warehouseResult['quantity'])];
        }

        // Try branch
        $sql1 = "SELECT quantity FROM branch_products WHERE branch_id = :destination_id AND product_id = :product_id";
        $stmt1 = $this->link->prepare($sql1);
        $stmt1->execute([
            'destination_id' => $destination_id,
            'product_id' => $product_id
        ]);
        $branchResult = $stmt1->fetch(PDO::FETCH_ASSOC);

        if ($branchResult) {
            return ['quantity' => floatval($branchResult['quantity'])];
        }

        //Try Third Party Company
        $sql2 = "SELECT quantity FROM third_party_products WHERE company_id=:destination_id AND product_id=:product_id";
        $stmt2 = $this->link->prepare($sql2);
        $stmt2->execute(['destination_id'=>$destination_id, 'product_id'=> $product_id]);
        $thirdCompanyResult = $stmt2->fetch(PDO::FETCH_ASSOC);

        if($thirdCompanyResult){
            return ['quantity'=>floatval($thirdCompanyResult['quantity'])];
        }

        // If not found in either
        return ['quantity' => 0];
    }

    public function getStockReportData($location, $filter = null)
{
    $sql = "SELECT 
                wp.product_id,
                p.product_name,
                wp.quantity,
                wp.min_level,
                pc.categoryname,

                /* 90-day sales per product */
                COALESCE(ps.product_sales_90, 0) AS sales_90_days,

                /* Total 90-day sales for this warehouse */
                COALESCE(ts.total_sales_90, 0) AS total_sales_90_days

            FROM 
                warehouse_products wp
            LEFT JOIN 
                products p ON wp.product_id = p.product_id
            LEFT JOIN 
                product_categories pc ON p.category = pc.category_id

            /* Product-level sales (last 90 days) */
            LEFT JOIN (
                SELECT 
                    product_code,
                    SUM(quantity) AS product_sales_90
            FROM 
                sales_order
            WHERE 
                department = :warehouse
                AND date_captured >= DATE_SUB(CURDATE(), INTERVAL 90 DAY)
            GROUP BY 
                product_code
                ) ps ON ps.product_code = wp.product_id

            /* Total sales (last 90 days) */
            LEFT JOIN (
                SELECT 
                    SUM(quantity) AS total_sales_90
                FROM 
                    sales_order
                WHERE 
                    department = :warehouse
                    AND date_captured >= DATE_SUB(CURDATE(), INTERVAL 90 DAY)
                ) ts ON 1 = 1

            WHERE 
                wp.warehouse_id = :warehouse";

            if ($filter === 'low_level') {
                $sql .= " AND wp.quantity > 0 AND wp.quantity <= wp.min_level";
            } elseif ($filter === 'out_of_stock') {
                $sql .= " AND wp.quantity = 0";
            }

            $sql .= " ORDER BY p.product_name ASC";

        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':warehouse', $location, PDO::PARAM_INT);
        $stmt->execute();

        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getProductSalesLast90Days($location, $product_id){
        $sql = "SELECT 
                    SUM(quantity)
                FROM 
                    sales_order
                WHERE 
                    department = :location
                    AND product_code = :product
                    AND date_captured >= DATE_SUB(CURDATE(), INTERVAL 90 DAY)";

        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':location', $location);
        $stmt->bindParam(':product', $product_id);
        $stmt->execute();
        return (int)$stmt->fetchColumn();
    }

    public function getTotalSalesLast90Days($location){
        $sql = "SELECT
                     SUM(quantity) 
                FROM 
                    sales_order
                WHERE 
                    department = :location
                    AND date_captured >= DATE_SUB(CURDATE(), INTERVAL 90 DAY)";

        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':location', $location);
        $stmt->execute();
        return (int)$stmt->fetchColumn();
    }

    public function sendNotification($sender, $receiver, $purpose, $identifier, $message){
        $sql = "INSERT INTO notifications 
                (sender, receiver, purpose, identifier, message, datecaptured, notificaton_state)
                VALUES 
                (:sender, :receiver, :purpose, :identifier, :message, NOW(), :state)";

        $stmt = $this->link->prepare($sql);

        $state = 0; // 0 = unread, 1 = read

        $stmt->bindParam(':sender', $sender, PDO::PARAM_INT);
        $stmt->bindParam(':receiver', $receiver, PDO::PARAM_INT);
        $stmt->bindParam(':purpose', $purpose, PDO::PARAM_STR);
        $stmt->bindParam(':identifier', $identifier, PDO::PARAM_STR);
        $stmt->bindParam(':message', $message, PDO::PARAM_STR);
        $stmt->bindParam(':state', $state, PDO::PARAM_INT);

        return $stmt->execute();
    }





}