<?php

include_once 'config.php';

class customers_model
{
    public $link;

    public function __construct()
    {
        $db_connection = new dbConnection();
        $this->link = $db_connection->connect();
    }

    public function getCustomersRegisteredThisMonth() {
        $startOfMonth = date('Y-m-01 00:00:00');
        $endOfMonth = date('Y-m-t 23:59:59');

        $query = "SELECT COUNT(*) as total FROM customers WHERE datecreated BETWEEN :startOfMonth AND :endOfMonth";

        $stmt = $this->link->prepare($query);
        $stmt->bindParam(':startOfMonth', $startOfMonth);
        $stmt->bindParam(':endOfMonth', $endOfMonth);
        $stmt->execute();

        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        return (int)$result['total'];
    }

    public function getCustomersRegisteredLastMonth() {
        $startOfLastMonth = date('Y-m-01 00:00:00', strtotime('first day of last month'));
        $endOfLastMonth = date('Y-m-t 23:59:59', strtotime('last day of last month'));

        $query = "SELECT COUNT(*) as total  FROM customers  WHERE datecreated BETWEEN :start AND :end";

        $stmt = $this->link->prepare($query);
        $stmt->bindParam(':start', $startOfLastMonth);
        $stmt->bindParam(':end', $endOfLastMonth);
        $stmt->execute();

        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        return (int)$result['total'];
    }


    public function getCustomerStats() {
        $sql_current = "SELECT COUNT(*) as total FROM customers WHERE MONTH(datecreated) = MONTH(CURDATE()) AND YEAR(datecreated) = YEAR(CURDATE())";
        $sql_last = "SELECT COUNT(*) as total FROM customers WHERE MONTH(datecreated) = MONTH(CURDATE() - INTERVAL 1 MONTH) AND YEAR(datecreated) = YEAR(CURDATE() - INTERVAL 1 MONTH)";
        $sql_total = "SELECT COUNT(*) as total FROM customers";

        $current = $this->link->query($sql_current)->fetch(PDO::FETCH_ASSOC)['total'];
        $last = $this->link->query($sql_last)->fetch(PDO::FETCH_ASSOC)['total'];
        $total = $this->link->query($sql_total)->fetch(PDO::FETCH_ASSOC)['total'];

        return [
            'total' => $total,
            'current' => $current,
            'last' => $last
        ];
    }

    public function getcustomerId() {
        $query = "SELECT 
                    MAX(customer_id) AS current
                FROM 
                    customers";
        $stmt = $this->link->prepare($query);
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
    
        // Check if the 'current' value is NULL, meaning no records in the table
        if ($row['current'] === null) {
            $id = 8000; // Set default value if no customer ID is found
        } else {
            $id = $row['current'] + 1;
        }
    
        return $id;
    }

    public function checkEmailExists($email) {
        $query = "SELECT COUNT(*) FROM customers WHERE email = :email";
        $stmt = $this->link->prepare($query);
        $stmt->bindParam(':email', $email, PDO::PARAM_STR);
        $stmt->execute();
        return $stmt->fetchColumn() > 0;
    }
    public function getNextCustomerId(){
        $sql = "SELECT MAX(customer_id) AS max_id FROM customers";
        $stmt = $this->link->prepare($sql);
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($row && $row['max_id'] !== null) {
            return intval($row['max_id']) + 1;
        } else {
            return 8000; // Starting ID when table is empty
        }
    }

    public function checkphoneExists($phone_number) {
        $query = "SELECT COUNT(*) FROM customers WHERE contact_number = :phone_number";
        $stmt = $this->link->prepare($query);
        $stmt->bindParam(':phone_number', $phone_number, PDO::PARAM_STR);
        $stmt->execute();
        return $stmt->fetchColumn() > 0;
    }
    
    public function checkWhatsaAppNumberExists($phone_number) {
        $query = "SELECT COUNT(*) FROM customers WHERE whatsapp_number = :phone_number AND whatsapp_number IS NOT NULL";
        $stmt = $this->link->prepare($query);
        $stmt->bindParam(':phone_number', $phone_number, PDO::PARAM_STR);
        $stmt->execute();
        return $stmt->fetchColumn() > 0;
    }

    public function getActiveInvoicesByCustomer($customer_id) {
        $sql = "SELECT
                cc.customer_name,
                so.order_number AS invoice_number,
                so.total_bill,
                so.orderdate AS invoice_date,
                cc.contact_number AS customer_contact,
                so.subtotal,
                so.total_tax AS vat,
                so.total_bill AS grand_total,
                IFNULL(SUM(p.amount_paid), 0) AS total_paid,
                (so.total_bill - IFNULL(SUM(p.amount_paid), 0)) AS balance
            FROM 
                saved_orders so
            LEFT JOIN
                customers cc ON so.customer = cc.customer_id 
            LEFT JOIN
                invoice_payments p ON p.invoice_number = so.order_number AND p.payment_state = 1
            WHERE 
                so.customer = :customer 
                AND so.order_state IN(0,1)
            GROUP BY
                so.order_number";
                    
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':customer', $customer_id, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function phoneExists($phone, $excludeCustomerId) {
        $sql = "SELECT COUNT(*) FROM customers WHERE contact_number = :phone AND customer_id != :id";
        $stmt = $this->link->prepare($sql);
        $stmt->execute(['phone' => $phone, 'id' => $excludeCustomerId]);
        return $stmt->fetchColumn() > 0;
    }

    public function checkWhatappNumber($whatsapp_number, $excludeCustomerId) {
        $sql = "SELECT COUNT(*) FROM customers WHERE whatsapp_number = :phone AND customer_id != :id AND whatsapp_number IS NOT NULL";
        $stmt = $this->link->prepare($sql);
        $stmt->execute(['phone' => $whatsapp_number, 'id' => $excludeCustomerId]);
        return $stmt->fetchColumn() > 0;
    }

    public function emailExists($email, $excludeCustomerId) {
        $sql = "SELECT COUNT(*) FROM customers WHERE email = :email AND customer_id != :id";
        $stmt = $this->link->prepare($sql);
        $stmt->execute(['email' => $email, 'id' => $excludeCustomerId]);
        return $stmt->fetchColumn() > 0;
    }

    public function updateCustomer($data) {
        $sql = "UPDATE customers SET  customer_name = :customer_name, contact_number = :contact_number, whatsapp_number = :whatsapp_number, email = :email, gender = :gender, customer_address = :customer_address, location = :location, type_customer = :type_customer, nationality = :nationality, customer_state = :customer_state WHERE customer_id = :customer_id";

        $stmt = $this->link->prepare($sql);

        return $stmt->execute([
            ':customer_name'     => $data['customer_name'],
            ':contact_number'    => $data['contact_number'],
            ':email'             => $data['email'],
            ':gender'            => $data['gender'],
            ':location'          => $data['location'],
            ':customer_address'  => $data['cust_address'],
            ':nationality'       => $data['nationality'],
            ':type_customer'     => $data['type_customer'],
            ':customer_state'    => $data['customer_state'],
            ':customer_id'       => $data['customer_id'],
            ':whatsapp_number'   => $data['whatsapp_number']
        ]);
    }




    

    public function getCustomers($types = [], $state = null) {

        $sql = "SELECT
                    cc.customer_id,
                    cc.customer_name,
                    cc.type_customer,
                    cc.contact_number AS phone,
                    cc.whatsapp_number,
                    cc.email,
                    cc.gender,
                    cc.location  as current_location,
                    cc.customer_address,
                    cc.nationality,
                    c.country_name as nationality_name,
                    cc.customer_state,
                    cc.whatsapp_number,
                    ll.location_name AS location,
                    dd.district,
                    cc.type_customer as type,

                    COALESCE(inv.total_invoiced, 0) AS total_invoiced,
                    COALESCE(pay.total_paid, 0) AS total_paid,
                    (COALESCE(inv.total_invoiced, 0) - COALESCE(pay.total_paid, 0)) AS outstanding_balance

                FROM customers cc

                LEFT JOIN locations ll 
                    ON cc.location = ll.loc_id

                LEFT JOIN districts dd 
                    ON ll.loc_dist = dd.id
                LEFT JOIN
                    countries c ON cc.nationality = c.id

                /* ---- TOTAL INVOICED PER CUSTOMER ---- */
                LEFT JOIN (
                    SELECT
                        customer,
                        SUM(total_bill) AS total_invoiced
                    FROM saved_orders
                    WHERE order_state IN (0,1,2)
                    GROUP BY customer
                ) inv ON inv.customer = cc.customer_id

                /* ---- TOTAL PAID PER CUSTOMER ---- */
                LEFT JOIN (
                    SELECT
                        so.customer,
                        SUM(p.amount_paid) AS total_paid
                    FROM invoice_payments p
                    INNER JOIN saved_orders so 
                        ON so.order_number = p.invoice_number
                    GROUP BY so.customer
                ) pay ON pay.customer = cc.customer_id";

        // ------------------------------------
        // Dynamic WHERE conditions
        // ------------------------------------
        $conditions = [];
        $params = [];

        // Filter by customer types
        if (!empty($types)) {
            $placeholders = implode(",", array_fill(0, count($types), "?"));
            $conditions[] = "cc.type_customer IN ($placeholders)";
            $params = array_merge($params, $types);
        }

        // 🔥 UPDATED: customer_state filter (single OR multiple)
        if ($state !== null) {
            if (is_array($state)) {
                $placeholders = implode(",", array_fill(0, count($state), "?"));
                $conditions[] = "cc.customer_state IN ($placeholders)";
                $params = array_merge($params, $state);
            } else {
                $conditions[] = "cc.customer_state = ?";
                $params[] = $state;
            }
        }

        // Apply WHERE clause if needed
        if (!empty($conditions)) {
            $sql .= " WHERE " . implode(" AND ", $conditions);
        }

        // Grouping & ordering
        $sql .= "
            GROUP BY 
                cc.customer_id,
                cc.customer_name,
                cc.type_customer,
                cc.contact_number,
                cc.email,
                ll.location_name,
                dd.district
            ORDER BY cc.customer_name ASC
        ";

        $stmt = $this->link->prepare($sql);
        $stmt->execute($params);

        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function addCustomer($data){
        $sql = "INSERT INTO customers (customer_id, customer_name, contact_number, whatsapp_number, email, gender, customer_address, location,  type_customer, nationality, operator, datecreated, customer_state) 
                VALUES (:customer_id, :customer_name, :contact_number, :whatsapp_number, :email, :gender, :customer_address, :location, :type_customer, :nationality, :operator, :datecreated, :customer_state)";

        $stmt = $this->link->prepare($sql);

        $stmt->bindParam(':customer_id',     $data['customer_id'], PDO::PARAM_INT);
        $stmt->bindParam(':customer_name',   $data['customer_name'], PDO::PARAM_STR);
        $stmt->bindParam(':contact_number',  $data['contact_number'], PDO::PARAM_STR);
        $stmt->bindParam(':whatsapp_number', $data['whatsapp_number'], PDO::PARAM_STR);
        $stmt->bindParam(':email',           $data['email'], PDO::PARAM_STR);
        $stmt->bindParam(':gender',          $data['gender'], PDO::PARAM_STR);
        $stmt->bindParam(':customer_address',$data['customer_address'], PDO::PARAM_STR);
        $stmt->bindParam(':location',        $data['location'], PDO::PARAM_STR);
        $stmt->bindParam(':type_customer',   $data['type_customer'], PDO::PARAM_STR);
        $stmt->bindParam(':nationality',     $data['nationality'], PDO::PARAM_INT);
        $stmt->bindParam(':operator',        $data['operator'], PDO::PARAM_INT);
        $stmt->bindParam(':datecreated',     $data['datecreated'], PDO::PARAM_STR);
        $stmt->bindParam(':customer_state',  $data['customer_state'], PDO::PARAM_INT);

        if ($stmt->execute()) {
            return true;
        } else {
            return false;
        }
    }

    public function getCustomersFiltered($types = [], $state = null) {
        $sql = "SELECT 
                    cc.customer_id,
                    cc.customer_name,
                    cc.type_customer,
                    cc.contact_number AS phone,
                    cc.email,
                    ll.location_name AS location,
                    dd.district,
                    
                    COALESCE(SUM(so_total.total_bill), 0) AS total_invoiced,
                    COALESCE(SUM(pmt.amount_paid), 0) AS total_paid,
                    (COALESCE(SUM(so_total.total_bill), 0) - COALESCE(SUM(pmt.amount_paid), 0)) AS outstanding_balance
                FROM 
                    customers cc
                LEFT JOIN 
                    locations ll ON cc.location = ll.loc_id
                LEFT JOIN 
                    districts dd ON ll.loc_dist = dd.id
                LEFT JOIN 
                    saved_orders so_total ON so_total.customer = cc.customer_id
                    AND so_total.order_state IN (0,1,2)
                LEFT JOIN 
                    invoice_payments pmt ON pmt.invoice_number = so_total.order_number
        ";

        // ----------------------------------------------------
        // BUILD DYNAMIC WHERE CLAUSE
        // ----------------------------------------------------
        $conditions = [];
        $params = [];

        // Filter by customer types
        if (!empty($types)) {
            $placeholders = implode(",", array_fill(0, count($types), "?"));
            $conditions[] = "cc.type_customer IN ($placeholders)";
            $params = array_merge($params, $types);
        }

        // Filter by customer_state
        if ($state !== null) {
            $conditions[] = "cc.customer_state = ?";
            $params[] = $state;
        }

        // Add WHERE if any condition exists
        if (!empty($conditions)) {
            $sql .= " WHERE " . implode(" AND ", $conditions);
        }

        // Group and order
        $sql .= " 
            GROUP BY cc.customer_id, cc.customer_name, cc.type_customer, 
                    cc.contact_number, cc.email, ll.location_name, dd.district
            ORDER BY cc.customer_name ASC
        ";

        $stmt = $this->link->prepare($sql);
        $stmt->execute($params);

        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getCustomerDetails($customer_id) {
        $query = "SELECT 
                    cc.customer_id,
                    cc.customer_name,
                    cc.customer_address,
                    COALESCE(emp.fullname,'Super User') as operator,
                    cc.customer_state,
                    cc.email,
                    cc.contact_number as phone_number,
                    cc.whatsapp_number,
                    cc.gender,
                    ll.location_name,
                    cc.datecreated,
                    c.country_name,
                    c.nationality,
                    cc.nationality as current_nationality,
                    cc.location,
                    cc.whatsapp_number,
                    cc.type_customer,
                    cc.nationality as nationality_id,
                    c.nationality as nationality_name,
                    CASE
                        WHEN cc.customer_state = 1 THEN 'Active Customer'
                        WHEN cc.customer_state = 0  THEN 'Suspended Customer'
                    END as customer_status
                FROM 
                    customers cc
                LEFT JOIN
                    employees emp ON cc.operator = emp.employee_code
                LEFT JOIN
                    locations ll ON cc.location = ll.loc_id 
                LEFT JOIN
                    countries c ON cc.nationality = c.id
                WHERE 
                    cc.customer_id = :customer_id
                    ";

        $stmt = $this->link->prepare($query);
        $stmt->bindParam(':customer_id', $customer_id, PDO::PARAM_STR);
        $stmt->execute();
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    public function getLastTransactionDate($customer_id) {
        // Query for latest payment
        $paymentQuery = "SELECT 
                            datecaptured, 
                            id AS transaction_id,
                            receipt_number,
                            CASE
                                WHEN payment_type = 'Cash Sale' THEN 'cash_sale'
                                WHEN payment_type = 'Customer Order' THEN 'customer_order'
                                ELSE ''
                            END AS source
                        FROM 
                            payments 
                        WHERE 
                            customer = :customer_id 
                        ORDER BY datecaptured DESC 
                        LIMIT 1";

        // Query for latest saved order
        $orderQuery = "SELECT 
                            datecaptured, 
                            order_number AS transaction_id,
                            'saved_orders' AS source
                    FROM 
                            saved_orders 
                    WHERE 
                            customer = :customer_id 
                    ORDER BY datecaptured DESC 
                    LIMIT 1";

        // Prepare and execute both queries
        $stmt1 = $this->link->prepare($paymentQuery);
        $stmt1->execute(['customer_id' => $customer_id]);
        $payment = $stmt1->fetch(PDO::FETCH_ASSOC);

        $stmt2 = $this->link->prepare($orderQuery);
        $stmt2->execute(['customer_id' => $customer_id]);
        $order = $stmt2->fetch(PDO::FETCH_ASSOC);

        // Compare and return the more recent transaction
        if ($payment && $order) {
            return (strtotime($payment['datecaptured']) > strtotime($order['datecaptured'])) 
                ? $payment 
                : $order;
        } elseif ($payment) {
            return $payment;
        } elseif ($order) {
            return $order;
        } else {
            return [
                'datecaptured'    => null,
                'transaction_id'  => null,
                'source'          => 'No transactions yet',
                'receipt_number'  => null
            ];
        }
    }


    public function getCustomerTransactions($customer_id, $from_date = null, $to_date = null, $filter_type = 'all') {

        $dateFilterPayments = '';
        $dateFilterInvoices = '';

        // Apply date filters ONLY when not filtering balances
        if ($filter_type !== 'balances' && !empty($from_date) && !empty($to_date)) {
            $dateFilterPayments = " AND p.payment_date BETWEEN :from_date AND :to_date ";
            $dateFilterInvoices = " AND so.orderdate BETWEEN :from_date AND :to_date ";
        }

        // Balance-only invoice restriction
        $invoiceBalanceFilter = '';
        if ($filter_type === 'balances') {
            $invoiceBalanceFilter = "
                AND so.order_state IN (0,1)
                AND so.total_bill >
                    (
                        SELECT COALESCE(SUM(ip2.amount_paid), 0)
                        FROM invoice_payments ip2
                        WHERE ip2.invoice_number = so.order_number
                        AND ip2.payment_state = 1
                    )
            ";
        }

        // Exclude cash sales when filtering balances
        $cashSaleFilter = ($filter_type === 'balances') ? ' AND 1=0 ' : '';

        $query = "SELECT
                    'invoice' AS source,
                    so.order_state AS state,
                    so.orderdate AS actual_date,
                    so.datecaptured AS date_captured,
                    so.order_number AS transaction_no,
                    COALESCE(emp.fullname, 'System') AS operator,
                    so.order_number AS reference_number,
                    'Customer Invoice' AS activity,
                    so.total_bill AS sales_amount,
                    0 AS payments_amount,
                    so.order_number AS group_key,
                    CASE 
                        WHEN so.order_state = 0 THEN 'Unpaid Invoice'
                        WHEN so.order_state = 1 THEN 'Partially Paid'
                        WHEN so.order_state = 2 THEN 'Completed Payment'
                    END AS transaction_status

                FROM 
                    saved_orders so
                LEFT JOIN 
                    employees emp ON so.operator = emp.employee_code
                WHERE
                    so.customer = :customer_id
                    AND so.order_state IN (0,1,2)
                    $invoiceBalanceFilter
                    $dateFilterInvoices

            UNION ALL
                /* ============================
                CUSTOMER INVOICES (SALES)
                ============================ */
                /* ============================
                CASH SALES
                ============================ */
                SELECT
                    'cash_sale' AS source,
                    p.payment_state AS state,
                    p.payment_date AS actual_date,
                    p.datecaptured AS date_captured,
                    p.receipt_number AS transaction_no,
                    COALESCE(e.fullname, 'Default Account') AS operator,
                    p.receipt_number AS reference_number,
                    'Cash Sale' AS activity,
                    p.amount_paid AS sales_amount,
                    p.amount_paid AS payments_amount,
                    p.receipt_number AS group_key,
                    CASE
                        WHEN p.payment_state = 1 THEN 'Completed Payment'
                        WHEN p.payment_state = 0 THEN 'Cancelled Payment'
                    END AS transaction_status

                FROM 
                    payments p
                LEFT JOIN 
                    employees e ON p.operator = e.employee_code
                WHERE
                    p.customer = :customer_id
                    AND p.payment_type = 'Cash Sale'
                    AND p.payment_state = 1
                    $cashSaleFilter
                    $dateFilterPayments

            UNION ALL

                /* ============================
                INVOICE PAYMENTS (ALLOCATIONS)
                ============================ */
                SELECT
                    'invoice_payment' AS source,
                    ip.payment_state AS state,
                    p.payment_date AS actual_date,
                    p.datecaptured AS date_captured,
                    so.order_number AS transaction_no,
                    COALESCE(e.fullname, 'System') AS operator,
                    p.receipt_number AS reference_number,
                    'Invoice Payment' AS activity,
                    0 AS sales_amount,
                    ip.amount_paid AS payments_amount,
                    so.order_number AS group_key,
                    CASE
                        WHEN p.payment_state = 1 THEN 'Completed Payment'
                        WHEN p.payment_state = 0 THEN 'Cancelled Payment'
                    END AS transaction_status

                FROM 
                    invoice_payments ip
                INNER JOIN 
                    payments p ON ip.payment_id = p.id
                INNER JOIN 
                    saved_orders so ON ip.invoice_number = so.order_number
                LEFT JOIN 
                    employees e ON p.operator = e.employee_code
                WHERE
                    p.customer = :customer_id
                    AND p.payment_type = 'Order Payment'
                    AND ip.payment_state = 1
                    $dateFilterPayments

                ORDER BY
                    group_key ASC,
                    actual_date ASC,
                    date_captured ASC";

        $stmt = $this->link->prepare($query);

        $params = ['customer_id' => $customer_id];

        if ($filter_type !== 'balances' && !empty($from_date) && !empty($to_date)) {
            $params['from_date'] = $from_date;
            $params['to_date']   = $to_date;
        }

        $stmt->execute($params);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }


    public function hasTransactions($customer_id) {
        // Check transactions in both payments and saved_orders
        $query = "SELECT 
                (
                    (SELECT COUNT(*) FROM payments WHERE customer = :customer_id) +
                    (SELECT COUNT(*) FROM saved_orders WHERE customer = :customer_id)
                ) AS transaction_count
        ";
    
        $stmt = $this->link->prepare($query);
        $stmt->execute(['customer_id' => $customer_id]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
    
        // Return true if any transactions exist in either table
        return $result['transaction_count'] > 0;
    }
    
    public function getCustomerName($customer_id) {
        $query = "SELECT customer_name FROM customers WHERE customer_id = :customer_id LIMIT 1";
        
        $stmt = $this->link->prepare($query);
        $stmt->execute(['customer_id' => $customer_id]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        
        return $result ? $result['customer_name'] : 'Unknown Customer';
    }

    public function denyCustomer($customer_id) {
        $sql = "UPDATE customers SET customer_state = 1, type_customer ='Walk-in'  WHERE customer_id = :customer_id";

        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':customer_id', $customer_id, PDO::PARAM_INT);

        return $stmt->execute();
    }

    public function approveCustomer($customer_id) {
        $sql = "UPDATE customers SET customer_state = 1 WHERE customer_id = :customer_id";

        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':customer_id', $customer_id, PDO::PARAM_INT);

        return $stmt->execute();
    }

    
    
    
    
    
}