<?php

include_once 'config.php';

class stock_model
{
    public $link;

    public function __construct()
    {
        $db_connection = new dbConnection();
        $this->link = $db_connection->connect();
    }

    public function updateProductImage($product_id, $image_name) {
        $sql = "UPDATE products SET image = :image WHERE product_id = :product_id";
        $stmt = $this->link->prepare($sql);
        return $stmt->execute([
            ':image' => $image_name,
            ':product_id' => $product_id
        ]);
    }

    public function getProductsByCompany($company_id) {
        $sql = "SELECT 
                    tpp.product_id, 
                    p.product_name,
                    pc.categoryname
                FROM 
                    third_party_products tpp 
                LEFT JOIN 
                    products p ON tpp.product_id = p.product_id
                LEFT JOIN 
                    product_categories pc ON p.category = pc.category_id
                WHERE
                	tpp.company_id = :company_id
                ORDER BY 
                    p.product_name ASC";
        $stmt = $this->link->prepare($sql);
        $stmt->execute([
            ':company_id' => $company_id
        ]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function get_all_items($company) {
        $sql = "SELECT
                    tpp.product_id,
                    p.product_code,
                    p.product_name,
                    tpp.quantity
                FROM 
                    third_party_products tpp
                LEFT JOIN 
                    products p ON tpp.product_id = p.product_id
                WHERE
                    tpp.company_id = :company_id
                ORDER BY 
                    product_name ASC";
        $stmt = $this->link->prepare($sql);
        $stmt->execute([
            ':company_id' => $company
        ]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    

    public function getAllProducts() {
        $sql = "SELECT 
                    p.product_id,
                    p.product_code,
                    p.product_name,
                    p.product_state,
                    c.categoryname,
                    u.unitname,
                    p.selling_price,
                    p.min_order_level,
                    IF(p.product_state = 1, 'Active Product', 'Deactivated Product') as product_status,
                    IFNULL((
                        SELECT SUM(bs.quantity)
                        FROM branch_products bs
                        WHERE bs.product_id = p.product_id
                    ), 0) AS total_branch_stock,

                    -- Total stock in warehouses
                    IFNULL((
                        SELECT SUM(ws.quantity)
                        FROM warehouse_products ws
                        WHERE ws.product_id = p.product_id
                    ), 0) AS total_warehouse_stock,

                    (
                        IFNULL((
                            SELECT SUM(bs.quantity)
                            FROM branch_products bs
                            WHERE bs.product_id = p.product_id
                        ), 0) +
                        IFNULL((
                            SELECT SUM(ws.quantity)
                            FROM warehouse_products ws
                            WHERE ws.product_id = p.product_id
                        ), 0) +
                        IFNULL((
                            SELECT SUM(tps.quantity)
                            FROM third_party_products tps
                            WHERE tps.product_id = p.product_id
                        ), 0)
                    ) AS overall_stock,

                    -- Total in third-party warehouse
                IFNULL((
                    SELECT SUM(tps.quantity)
                    FROM third_party_products tps
                    WHERE tps.product_id = p.product_id
                ), 0) AS total_third_party_stock
                FROM 
                    products p
                LEFT JOIN
                    product_categories c ON p.category = c.category_id
                LEFT JOIN
                    product_units u ON p.unit = u.unit_id
                ORDER BY
                    p.product_state DESC, p.product_name ASC";
    
        $stmt = $this->link->prepare($sql);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getStockByProductId($product_id) {
        $sql = "SELECT 
                b.branch_name AS name, 
                'branch' AS type, 
                bp.quantity, 
                p.product_name,
                bp.datecaptured AS date_added,
                bp.min_level as min_order_level,
                p.selling_price
            FROM 
                branch_products bp
            JOIN 
                branches b ON bp.branch_id = b.branch_id
            LEFT JOIN
                products p ON p.product_id = bp.product_id
            WHERE 
                bp.product_id = ?

            UNION ALL

            SELECT 
                w.warehouse_name AS name, 
                'warehouse' AS type, 
                wp.quantity, 
                p.product_name,
                wp.datecaptured AS date_added,
                wp.min_level as min_order_levels,
                p.selling_price
            FROM 
                warehouse_products wp
            JOIN 
                warehouses w ON wp.warehouse_id = w.warehouse_id
            LEFT JOIN
                products p ON p.product_id = wp.product_id
            WHERE 
                wp.product_id = ?
        ";

        $stmt = $this->link->prepare($sql);
        // Bind the same product_id for both placeholders in the UNION
        $stmt->execute([$product_id, $product_id]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }



    public function getproductsBelowMinStockLevel(){
        $sql = "SELECT 
                p.product_name,
                p.min_order_level,
                p.product_code,
                c.categoryname,
                u.unitname,
                p.product_state,
                p.product_id,
                IF(p.product_state = 1, 'Active Product', 'Deactivated Product') as product_status,
                (
                    IFNULL((
                        SELECT SUM(bs.quantity)
                        FROM branch_products bs
                        WHERE bs.product_id = p.product_id
                    ), 0) +
                    IFNULL((
                        SELECT SUM(ws.quantity)
                        FROM warehouse_products ws
                        WHERE ws.product_id = p.product_id
                    ), 0)
                ) AS overall_stock

            FROM
                products p
            LEFT JOIN
                product_categories c ON p.category = c.category_id
            LEFT JOIN
                product_units u ON p.unit = u.unit_id
            WHERE
                p.min_order_level > (
                    IFNULL((
                        SELECT SUM(bs.quantity)
                        FROM branch_products bs
                        WHERE bs.product_id = p.product_id
                    ), 0) +
                    IFNULL((
                        SELECT SUM(ws.quantity)
                        FROM warehouse_products ws
                        WHERE ws.product_id = p.product_id
                    ), 0)
                )
                AND 
                    (
                    IFNULL((
                        SELECT SUM(bs.quantity)
                        FROM branch_products bs
                        WHERE bs.product_id = p.product_id
                    ), 0) +
                    IFNULL((
                        SELECT SUM(ws.quantity)
                        FROM warehouse_products ws
                        WHERE ws.product_id = p.product_id
                    ), 0)
                ) > 0
                AND p.product_state = 1
            ORDER BY
                overall_stock DESC, p.product_name ASC";

        $smtp = $this->link->prepare($sql);
        $smtp->execute();
        $data = $smtp->fetchAll(PDO::FETCH_ASSOC);
        return $data;
    }


    public function getproductsoutofStock(){
        $sql = "SELECT 
                p.product_name,
                p.min_order_level,
                p.product_code,
                c.categoryname,
                u.unitname,
                p.product_state,
                p.product_id,
                IF(p.product_state = 1, 'Active Product', 'Deactivated Product') as product_status,
                (
                    IFNULL((
                        SELECT SUM(bs.quantity)
                        FROM branch_products bs
                        WHERE bs.product_id = p.product_id
                    ), 0) +
                    IFNULL((
                        SELECT SUM(ws.quantity)
                        FROM warehouse_products ws
                        WHERE ws.product_id = p.product_id
                    ), 0)
                ) AS overall_stock

            FROM
                products p
            LEFT JOIN
                product_categories c ON p.category = c.category_id
            LEFT JOIN
                product_units u ON p.unit = u.unit_id
            WHERE
                (
                    IFNULL((
                        SELECT SUM(bs.quantity)
                        FROM branch_products bs
                        WHERE bs.product_id = p.product_id
                    ), 0) +
                    IFNULL((
                        SELECT SUM(ws.quantity)
                        FROM warehouse_products ws
                        WHERE ws.product_id = p.product_id
                    ), 0)
                ) = 0
                AND p.product_state = 1
            ORDER BY
                overall_stock DESC, p.product_name ASC";

        $smtp = $this->link->prepare($sql);
        $smtp->execute();
        $data = $smtp->fetchAll(PDO::FETCH_ASSOC);
        return $data;
    }

    

    public function getProductDetails($product_code) {
        $sql = "SELECT
            p.product_id,
            p.product_code,
            p.product_name,
            pc.categoryname,
            pu.unitname,
            p.category,
            p.unit,
            p.min_order_level,
            p.product_state,
            p.datecreated,
            COALESCE(emp.fullname, 'Super User') as operator,
            IF(p.product_state = 1, 'Active Product', 'Deactivated Product') as product_status,
            p.image
            FROM 
                products p
            LEFT JOIN
                product_categories pc ON p.category = pc.category_id
            LEFT JOIN
                product_units pu ON p.unit = pu.unit_id
            LEFT JOIN
                employees emp ON p.operator = emp.employee_code
            WHERE 
                p.product_code = ?";
        
        $stmt = $this->link->prepare($sql);
        $stmt->execute([$product_code]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
    
        // Check if image is null, empty, or file doesn't exist
        if ($result) {
            if (!isset($result['image']) || empty($result['image']) || !file_exists("images/products/".$result['image'])) {
                $result['image'] = "default.png"; // use a default image
            }
        }
    
        return $result;
    }

    

    public function getProductDetailsbyID2($product_id) {
        $sql = "SELECT
            p.product_id,
            p.product_code,
            p.product_name,
            pc.categoryname,
            pu.unitname,
            p.category,
            p.unit,
            p.min_order_level,
            p.product_state,
            p.selling_price,
            p.datecreated,
            COALESCE(emp.fullname, 'Super User') as operator,
            IF(p.product_state = 1, 'Active Product', 'Deactivated Product') as product_status,
            p.image
            FROM 
                products p
            LEFT JOIN
                product_categories pc ON p.category = pc.category_id
            LEFT JOIN
                product_units pu ON p.unit = pu.unit_id
            LEFT JOIN
                employees emp ON p.operator = emp.employee_code
            WHERE 
                p.product_id = ?";
        
        $stmt = $this->link->prepare($sql);
        $stmt->execute([$product_id]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
    
        // Check if image is null, empty, or file doesn't exist
        if ($result) {
            if (!isset($result['image']) || empty($result['image']) || !file_exists("images/products/".$result['image'])) {
                $result['image'] = "default.png"; // use a default image
            }
        }
    
        return $result;
    }

    public function getProductDetailsbyID($product_id) {
        $sql = "SELECT
            p.product_id,
            p.product_code,
            p.product_name,
            pc.categoryname,
            pu.unitname,
            p.category,
            p.unit,
            p.selling_price,
            p.min_order_level,
            p.product_state,
            p.datecreated,
            COALESCE(emp.fullname, 'Super User') as operator,
            IF(p.product_state = 1, 'Active Product', 'Deactivated Product') as product_status,
            p.image
            FROM 
                products p
            LEFT JOIN
                product_categories pc ON p.category = pc.category_id
            LEFT JOIN
                product_units pu ON p.unit = pu.unit_id
            LEFT JOIN
                employees emp ON p.operator = emp.employee_code
            WHERE 
                p.product_id = ?";
        
        $stmt = $this->link->prepare($sql);
        $stmt->execute([$product_id]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
    
        // Check if image is null, empty, or file doesn't exist
        if ($result) {
            if (!isset($result['image']) || empty($result['image']) || !file_exists("images/".$result['image'])) {
                $result['image'] = "default.png"; // use a default image
            }
        }
    
        return $result;
    }

    public function productCodeExistsAlready($code, $excludeId = null) {
        $query = "SELECT COUNT(*) as count FROM products WHERE product_code = ? ";
        $params = [$code];
        
        if ($excludeId) {
            $query .= "AND product_id != ?";
            $params[] = $excludeId;
        }
    
        $stmt = $this->link->prepare($query);
        $stmt->execute($params);
        $result = $stmt->fetch();
        return $result['count'] > 0;
    }
    
    public function productNameExistsAlready($product_name, $product_id) {
        $query = "SELECT COUNT(*) as count FROM products WHERE product_name = :product_name AND product_id != :product_id";
        $smtp = $this->link->prepare($query);
        $smtp->bindParam(':product_name', $product_name, PDO::PARAM_STR);
        $smtp->bindParam(':product_id', $product_id, PDO::PARAM_INT);
    
        if ($smtp->execute()) {
            $result = $smtp->fetch(PDO::FETCH_ASSOC);
            return $result['count'] > 0; // Returns true if duplicate exists
        }
    
        return false; // In case of query failure
    }   
    

    public function getCategories(){
        $sql = "SELECT *,
            IF(category_state = 1, 'Active Category', 'Disabled Category') as category_status 
        FROM 
            product_categories
        ORDER BY 
            category_state DESC, categoryname ASC";
        $stmt = $this->link->prepare($sql);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getCategoryDetails($category_id){
        try {
            $query = "SELECT * FROM product_categories WHERE category_id = ?";
            $stmt = $this->link->prepare($query);
            $stmt->execute([$category_id]);
            return $stmt->fetch(PDO::FETCH_ASSOC); // returns single category as associative array
        } catch (PDOException $e) {
            // Log error or handle as needed
            return false;
        }
    }

    public function getunitDetails($unit_id){
        try {
            $query = "SELECT * FROM product_units WHERE unit_id = ?";
            $stmt = $this->link->prepare($query);
            $stmt->execute([$unit_id]);
            return $stmt->fetch(PDO::FETCH_ASSOC); // returns single unit as associative array
        } catch (PDOException $e) {
            // Log error or handle as needed
            return false;
        }
    }
    
    
    public function getUnits(){
        $sql = "SELECT *,
            IF(unit_state = 1, 'Active Unit', 'Disabled Unit') as unit_status   
        FROM 
            product_units";
        $stmt = $this->link->prepare($sql);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function addCategory($categoryname, $state){
        $sql = "INSERT into product_categories(categoryname, category_state)VALUES(:categoryname, :state)";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':categoryname',$categoryname, PDO::PARAM_STR);
        $stmt->bindParam(':state',$state, PDO::PARAM_INT);

        if ($stmt->execute()) {
            return true;
        } else {
            return false;
        }


    }
    
    public function addUnit($unitname, $state){
        $sql = "INSERT into product_units(unitname, unit_state)VALUES(:unitname, :state)";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':unitname',$unitname, PDO::PARAM_STR);
        $stmt->bindParam(':state',$state, PDO::PARAM_INT);

        if ($stmt->execute()) {
            return true;
        } else {
            return false;
        }


    }

    public function updateUnit($unitname, $id){
        $sql = "UPDATE product_units SET unitname = :unitname WHERE unit_id = :id";

        $stmt =$this->link->prepare($sql);
        $stmt->bindParam(':unitname', $unitname, PDO::PARAM_STR);
        $stmt->bindParam(':id', $id, PDO::PARAM_INT);

        if($stmt->execute()){
            return true;
        }else{
            return false;
        }
    }

    public function updateCategory($categoryname, $id){
        $sql = "UPDATE product_categories SET categoryname = :categoryname WHERE category_id = :id";

        $stmt =$this->link->prepare($sql);
        $stmt->bindParam(':categoryname', $categoryname, PDO::PARAM_STR);
        $stmt->bindParam(':id', $id, PDO::PARAM_INT);

        if($stmt->execute()){
            return true;
        }else{
            return false;
        }
    }

    public function disableCategory($id, $state){
        $sql = "UPDATE product_categories SET category_state = :state WHERE category_id = :id";

        $stmt =$this->link->prepare($sql);
        $stmt->bindParam(':state', $state, PDO::PARAM_STR);
        $stmt->bindParam(':id', $id, PDO::PARAM_INT);

        if($stmt->execute()){
            return true;
        }else{
            return false;
        }
    }

    public function getActiveProductsByCategory($category_id) {
        $sql = "SELECT * FROM products WHERE category = :category_id AND product_state = 1";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':category_id', $category_id, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getActiveProductsByUnit($unit_id) {
        $sql = "SELECT * FROM products WHERE unit = :unit_id AND product_state = 1";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':unit_id', $unit_id, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    

    public function enableCategory($id, $state){
        $sql = "UPDATE product_categories SET category_state = :state WHERE category_id = :id";

        $stmt =$this->link->prepare($sql);
        $stmt->bindParam(':state', $state, PDO::PARAM_STR);
        $stmt->bindParam(':id', $id, PDO::PARAM_INT);

        if($stmt->execute()){
            return true;
        }else{
            return false;
        }
    }

    public function disableUnit($id,$state){
        $sql = "UPDATE product_units SET unit_state = :state WHERE unit_id = :id";

        $stmt =$this->link->prepare($sql);
        $stmt->bindParam(':state', $state, PDO::PARAM_STR);
        $stmt->bindParam(':id', $id, PDO::PARAM_INT);

        if($stmt->execute()){
            return true;
        }else{
            return false;
        }
    }

    public function enableUnit($id,$state){
        $sql = "UPDATE product_units SET unit_state = :state WHERE unit_id = :id";

        $stmt =$this->link->prepare($sql);
        $stmt->bindParam(':state', $state, PDO::PARAM_STR);
        $stmt->bindParam(':id', $id, PDO::PARAM_INT);

        if($stmt->execute()){
            return true;
        }else{
            return false;
        }
    }

    public function deleteCategory($category_id){
        try {
            $sql = "DELETE FROM product_categories WHERE category_id = :category_id";
            $stmt = $this->link->prepare($sql);
            $stmt->bindParam(':category_id', $category_id, PDO::PARAM_INT);
            return $stmt->execute();
        } catch (PDOException $e) {
            // Optional: log error if needed
            return false;
        }
    }

    public function deleteUnit($unit_id){
        try {
            $sql = "DELETE FROM product_units WHERE unit_id = :unit_id";
            $stmt = $this->link->prepare($sql);
            $stmt->bindParam(':unit_id', $unit_id, PDO::PARAM_INT);
            return $stmt->execute();
        } catch (PDOException $e) {
            // Optional: log error if needed
            return false;
        }
    }
    
    public function addProduct($product_code, $product_name, $selling_price, $min_order_level, $category, $unit, $operator, $product_state) {
        date_default_timezone_set('Africa/Blantyre'); // Adjusted to your timezone

        $datecreated = date('Y-m-d H:i:s');
        
        try {
            $sql = "INSERT INTO products (product_code, product_name, selling_price, category, unit, min_order_level, datecreated, operator, product_state)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)";
            $stmt = $this->link->prepare($sql);

            if ($stmt->execute([$product_code, $product_name, $selling_price, $category,  $unit, $min_order_level, $datecreated, $operator, $product_state])) {
                return $this->link->lastInsertId(); // ✅ Return newly inserted product ID
            } else {
                return false;
            }
        } catch (PDOException $e) {
            error_log("Add Product Failed: " . $e->getMessage());
            return false;
        }
    }

    

    public function productCodeExists($product_code) {
        $sql = "SELECT COUNT(*) FROM products WHERE product_code = ?";
        $stmt = $this->link->prepare($sql);
        $stmt->execute([$product_code]);
        return $stmt->fetchColumn() > 0;
    }
    
    public function productNameExists($product_name) {
        $sql = "SELECT COUNT(*) FROM products WHERE product_name = ?";
        $stmt = $this->link->prepare($sql);
        $stmt->execute([$product_name]);
        return $stmt->fetchColumn() > 0;
    }
    
    
    
    
    
    public function updateProduct($product_id, $product_code, $selling_price, $product_name, $min_order_level, $category, $unit, $product_state) {
        $query = "UPDATE products SET product_code = :product_code, selling_price =:selling_price, product_name = :product_name, min_order_level = :min_order_level, category = :category, unit = :unit, product_state = :product_state WHERE product_id = :product_id";
                  
        $stmt = $this->link->prepare($query);
        $stmt->bindParam(':product_code', $product_code);
        $stmt->bindParam(':product_name', $product_name);
        $stmt->bindParam(':selling_price', $selling_price);
        $stmt->bindParam(':min_order_level', $min_order_level);
        $stmt->bindParam(':category', $category);
        $stmt->bindParam(':unit', $unit);
        $stmt->bindParam(':product_state', $product_state);
        $stmt->bindParam(':product_id', $product_id);
    
        return $stmt->execute();
    }

    public function getWarehouses() {
        $stmt = $this->link->prepare("SELECT * FROM warehouses");
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    public function getBranches() {
        $stmt = $this->link->prepare("SELECT * FROM branches");
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getUnlinkedProducts($type, $id) {
        if ($type === 'warehouse') {
            $sql = "SELECT p.*,
                    pc.categoryname,
                    pu.unitname
                FROM 
                    products p
                LEFT JOIN 
                    product_categories pc ON p.category = pc.category_id
                LEFT JOIN
                    product_units pu ON p.unit = pu.unit_id
                    WHERE product_id NOT IN (
                        SELECT product_id FROM warehouse_products WHERE warehouse_id = ?
                    )";
        } elseif ($type === 'branch') {
            $sql = "SELECT pp.*,
                    pc.categoryname,
                    pu.unitname
                FROM 
                    products pp
                LEFT JOIN 
                    product_categories pc ON pp.category = pc.category_id
                LEFT JOIN
                    product_units pu ON pp.unit = pu.unit_id
                    WHERE product_id NOT IN (
                        SELECT product_id FROM branch_products WHERE branch_id = ?
                    )";
        } else {
            return [];
        }
    
        $stmt = $this->link->prepare($sql);
        $stmt->bindValue(1, $id, PDO::PARAM_INT); // Use bindValue with positional placeholder
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC); // Use PDO::FETCH_ASSOC instead of MYSQLI_ASSOC
    }
    
    public function linkSingleProductToLocation($type, $location, $product, $opening_stock, $quantity, $min_level, $datecaptured, $operator, $product_state ) {
        if ($type === 'warehouse') {
            $sql = "INSERT INTO warehouse_products (warehouse_id, product_id, opening_stock, quantity, min_level, datecaptured, operator, product_state) 
                    VALUES (:location, :product_id, :opening_stock, :quantity, :min_level, :datecaptured, :operator, :product_state)";
        } elseif ($type === 'branch') {
            $sql = "INSERT INTO branch_products (branch_id, product_id, opening_stock, quantity, min_level, datecaptured, operator, product_state) 
                     VALUES (:location, :product_id, :opening_stock, :quantity, :min_level, :datecaptured, :operator, :product_state)";
        } else {
            return false;
        }
    
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':location', $location, PDO::PARAM_INT);
        $stmt->bindParam(':product_id', $product, PDO::PARAM_INT);
        $stmt->bindParam(':opening_stock', $opening_stock, PDO::PARAM_INT);
        $stmt->bindParam(':quantity', $quantity, PDO::PARAM_INT);
        $stmt->bindParam(':min_level', $min_level, PDO::PARAM_INT);
        $stmt->bindParam(':datecaptured', $datecaptured, PDO::PARAM_STR);
        $stmt->bindParam(':operator', $operator, PDO::PARAM_INT);
        $stmt->bindParam(':product_state', $product_state, PDO::PARAM_INT);
        return $stmt->execute();
    }

    public function getProductQuantityAtDestination($product_id, $destination_id) {
        // First, try warehouse
        $sql = "SELECT quantity FROM warehouse_products WHERE warehouse_id = :destination_id AND product_id = :product_id";
        $stmt = $this->link->prepare($sql);
        $stmt->execute([
            'destination_id' => $destination_id,
            'product_id' => $product_id
        ]);
        $warehouseResult = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($warehouseResult) {
            return ['quantity' => floatval($warehouseResult['quantity'])];
        }

        // Try branch
        $sql = "SELECT quantity FROM branch_products WHERE branch_id = :destination_id AND product_id = :product_id";
        $stmt = $this->link->prepare($sql);
        $stmt->execute([
            'destination_id' => $destination_id,
            'product_id' => $product_id
        ]);
        $branchResult = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($branchResult) {
            return ['quantity' => floatval($branchResult['quantity'])];
        }

        // If not found in either
        return ['quantity' => 0];
    }


    public function orderNumberExists($orderNumber) {
        $query = "SELECT transfer_id FROM stock_transfers_final WHERE transfer_id = :order_number";

        $stmt = $this->link->prepare($query);
        $stmt->execute(['order_number' => $orderNumber]);

        return $stmt->rowCount() > 0;
    }

    public function getstockSheetforaproduct($warehouse_id, $product_id){
        $sql = "SELECT
                p.product_name,
                pc.categoryname,
                
                -- Show warehouse name from either transfer_from or transfer_to
                CASE 
                    WHEN w_from.warehouse_id = :warehouse_id THEN w_from.warehouse_name
                    WHEN w_to.warehouse_id = :warehouse_id THEN w_to.warehouse_name
                    ELSE NULL
                END AS warehouse_name,
                
                -- Show warehouse name from either transfer_from or transfer_to
                CASE 
                    WHEN w_from.warehouse_id = :warehouse_id THEN w_from.warehouse_type
                    WHEN w_to.warehouse_id = :warehouse_id THEN w_to.warehouse_type
                    ELSE NULL
                END AS warehouse_type,
                sti.qty_original as opening_stock,
                sti.qty_accepted as stock_added,
                sti.qty_final as closing_stock 

            FROM stock_transfer_items sti

            LEFT JOIN 
                products p ON sti.product_id = p.product_id
            LEFT JOIN 
                product_categories pc ON p.category = pc.category_id
            LEFT JOIN 
                warehouses w_from ON sti.transfer_from = w_from.warehouse_id
            LEFT JOIN 
                warehouses w_to ON sti.transfer_to = w_to.warehouse_id

            WHERE 
                sti.product_id = :product_id
                AND (:warehouse_id IN (sti.transfer_from, sti.transfer_to))
        ";

        $smtp = $this->link->prepare($sql);
        $smtp->bindParam(':warehouse_id', $warehouse_id, PDO::PARAM_INT);
        $smtp->bindParam(':product_id', $product_id, PDO::PARAM_INT);
        $smtp->execute();

        return $smtp->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getStockMovementHistory($warehouse_id, $product_id) {
        $sql = "SELECT
            sti.transfer_date AS transaction_date,
            p.product_name,
            pc.categoryname,
            sti.qty_accepted AS quantity,

            -- Direction relative to the current warehouse
            CASE 
                WHEN sti.transfer_to = :warehouse_id THEN 'IN'
                WHEN sti.transfer_from = :warehouse_id THEN 'OUT'
                ELSE 'N/A'
            END AS direction,

            -- FROM location name
            COALESCE(wf.warehouse_name, bf.branch_name, cf.company_name) AS from_location,

            -- TO location name
            COALESCE(wt.warehouse_name, bt.branch_name, ct.company_name) AS to_location,

            -- FROM location type
            CASE 
                WHEN wf.warehouse_id IS NOT NULL THEN 'Warehouse'
                WHEN bf.branch_id IS NOT NULL THEN 'Branch'
                WHEN cf.company_id IS NOT NULL THEN 'Third Party Company'
                ELSE 'Unknown'
            END AS from_type,

            -- TO location type
            CASE 
                WHEN wt.warehouse_id IS NOT NULL THEN 'Warehouse'
                WHEN bt.branch_id IS NOT NULL THEN 'Branch'
                WHEN ct.company_id IS NOT NULL THEN 'Third Party'
                ELSE 'Unknown'
            END AS to_type

        FROM stock_transfer_items sti

        LEFT JOIN products p ON sti.product_id = p.product_id
        LEFT JOIN product_categories pc ON p.category = pc.category_id

        -- FROM locations
        LEFT JOIN warehouses wf ON sti.transfer_from = wf.warehouse_id
        LEFT JOIN branches bf ON sti.transfer_from = bf.branch_id
        LEFT JOIN third_party_companies cf ON sti.transfer_from = cf.company_id

        -- TO locations
        LEFT JOIN warehouses wt ON sti.transfer_to = wt.warehouse_id
        LEFT JOIN branches bt ON sti.transfer_to = bt.branch_id
        LEFT JOIN third_party_companies ct ON sti.transfer_to = ct.company_id

        WHERE sti.product_id = :product_id
        AND (:warehouse_id IN (sti.transfer_from, sti.transfer_to))

        ORDER BY sti.transfer_date DESC


        ";

        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':warehouse_id', $warehouse_id, PDO::PARAM_INT);
        $stmt->bindParam(':product_id', $product_id, PDO::PARAM_INT);
        $stmt->execute();

        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }



    public function orderHasItems($orderNumber) {
        $query = "SELECT COUNT(*) FROM (SELECT id FROM stock_transfers_final WHERE transfer_id = :order_number) AS combined
        ";
        $stmt = $this->link->prepare($query);
        $stmt->execute(['order_number' => $orderNumber]);
        return $stmt->fetchColumn() > 0;
    }

    public function getOrderLocation($orderNumber) {
        $query = "SELECT transfer_from FROM stock_transfers_final WHERE transfer_id = :order_number
            LIMIT 1
        ";
        $stmt = $this->link->prepare($query);
        $stmt->execute(['order_number' => $orderNumber]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
    

    public function getFinalLocation($orderNumber) {
        $query = "SELECT transfer_to FROM stock_transfers_final WHERE transfer_id = :order_number LIMIT 1
        ";
        $stmt = $this->link->prepare($query);
        $stmt->execute(['order_number' => $orderNumber]);
        return $stmt->fetch(PDO::FETCH_ASSOC); // ⬅️ fetch both values as associative array
    }
    

    public function getWarehouseProducts($warehouseId) {
        $stmt = $this->link->prepare("SELECT wp.product_id, p.product_name FROM warehouse_products wp LEFT JOIN products p ON wp.product_id = p.product_id  WHERE warehouse_id = ?");
        $stmt->execute([$warehouseId]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getBranchProducts($branchId) {
        $stmt = $this->link->prepare("SELECT br.product_id, p.product_name FROM branch_products br LEFT JOIN products p ON br.product_id = p.product_id WHERE branch_id = ?");
        $stmt->execute([$branchId]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getProductQuantityByLocationType($productCode, $locationType, $locationId) {
        if ($locationType === 'W') {
            $query = "SELECT quantity FROM warehouse_products WHERE warehouse_id = :location_id AND product_id = :product_code";
        } elseif ($locationType === 'B') {
            $query = "SELECT quantity FROM branch_products WHERE branch_id = :location_id AND product_id = :product_code";
        } else {
            return 0; // Invalid type
        }
    
        $stmt = $this->link->prepare($query);
        $stmt->execute([
            'location_id' => $locationId,
            'product_code' => $productCode
        ]);
    
        return $stmt->fetchColumn() ?: 0;
    }
    
    public function getLocationStockDetails($location, $location_from_type, $product_id) {
        // Check if the location type is Warehouse ('W')
        if ($location_from_type === 'W') {
            $sql = "SELECT 
                        wp.quantity
                    FROM
                        warehouse_products wp
                    LEFT JOIN
                        products p ON wp.product_id = p.product_id
                    WHERE
                        wp.warehouse_id = :location 
                        AND wp.product_id = :product_id";
        } 
        // Check if the location type is Branch ('B')
        else if ($location_from_type === 'B') {
            $sql = "SELECT 
                        bp.quantity
                    FROM
                        branch_products bp
                    LEFT JOIN
                        products p ON bp.product_id = p.product_id
                    WHERE
                        bp.branch_id = :location 
                        AND bp.product_id = :product_id";
        }
    
        try {
            // Prepare the SQL statement
            $stmt = $this->link->prepare($sql);
            
            // Bind parameters
            $stmt->bindParam(':location', $location, PDO::PARAM_INT);
            $stmt->bindParam(':product_id', $product_id, PDO::PARAM_INT);
            
            // Execute the query
            $stmt->execute();
    
            // Fetch the result
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
    
            // Return quantity if found, otherwise return 0 or false
            if ($result) {
                return $result['quantity'];
            } else {
                return 0; // If no result is found, return 0 (no stock)
            }
    
        } catch (PDOException $e) {
            // Log any error and return false
            error_log("Error fetching stock details: " . $e->getMessage());
            return false;
        }
    }

    
    public function getProductsByThirdParty($company_id) {
        try {
            $sql = "SELECT 
                        tpp.opening_stock,
                        tpp.quantity,
                        tpp.min_level,
                        tpp.datecaptured,
                        tpp.product_state,
                        p.product_name
                    FROM
                        third_party_products tpp
                    JOIN
                        products p ON tpp.product_id = p.product_id
                    WHERE
                        tpp.company_id = :company_id";

            $stmt = $this->link->prepare($sql);
            $stmt->bindParam(':company_id', $company_id, PDO::PARAM_STR);
            $stmt->execute();

            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            error_log("Product fetch error: " . $e->getMessage());
            return [];
        }
    }

    public function getThirdPartyProductQuantity($company_id, $product_id) {
        $sql = "SELECT quantity FROM third_party_products WHERE company_id = :company_id AND product_id = :product_id LIMIT 1";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':company_id', $company_id, PDO::PARAM_STR);
        $stmt->bindParam(':product_id', $product_id, PDO::PARAM_INT);
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row ? floatval($row['quantity']) : 0;
    }


    public function getDestinations(){
        $sql = "
            SELECT
                w.warehouse_name AS destiny_name,
                w.warehouse_id AS destiny_id
            FROM
                warehouses w
            WHERE 
                w.warehouse_state = 1

            UNION ALL

            SELECT
                b.branch_name AS destiny_name,
                b.branch_id AS destiny_id
            FROM
                branches b
            WHERE 
                b.branch_state = 1
        ";

        $smtp = $this->link->prepare($sql);
        $smtp->execute();
        return $smtp->fetchAll(PDO::FETCH_ASSOC);
    }

    public function productExistsAtDestination($product_id, $destination_id) {
        $sql = "SELECT COUNT(*) FROM (
                SELECT product_id FROM warehouse_products WHERE warehouse_id = :destination_id AND product_id = :product_id
                UNION
                SELECT product_id FROM branch_products WHERE branch_id = :destination_id AND product_id = :product_id
            ) AS combined
        ";

        $stmt = $this->link->prepare($sql);
        $stmt->execute([
            ':destination_id' => $destination_id,
            ':product_id' => $product_id
        ]);
        
        return $stmt->fetchColumn() > 0;
    }

    public function getCompanyProductStock($company_id, $product_id) {
        $sql = "SELECT quantity FROM third_party_products WHERE company_id = :company_id AND product_id = :product_id";
        $stmt = $this->link->prepare($sql);
        $stmt->execute([
            ':company_id' => $company_id,
            ':product_id' => $product_id
        ]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    public function updateSourceQuantity($transfer_from, $product_id, $new_quantity) {

        /* ---------------------------
            1. Update warehouse_products
        ----------------------------*/
        $sql = "UPDATE 
                warehouse_products 
            SET 
                quantity = :qty 
            WHERE 
                warehouse_id = :src 
                AND product_id = :pid";
        $stmt = $this->link->prepare($sql);
        $stmt->bindValue(':qty', $new_quantity);
        $stmt->bindValue(':src', $transfer_from);
        $stmt->bindValue(':pid', $product_id, PDO::PARAM_INT);
        $stmt->execute();

        if ($stmt->rowCount() > 0) {
            return true;
        }


        /* --------------------------------
            2. Update third_party_products
        ---------------------------------*/
        $sql = "UPDATE
                    third_party_products 
                SET 
                    quantity = :qty 
                WHERE 
                    company_id = :src 
                    AND product_id = :pid";
        $stmt = $this->link->prepare($sql);
        $stmt->bindValue(':qty', $new_quantity);
        $stmt->bindValue(':src', $transfer_from);
        $stmt->bindValue(':pid', $product_id, PDO::PARAM_INT);
        $stmt->execute();

        if ($stmt->rowCount() > 0) {
            return true;
        }


        /* ------------------------------
            3. Update branch_products
        -------------------------------*/
        $sql = "UPDATE
                branch_products 
            SET 
                quantity = :qty 
            WHERE 
                branch_id = :src 
                AND product_id = :pid";
        $stmt = $this->link->prepare($sql);
        $stmt->bindValue(':qty', $new_quantity);
        $stmt->bindValue(':src', $transfer_from);
        $stmt->bindValue(':pid', $product_id, PDO::PARAM_INT);
        $stmt->execute();

        if ($stmt->rowCount() > 0) {
            return true;
        }


        /* ------------------------------------
            No table matched this transfer_from
        -------------------------------------*/
        return false;
    }




}