<?php

include_once 'config.php';

class warehouse_model
{
    public $link;

    public function __construct()
    {
        $db_connection = new dbConnection();
        $this->link = $db_connection->connect();
    }

    public function checkWarehouseCode($code) {
        $stmt = $this->link->prepare("SELECT COUNT(*) FROM warehouses WHERE warehouse_code = ?");
        $stmt->execute([$code]);
        $count = $stmt->fetchColumn();
        return $count > 0;
    }

    public function generateNextID() {
        // Get max ID from warehouses
        $sql1 = "SELECT MAX(warehouse_id) AS max_warehouse_id FROM warehouses";
        $stmt1 = $this->link->prepare($sql1);
        $stmt1->execute();
        $result1 = $stmt1->fetch(PDO::FETCH_ASSOC);
        $maxWarehouseId = (int) $result1['max_warehouse_id'];

        // Get max ID from branches
        $sql2 = "SELECT MAX(branch_id) AS max_branch_id FROM branches";
        $stmt2 = $this->link->prepare($sql2);
        $stmt2->execute();
        $result2 = $stmt2->fetch(PDO::FETCH_ASSOC);
        $maxBranchId = (int) $result2['max_branch_id'];

        // Determine the highest of the two
        $maxId = max($maxWarehouseId, $maxBranchId);

        // If no records exist in either table
        if ($maxId < 7000) {
            return 7000;
        }

        return $maxId + 1;
    }

    public function doesLocationAlreadyHaveWarehouse($location_id) {
        $sql = "SELECT COUNT(*) as total FROM warehouses WHERE location = :location_id";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':location_id', $location_id, PDO::PARAM_STR);
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row && $row['total'] > 0;
    }

    public function doesLocationExistExcludingWarehouse($location_id, $exclude_warehouse_id) {
        $sql = "SELECT COUNT(*) as total FROM warehouses WHERE location = :location_id AND warehouse_id != :exclude_id";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':location_id', $location_id, PDO::PARAM_STR);
        $stmt->bindParam(':exclude_id', $exclude_warehouse_id, PDO::PARAM_INT);
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row && $row['total'] > 0;
    }

    public function isSupervisorInAnotherWarehouseExcludingCurrent($supervisor_id, $exclude_warehouse_id) {
        $sql = "SELECT COUNT(*) as total FROM warehouses 
                WHERE supervisor = :supervisor_id AND warehouse_id != :exclude_id";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':supervisor_id', $supervisor_id, PDO::PARAM_INT);
        $stmt->bindParam(':exclude_id', $exclude_warehouse_id, PDO::PARAM_INT);
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row && $row['total'] > 0;
    }

    public function getBranchByShopManager($employee_code) {
        $sql = "SELECT branch_id, location, branch_state 
                FROM branches 
                WHERE shop_manager = :employee_code 
                LIMIT 1";

        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':employee_code', $employee_code, PDO::PARAM_INT);
        $stmt->execute();

        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    public function doesWarehouseNameExistExcludingCurrent($warehouse_name, $warehouse_id) {
        $sql = "SELECT COUNT(*) FROM warehouses WHERE warehouse_name = :name AND warehouse_id != :id";
        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':name', $warehouse_name);
        $stmt->bindParam(':id', $warehouse_id);
        $stmt->execute();
        return $stmt->fetchColumn() > 0;
    }





    public function saveWarehouse($data) {
        try {
            $stmt = $this->link->prepare("INSERT INTO warehouses (warehouse_id, warehouse_name, warehouse_code, location, datecaptured, operator)
                VALUES (:warehouse_id, :warehouse_name, :warehouse_code, :location, :datecaptured, :operator)
            ");
    
            $stmt->bindParam(':warehouse_id', $data['warehouse_id']);
            $stmt->bindParam(':warehouse_name', $data['warehouse_name']);
            $stmt->bindParam(':warehouse_code', $data['warehouse_code']);
            $stmt->bindParam(':location', $data['location']);
            $stmt->bindParam(':datecaptured', $data['datecaptured']);
            $stmt->bindParam(':operator', $data['operator']);
    
            return $stmt->execute();
        } catch (PDOException $e) {
            // Optionally log error
            error_log("Save warehouse error: " . $e->getMessage());
            return false;
        }
    }
    
    public function getWarehouses() {
        try {
            $stmt = $this->link->prepare("SELECT 
                                    w.*,
                                    l.location_name,

                                    GROUP_CONCAT(DISTINCT e.fullname ORDER BY e.fullname SEPARATOR ', ') 
                                        AS supervisor_name,

                                    GROUP_CONCAT(DISTINCT e.phone_number ORDER BY e.fullname SEPARATOR ', ') 
                                        AS phone_number,

                                    COUNT(DISTINCT wp.product_id) AS product_count,

                                    w.warehouse_state,
                                    CASE 
                                        WHEN w.warehouse_state = 1 THEN 'Active warehouse'
                                        WHEN w.warehouse_state = 0 THEN 'Deactivated warehouse'
                                    END AS warehouse_status

                                FROM warehouses w

                                LEFT JOIN locations l 
                                    ON w.location = l.loc_id

                                LEFT JOIN users uu 
                                    ON uu.location = w.warehouse_id
                                AND uu.role = 'supervisor'
                                AND uu.user_state = 1

                                LEFT JOIN employees e 
                                    ON uu.employee_code = e.employee_code

                                LEFT JOIN warehouse_products wp 
                                    ON w.warehouse_id = wp.warehouse_id

                                WHERE w.warehouse_state IN (0,1)

                                GROUP BY w.warehouse_id

                                ORDER BY w.warehouse_name ASC");
            $stmt->execute();
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            error_log("Get warehouses error: " . $e->getMessage());
            return [];
        }
    }

    public function isSupervisorInAnotherWarehouse($employee_code, $location) {
        $query = "SELECT location FROM warehouses WHERE supervisor = :supervisor AND warehouse_state = 1 LIMIT 1";
        $stmt = $this->link->prepare($query);
        $stmt->bindParam(':supervisor', $employee_code, PDO::PARAM_INT);
        $stmt->execute();

        $existing_location = $stmt->fetchColumn();

        // If location exists and is not the same, it's an invalid assignment
        return $existing_location && $existing_location !== $location;
    }

    public function isAssignedToBranchInDifferentLocation($employee_code, $location) {
        $query = "SELECT location FROM users WHERE employee_code = :employee AND role IN('Supervisor', 'Cashier') LIMIT 1";
        $stmt = $this->link->prepare($query);
        $stmt->bindParam(':employee', $employee_code, PDO::PARAM_INT);
        $stmt->execute();

        $branch_location = $stmt->fetchColumn();

        return $branch_location && $branch_location !== $location;
    }



    public function warehouse_stock($warehouse_id) {
        try {
            $sql = "SELECT 
                        wp.*, 
                        p.product_name, 
                        p.product_code, 
                        wp.opening_stock,
                        wp.quantity,
                        pc.categoryname,
                        pu.unitname,
                        w.warehouse_name,
                        wp.min_level as min_order_level,
                        p.selling_price
                    FROM 
                        warehouse_products wp
                    LEFT JOIN
                        warehouses w ON wp.warehouse_id = w.warehouse_id
                    LEFT JOIN 
                        products p ON wp.product_id = p.product_id
                    LEFT JOIN
                        product_categories pc ON p.category = pc.category_id
                    LEFT JOIN
                        product_units pu ON p.unit = pu.unit_id
                    WHERE 
                        wp.warehouse_id = :warehouse_id";
    
            $stmt = $this->link->prepare($sql);
            $stmt->bindParam(':warehouse_id', $warehouse_id, PDO::PARAM_INT);
            $stmt->execute();
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
    
        } catch (PDOException $e) {
            error_log("Warehouse stock fetch error: " . $e->getMessage());
            return [];
        }
    }

    public function getWarehouseDetails($warehouse_id) {
        try {
            $sql = "SELECT 
                        w.*, 
                        w.location,
                        l.location_name,
                        (
                            SELECT COUNT(*) 
                            FROM warehouse_products wp 
                            WHERE wp.warehouse_id = w.warehouse_id
                        ) AS total_products
                    FROM 
                        warehouses w
                    LEFT JOIN 
                        locations l ON w.location = l.loc_id
                    WHERE 
                        w.warehouse_id = :warehouse_id";
    
            $stmt = $this->link->prepare($sql);
            $stmt->bindParam(':warehouse_id', $warehouse_id, PDO::PARAM_INT);
            $stmt->execute();
    
            return $stmt->fetch(PDO::FETCH_ASSOC);
    
        } catch (PDOException $e) {
            error_log("Get warehouse details error: " . $e->getMessage());
            return false;
        }
    }

    public function updateWarehouse($id, $name, $location) {
        $sql = "UPDATE warehouses SET warehouse_name = ?, location = ? WHERE warehouse_id = ?";
        $stmt = $this->link->prepare($sql);
        return $stmt->execute([$name, $location, $id]);
    }
    
    
    
    

}