<?php
session_start();
date_default_timezone_set('Africa/Blantyre');

include_once '../model/payments_model.php';
include_once '../model/sales_model.php';

$sales_model    = new sales_model();
$payments_model = new payments_model();

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {

    switch ($_POST['action']) {

        case 'process_invoice_payment':
            processInvoicePayment($payments_model, $sales_model);
            break;

        default:
            $_SESSION['notification'] = [
                'type' => 'error',
                'title'=> 'Error',
                'message'=> 'Invalid action'
            ];
            header('Location: ../dashboard.php');
            exit();
    }
}

function processInvoicePayment($payments_model, $sales_model) {
    header('Content-Type: application/json');

    $customer_id    = intval($_POST['customer_id']);
    $payment_method = trim($_POST['payment_method']);
    $transaction_id = trim($_POST['transactionId'] ?? '');
    $payment_date   = $_POST['payment_date'];
    $amount_paid    = floatval($_POST['amount_paid']);
    $user_id        = $_SESSION['sess_employeecode'] ?? null;
    $location       = $_POST['location'];
    $datecaptured   = date('Y-m-d H:i:s');
    $debug          = isset($_POST['debug']) && $_POST['debug'] == 1;

    // Helper to return error as JSON
    $jsonError = function($message) {
        echo json_encode([
            'status' => 'error',
            'message' => $message
        ]);
        exit;
    };

    // ---- Basic Validation ----
    if (!$customer_id || !$payment_method || !$payment_date || $amount_paid <= 0) {
        $jsonError("Missing or invalid payment data");
    }

    if ($payment_method !== 'Cash' && empty($transaction_id)) {
        $jsonError("Transaction ID is required for non-cash payments");
    }

    if (!empty($transaction_id) && $sales_model->transactionIdExists($transaction_id)) {
        $jsonError("Transaction ID already exists");
    }

    $total_balance = $sales_model->getTotalCustomerInvoiceBalances($customer_id);
    if ($total_balance <= 0) $jsonError("Customer has no outstanding balance");
    if ($amount_paid > $total_balance) $jsonError("Payment amount exceeds outstanding balance");

    // ---- Generate TX ID & Receipt Number ----
    $tx_id = 'TX-' . date('YmdHis') . '-' . strtoupper(substr(md5(rand()), 0, 4));
    $receipt_num = $sales_model->getNextDocumentNumber('receipt', $location, 'warehouse');

    // ---- Prepare payment preview ----
    $preview = [];
    $preview['payment'] = [
        'tx_id'          => $tx_id,
        'sale_id'        => null,
        'location'       => $location,
        'customer'       => $customer_id,
        'receipt_number' => $receipt_num,
        'payment_method' => $payment_method,
        'payment_type'   => 'Order Payment',
        'transaction_id' => $transaction_id,
        'payment_date'   => $payment_date,
        'discount'       => 0,
        'amount_paid'    => $amount_paid,
        'balance'        => $total_balance - $amount_paid,
        'date_captured'  => $datecaptured,
        'operator'       => $user_id,
        'payment_state'  => 1
    ];

    // ---- Allocate Payment to Selected Invoices ----
    if (empty($_POST['selected_invoices']) || !is_array($_POST['selected_invoices'])) {
        $jsonError('No invoices selected for payment');
    }

    $selectedInvoices = $_POST['selected_invoices'];
    $remaining = $amount_paid;

    $preview['invoice_payments'] = [];
    $preview['updated_invoices'] = [];

    foreach ($selectedInvoices as $row) {
        $invoice_number = $row['invoice_number'];
        $pay_amount     = floatval($row['amount']);

        if ($pay_amount <= 0) {
            $jsonError("Invalid payment amount for invoice {$invoice_number}");
        }

        $inv = $sales_model->getInvoiceForPayment($invoice_number, $customer_id);
        if (!$inv) {
            $jsonError("Invoice {$invoice_number} not found");
        }

        $invoice_balance = $inv['total_bill'] - $inv['paid_amount'];

        if ($pay_amount > $invoice_balance) {
            $jsonError("Payment exceeds balance for invoice {$invoice_number}");
        }

        if ($pay_amount > $remaining) {
            $jsonError("Total selected invoice payments exceed entered payment amount");
        }

        $balance_after = $invoice_balance - $pay_amount;

        $preview['invoice_payments'][] = [
            'payment_id'       => null,
            'invoice_number'   => $invoice_number,
            'customer'         => $customer_id,
            'receipt_number'   => $receipt_num,
            'amount_paid'      => $pay_amount,
            'balance'          => $balance_after,
            'date_paid'        => $payment_date,
            'transaction_date' => $datecaptured,
            'operator'         => $user_id,
            'payment_state'    => 1
        ];

        $new_state = ($balance_after == 0) ? 2 : 1;

        $preview['updated_invoices'][] = [
            'invoice_number' => $invoice_number,
            'old_state'      => $inv['order_state'],
            'new_state'      => $new_state,
            'total_paid'     => $inv['paid_amount'] + $pay_amount,
            'remaining'      => $balance_after
        ];

        $remaining -= $pay_amount;
    }

    if ($remaining != 0) {
        $jsonError('Entered payment amount does not match selected invoice totals');
    }

    $preview['total_applied']    = $amount_paid;
    $preview['unapplied_amount'] = $remaining;

    // ---- DEBUG MODE: Return Preview Only ----
    if ($debug) {
        echo json_encode([
            'status' => 'preview',
            'data'   => $preview
        ]);
        exit;
    }

    // ---- INSERT INTO DB ----
    $payments_model->beginTransaction();
    try {
        $payment_id = $payments_model->insertPayment($preview['payment']);
        if (!$payment_id) throw new Exception("Failed to record payment");

        foreach ($preview['invoice_payments'] as &$invPay) {
            $invPay['payment_id'] = $payment_id;
            $payments_model->insertInvoicePayment($invPay);

            $inv_update = array_filter($preview['updated_invoices'], fn($u) => $u['invoice_number'] == $invPay['invoice_number']);
            $inv_update = reset($inv_update);
            $payments_model->updateInvoiceState($invPay['invoice_number'], $inv_update['new_state']);
        }

        $payments_model->commit();

        echo json_encode([
            'status' => 'success',
            'payment_id' => $payment_id
        ]);
        exit();

    } catch (Exception $e) {
        $payments_model->rollback();
        $jsonError($e->getMessage());
    }
}

