<?php
session_start();
date_default_timezone_set('Africa/Blantyre');
include_once '../model/quotations_model.php';
include_once '../model/common_model.php';
include_once '../model/stock_model.php';

$stock_model = new stock_model();
$common_model = new common_model();
$quotations_model = new quotations_model();

// Ensure action exists
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['action'])) {
    $action = $_POST['action'];

    switch ($action) {

        case "save_quotation":
            saveQuotation($quotations_model, $common_model);
            break;

        case "update_quotation":
            updateQuotation($quotations_model);
            break;
        
        case "get_products":
            getQuoteProducts($stock_model);
            break;

        case "get_quotation":
            getQuotation($quotations_model);
            break;

        case "delete_quotation":
            deleteQuotation($quotations_model);
            break;

        case "delete_item":
            removeQuotationItem($quotations_model);
            break;

        case "get_products_new":
            getQuoteProductsNew($stock_model);
            break;


        default:
            $_SESSION['notification']=[
                'type' => 'error',
                'title' => 'FAILED',
                'message' => 'Unverified Action Requested'
            ];
            header('location:../dashboard.php');
            exit();
            break;  
    }
}
// ----------------------------
// FUNCTIONS
// ----------------------------

function getQuoteProducts($stock_model){
    $location = $_POST['location'];
    $products = $stock_model->getproducts($location);    
    echo json_encode($products);
    exit();
}

function getQuotation($quotation_model){
    $quote_number = $_POST['quote_number'];
    $header = $quotation_model->getQuotationHeader($quote_number);
    $items  = $quotation_model->getQuotationItems($quote_number);

    echo json_encode(['header' => $header, 'items' => $items]);
    exit;
}

function saveQuotation($quotation_model, $common_model){
    
    $quote_number = trim($_POST['quotation_number']);
    if($_POST['locked_customer_id']){
        $customer_id = intval($_POST['locked_customer_id']);
    } else {
        $customer_id = intval($_POST['customer_id']);
    }

    $locationRaw = $_POST['location'] ?? '';

    if (!$locationRaw || !strpos($locationRaw, '|')) {
        $_SESSION['notification'] = [
            'type' => 'error',
            'title' => 'Invalid Location',
            'message' => 'Please select a valid location.'
        ];
        header('Location: ../new_quote.php');
        exit();
    }

    [$location_type, $location_id] = explode('|', $locationRaw);
    $location_id = (int)$location_id;

    /* ✅ Validate existence */
    if (!validateLocation($quotation_model, $location_type, $location_id)) {
        $_SESSION['notification'] = [
            'type' => 'error',
            'title' => 'Invalid Location',
            'message' => ucfirst($location_type) . ' does not exist.'
        ];
        header('Location: ../new_quote.php');
        exit();
    }

    $vat_applied = isset($_POST['vat_applied']) && $_POST['vat_applied'] == 1;

    $operator = $_SESSION['sess_employeecode'] ?? 1;
    $transaction_date = date('Y-m-d H:i:s');
    $quotation_valid_until = date('Y-m-d H:i:s', strtotime('+30 days'));
    $quotation_state = 0; // default state, e.g., draft

    // Quotation items
    $products = $_POST['products'] ?? [];

    if(empty($customer_id)){
        $_SESSION['notification'] = [
            'type'=>'error',
            'title'=>'Failed',
            'message'=>'Please select customer.'
        ];
        header('Location: ../new_quote.php');
        exit();
    }
    if(empty($quote_number)){
        $_SESSION['notification'] = [
            'type'=>'error',
            'title'=>'Failed',
            'message'=>'Quotation number is required.'
        ];
        header('Location: ../new_quote.php?customer_id='.urldecode($customer_id));
        exit();
    }
    if(empty($products)){
        $_SESSION['notification'] = [
            'type'=>'error',
            'title'=>'Failed',
            'message'=>'Quotation must have at least one product.'
        ];
        header('Location: ../new_quote.php?customer_id='.urldecode($customer_id).'&quote_number='.urldecode($quote_number));
        exit();
    }
    $companyDetails = $common_model->getCompanyDetails();
    $vat_value = $companyDetails['vat_percentage'];
    $vat_rate = $vat_value; 
    $gross_total = 0;

    foreach ($products as $product_id => $details) {
        $qty   = (float)$details['quantity'];
        $price = (float)$details['price']; // always what UI sends
        $gross_total += ($price * $qty);
    }

    if ($vat_applied) {
        $total_subtotal = $gross_total / (1 + $vat_rate);
        $total_tax      = $gross_total - $total_subtotal;
        $total          = $gross_total;
    } else {
        $total_subtotal = $gross_total;
        $total_tax      = 0;
        $total          = $gross_total;
    }



    // Now we can insert the header
    $header_id = $quotation_model->insertQuotation([
        'quote_number'           => $quote_number,
        'customer'               => $customer_id,
        'quote_from'             => $location_id,
        'location_type'          => $location_type,
        'operator'               => $operator,
        'transaction_date'       => $transaction_date,
        'vat_value'              => $vat_value,
        'quotation_valid_until'  => $quotation_valid_until,
        'quotation_state'        => $quotation_state,
        'total_subtotal'         => $total_subtotal,
        'total_tax'              => $total_tax,
        'total'                  => $total
    ]);


    if(!$header_id){
        $_SESSION['notification'] = [
            'type'=>'error',
            'title'=>'Failed',
            'message'=>'Could not save quotation header.'
        ];
        header('Location: ../new_quote.php?qoute_number='.urldecode($quote_number));
        exit();
    }

    // Insert each item
    $vat_rate = $vat_value; // already 0.165

    foreach ($products as $product_id => $details) {

        $quantity = (float)$details['quantity'];
        $unit_price_input = (float)$details['price']; // what UI sent

        if ($vat_applied) {
            // Convert to VAT-exclusive
            $unit_price_exclusive = $unit_price_input / (1 + $vat_rate);
            $line_subtotal = $unit_price_exclusive * $quantity;
            $line_tax      = $line_subtotal * $vat_rate;
            $line_total    = $line_subtotal + $line_tax;
        } else {
            // VAT not applied, store as-is
            $unit_price_exclusive = $unit_price_input;
            $line_subtotal = $unit_price_exclusive * $quantity;
            $line_tax      = 0;
            $line_total    = $line_subtotal;
        }

        $quotation_model->insertQuotationItem([
            'quotation_id' => $quote_number,
            'product_id'   => $product_id,
            'quantity'     => $quantity,
            'price'        => $unit_price_exclusive, // ✅ ALWAYS EXCL VAT
            'subtotal'     => $line_subtotal,         // ✅ ALWAYS EXCL VAT
            'tax'          => $line_tax,
            'total'        => $line_total
        ]);
    }



    $_SESSION['notification'] = [
        'type'=>'success',
        'title'=>'Success',
        'message'=>'Quotation saved successfully.'
    ];
    header('Location: ../customer_quote.php?quote_number='.urldecode($quote_number));
    exit();
}

function validateLocation($quotation_model, $location_type, $location_id) {

    if ($location_type === 'warehouse') {
        $sql = "SELECT warehouse_id FROM warehouses WHERE warehouse_id = ? LIMIT 1";
    } elseif ($location_type === 'branch') {
        $sql = "SELECT branch_id FROM branches WHERE branch_id = ? LIMIT 1";
    } else {
        return false;
    }

    $stmt = $quotation_model->link->prepare($sql);
    $stmt->execute([$location_id]);

    return (bool) $stmt->fetchColumn();
}



function updateQuotation($quotation_model)
{
    parse_str($_POST['data'], $data); // convert serialized data to array

    $quote_number = $data['quote_number'] ?? '';
    $items = $data['items'] ?? [];

    $success = $quotation_model->updateQuotationItems($quote_number, $items);

    echo json_encode(['success' => $success]);
    exit;
}

function deleteQuotation($quotation_model)
{
    // future use
}

function getQuoteProductsNew($stock_model){
    $location = $_POST['location']; // warehouse|1 OR branch|2
    [$type, $id] = explode('|', $location);

    $products = $stock_model->getProductsByLocation($type, (int)$id);
    echo json_encode($products);
    exit();
}



function removeQuotationItem($quotation_model)
{
    $quote_number = $_POST['quote_number'];
    $product_id = $_POST['product_id'];   

    $success = $quotation_model->deleteQuotationItem($quote_number, $product_id);

    echo json_encode(['success' => $success]);
    exit;
}
?>
