<?php
session_start();
header('Content-Type: application/json; charset=utf-8');

try {
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        throw new Exception('Invalid request method');
    }

    $action = $_POST['action'] ?? null;

    if (empty($action)) {
        throw new Exception('Action is required');
    }

    include_once __DIR__ . '/../model/stock_model.php';
    include_once __DIR__ . '/../model/transfer_model.php';
    
    $stock_model = new stock_model();
    $transfer_model = new transfer_model();

    if (empty($_SESSION['sess_employeecode'])) {
        throw new Exception('User not authenticated');
    }

    date_default_timezone_set('Africa/Blantyre');
    $operator = $_SESSION['sess_employeecode'];
    $date = date('Y-m-d H:i:s');

    switch ($action) {
        case 'approve_transfer':
            approveTransfer($stock_model, $operator, $date);
            break;
            
        case 'add_item':
            addTransferItemToDB($stock_model, $transfer_model, $operator, $date);
            break;
            
        case 'get_items':
            getTransferItems($transfer_model);
            break;

        case 'check_has_items':
            checkTransferHasItems($transfer_model);
            break;
        case 'update_all_dates':
            updateTransferDate($transfer_model);
            break;
        case 'update_item_quantity':
            updateItemQuantity($transfer_model, $stock_model);
            break;
        case 'delete_item':
            deleteTransferItem($transfer_model, $stock_model);
            break;
        case 'finalize_transfer':
            finalizeTransfer($transfer_model);
            break;

        default:
            throw new Exception('Invalid action');
    }

} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode([
        'status' => 'error',
        'message' => 'Database error: ' . $e->getMessage()
    ]);
    exit;
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'status' => 'error',
        'message' => $e->getMessage()
    ]);
    exit;
}

// ==================== ACTION FUNCTIONS ====================
function finalizeTransfer($transfer_model) {
    header('Content-Type: application/json; charset=utf-8');
    date_default_timezone_set('Africa/Blantyre');

    $transfer_id = trim($_POST['transfer_id'] ?? '');
    $driver = trim($_POST['driver'] ?? '');
    $operator = $_SESSION['sess_employeecode'] ?? 0;

    if (!$transfer_id) {
        echo json_encode([
            'success' => false,
            'message' => 'Transfer ID is required.'
        ]);
        exit;
    }

    $transfer_details = $transfer_model->getFirstTransferItemDetails($transfer_id);

    if (!$transfer_details) {
        echo json_encode([
            'success' => false,
            'message' => 'Unable to finalize: transfer details not found.'
        ]);
        exit;
    }

    // Prevent duplicate finalization
    if ($transfer_model->orderNumberExists($transfer_id)) {
        echo json_encode([
            'success' => false,
            'message' => 'This transfer has already been finalized. No duplicate entries allowed.'
        ]);
        exit;
    }

    // Insert record in stock_transfer_finals
    $saved = $transfer_model->finalizeStockTransfer([
        'transfer_id'    => $transfer_id,
        'transfer_from'  => $transfer_details['transfer_from'],
        'transfer_to'    => $transfer_details['transfer_to'],
        'transfer_date'  => $transfer_details['transfer_date'],
        'operator'       => $operator,
        'transfer_state' => 1,
        'datecaptured'   => date('Y-m-d H:i:s'),
        'driver'         => $driver
    ]);

    if ($saved) {
        echo json_encode([
            'success' => true,
            'message' => 'Stock transfer has been finalized successfully.'
        ]);
    } else {
        echo json_encode([
            'success' => false,
            'message' => 'Failed to finalize the transfer.'
        ]);
    }

    exit;
}


function updateItemQuantity($transfer_model, $stock_model) {
    $tn_number = $_POST['tn_number'] ?? null;
    $item_id = $_POST['item_id'] ?? null;
    $new_quantity = $_POST['new_quantity'] ?? null;

    $old_qty = $_POST['old_qty'];
    $product = $_POST['product_id'];
    $warehouse = $_POST['warehouse_id'];

    $current_quantity = $stock_model->getWarehouseQuantity($product, $warehouse);
    $original_stock = $current_quantity + $old_qty;

    $updated_quantity = $original_stock - $new_quantity;

    if (empty($tn_number) || empty($item_id) || empty($new_quantity)) {
        echo json_encode(['success' => false, 'message' => 'Required fields missing']);
        return;
    }
    
    $updated_stock = $stock_model->updateSourceQuantity($warehouse, $product, $updated_quantity);
    if($updated_stock){
        $updated = $transfer_model->updateItemQuantity($item_id, $new_quantity);
    }
    

    if ($updated) {
        echo json_encode(['success' => true]);
    } else {
        echo json_encode(['success' => false, 'message' => 'Failed to update quantity']);
    }
}

function updateTransferDate($transfer_model) {
    $tn_number = $_POST['tn_number'] ?? null;
    $new_date = $_POST['new_date'] ?? null;

    // Clear any previous output
    ob_clean();

    if (empty($tn_number) || empty($new_date)) {
        echo json_encode(['success' => false, 'message' => 'TN number and new date are required']);
        return;
    }

    $updated = $transfer_model->updateTransferDate($tn_number, $new_date);

    if ($updated) {
        echo json_encode(['success' => true, 'updated_id' => $tn_number]);
    } else {
        echo json_encode(['success' => false, 'message' => 'Failed to update transfer date']);
    }

    exit; // prevent any further output
}

function checkTransferHasItems($transfer_model) {
    $tn_number = $_POST['tn_number'] ?? null;

    if (empty($tn_number)) {
        echo json_encode([
            'success' => false,
            'message' => 'Transfer Note number is required.'
        ]);
        return;
    }

    $items = $transfer_model->getTransferItems($tn_number);
    $has_items = count($items) > 0;

    // Default locations
    $from_warehouse_id = null;
    $from_warehouse_name = null;
    $destination_type = null;
    $destination_id = null;
    $destination_name = null;

    if ($has_items) {
        // Get first item's locations
        $locations = $transfer_model->getTransferLocations($tn_number);

        $from_warehouse_id = $locations['from_warehouse_id'] ?? null;
        $from_warehouse_name = $locations['from_warehouse_name'] ?? null;
        $destination_type = $locations['destination_type'] ?? null;
        $destination_id = $locations['destination_id'] ?? null;
        $destination_name = $locations['destination_name'] ?? null;
    }

    echo json_encode([
        'success' => true,
        'has_items' => $has_items,
        'from_warehouse_id' => $from_warehouse_id,
        'from_warehouse_name' => $from_warehouse_name,
        'destination_type' => $destination_type,
        'destination_id' => $destination_id,
        'destination_name' => $destination_name
    ]);
    return;
}


function getTransferItems($transfer_model) {
    $tn_number = $_POST['tn_number'] ?? null;
    // Get table HTML
    $html = $transfer_model->buildItemsTableHTML($tn_number);

    // Get modals
    $modals = $transfer_model->buildItemModalsHTML($tn_number);

    // Determine if finalize should be enabled
    $items = $transfer_model->getTransferItems($tn_number);
    $enable_finalize = count($items) > 0;

    echo json_encode([
        'success' => true,
        'html' => $html,
        'modals' => $modals,
        'finalize_enabled' => $enable_finalize
    ]);
    return;
}



function addTransferItemToDB($stock_model, $transfer_model, $operator, $date){

    // Collect data
    $tn_number          = trim($_POST['tn_number'] ?? '');
    $warehouse          = trim($_POST['warehouse_id'] ?? '');
    $transfer_date      = trim($_POST['transfer_date'] ?? '');
    $product            = intval($_POST['product_id'] ?? 0);
    $quantity           = floatval($_POST['quantity'] ?? 0);
    $remaining_quantity = floatval($_POST['remaining_quantity'] ?? 0);
    $destination_id     = intval($_POST['destination'] ?? 0);
    $transfer_state     = 1;

    // Validate
    if (!$warehouse || !$product || !$destination_id || $quantity <= 0) {
        echo json_encode([
            'success' => false,
            'message' => 'Please fill in all required fields.'
        ]);
        return;
    }

    if (empty($operator)) {
        echo json_encode([
            'success' => false,
            'message' => 'Please login again.'
        ]);
        return;
    }

    if (!isValidDate($transfer_date)) {
        echo json_encode([
            'success' => false,
            'message' => 'Invalid transfer date.'
        ]);
        return;
    }

    // Stock check
    $current_quantity = $stock_model->getWarehouseQuantity($product, $warehouse);

    if ($current_quantity <= 0) {
        echo json_encode([
            'success' => false,
            'message' => 'Product has no stock.'
        ]);
        return;
    }

    if ($quantity > $current_quantity) {
        echo json_encode([
            'success' => false,
            'message' => 'Quantity exceeds available stock.'
        ]);
        return;
    }

    // Pending quantity
    $pending_quantity = $transfer_model->getPendingQuantityForProduct($warehouse, $product);
    $total_requested  = $pending_quantity + $quantity;

    if ($total_requested > $remaining_quantity) {
        echo json_encode([
            'success' => false,
            'message' => 'Total requested exceeds remaining stock.'
        ]);
        return;
    }

    // Product exists at destination?
    if (!$stock_model->productExistsAtDestination($product, $destination_id)) {
        echo json_encode([
            'success' => false,
            'message' => 'Selected Product is not attached to the selected Destination.'
        ]);
        return;
    }

    // Duplicate check
    if ($transfer_model->isProductAlreadyInTransfer($tn_number, $product)) {
        echo json_encode([
            'success' => false,
            'message' => 'Product already added in this transfer.'
        ]);
        return;
    }

    // Receiver stock
    $receiver_stock = $stock_model->getProductQuantityAtDestination($product, $destination_id);
    $receiver_open  = floatval($receiver_stock['quantity'] ?? 0);

    // Prepare save data
    $data = [
        'transfer_id'            => $tn_number,
        'product_id'             => $product,
        'transfer_from'          => $warehouse,
        'transfer_to'            => $destination_id,
        'sender_opening_stock'   => $current_quantity,
        'receiver_opening_stock' => $receiver_open,
        'qty_initiated'          => $quantity,
        'original_qty_initiated' => $quantity,
        'sender_closing_stock'   => $current_quantity - $quantity,
        'receiver_closing_stock' => $receiver_open + $quantity,
        'operator'               => $operator,
        'transfer_date'          => $transfer_date,
        'transfer_state'         => $transfer_state,
        'datecaptured'           => $date
    ];
    $new_quantity = $current_quantity - $quantity;
    $reduce_stock = $stock_model->updateWarehouseQuantity($product, $warehouse, $new_quantity);
    $result = $transfer_model->transferStock($data);

    if (!$result) {
        echo json_encode([
            'success' => false,
            'message' => 'Failed to save transfer item.'
        ]);
        return;
    }

    // Reload items HTML
    $items_html  = $transfer_model->buildItemsTableHTML($tn_number);
    $modals_html = $transfer_model->buildItemModalsHTML($tn_number);

    echo json_encode([
        'success' => true,
        'message' => 'Item added successfully.',
        'html'    => $items_html,
        'modals'  => $modals_html,
        'finalize_enabled' => true
    ]);
}


function approveTransfer($stock_model, $operator, $date) {
    $tn_number = $_POST['tn_number'] ?? null;
    $driver_date_raw = $_POST['delivery_date'] ?? null;
    $deliverynoteNumber = $_POST['deliverynoteNumber'];
    $debug = $_POST['debug'] ?? 0; // <-- debug flag
    $receiver = $_POST['receiver'];

    if (empty($tn_number)) throw new Exception('Missing transfer note number');
    if (empty($driver_date_raw)) throw new Exception('Delivery Date & Time is required');
    if (empty($deliverynoteNumber)) throw new Exception('Delivery Note number is required');

    $driver_date = str_replace('T', ' ', $driver_date_raw);
    if (strlen($driver_date) === 16) $driver_date .= ':00';

    $state = 2;
    $reason = null;

    $items = $stock_model->getDeliveryNoteItems($tn_number);
    if (!$items) throw new Exception('No transfer items found for this Transfer Note.');

    $preview = [];

    foreach ($items as $item) {
        $item_id = $item['id'];
        $product_id = $item['product_id'];
        $transfer_from = $item['transfer_from'];
        $transfer_to = $item['transfer_to'];
        $qty_initiated = floatval($item['qty_initiated']);
        $original_qty_initiated = floatval($item['original_qty_initiated']);
        $qty_accepted = floatval($item['qty_accepted']);

        if ($qty_accepted <= 0) $qty_accepted = $original_qty_initiated;

        $receiver_opening = $stock_model->getWarehouseQuantity($product_id, $transfer_to);

        $receiver_closing = $receiver_opening + $qty_accepted;

        $preview[] = [
            'item_id' => $item_id,
            'product_id' => $product_id,
            'transfer_from' => $transfer_from,
            'transfer_to' => $transfer_to,
            'receiver_opening' => $receiver_opening,
            'qty_initiated' => $qty_initiated,
            'qty_accepted' => $qty_accepted,
            'receiver_closing' => $receiver_closing
        ];

        if (!$debug) {
            $ok = $stock_model->updateTransferItem($item_id, $tn_number, $receiver_opening, $qty_accepted, $receiver_closing, $state, $date, $operator);
            if (!$ok) throw new Exception("Failed to update stock_transfer_item ID: $item_id");

            $stock_model->updateWarehouseQuantity($product_id, $transfer_to, $receiver_closing);
        }
    }

    if ($debug) {
        echo json_encode([
            'status' => 'preview',
            'items' => $preview,
            'message' => 'Preview of stock changes before committing.'
        ]);
        exit;
    }

    $ok = $stock_model->approveTransferNote($tn_number, $state, $operator, $date, $reason, $driver_date, $deliverynoteNumber);
    if (!$ok) throw new Exception('Failed to update transfer final record.');

    $message = "Your Delivery Note - '.$tn_number'. was Approved";
    $purpose = 'delivery_note';
    $notification = $stock_model->sendNotification($operator, $receiver, $purpose, $tn_number, $message);
    if(!$notification) throw new Exception('Failed To Send Notification.');

    echo json_encode([
        'status' => 'success',
        'message' => 'Transfer Note approved successfully.'
    ]);
    exit;
}

function isValidDate($date, $format = 'Y-m-d')
{
    if (empty($date)) return false;

    $d = DateTime::createFromFormat($format, $date);

    return $d && $d->format($format) === $date;
}

function deleteTransferItem($transfer_model, $stock_model) {
    $item_id = $_POST['item_id'] ?? null;
    $tn_number = $_POST['tn_number'] ?? null;
    $product_id = $_POST['product_id'];
    $warehouse = $_POST['warehouse_id'];
    $qty = $_POST['qty'];
    $current_quantity = $stock_model->getWarehouseQuantity($product_id, $warehouse);
    $new_quantity = $current_quantity + $qty;

    if (!$item_id) {
        echo json_encode(['success' => false, 'message' => 'Item ID required']);
        return;
    }

    $count_items = $transfer_model->countTransferItems($tn_number);
    if ($count_items == 1) {
        $delete_record = $transfer_model->deletedocumentNumber($tn_number);
        }
    $restore_stock = $stock_model->updateWarehouseQuantity($product_id, $warehouse, $new_quantity);
    $deleted = $transfer_model->deleteTransferItem($item_id);

    if ($deleted) {
        echo json_encode(['success' => true, 'message' => 'Item deleted successfully']);
    } else {
        echo json_encode(['success' => false, 'message' => 'Failed to delete item']);
    }
}

