<?php

include_once 'config.php';

class customers_model
{
    public $link;

    public function __construct()
    {
        $db_connection = new dbConnection();
        $this->link = $db_connection->connect();
    }

    /**
     * Get all customers
     * @return array
     */
    public function getCustomers($location, $types = [], $state = null) {

        $sql = "SELECT
                    cc.customer_id,
                    cc.customer_name,
                    cc.type_customer,
                    cc.contact_number AS phone,
                    cc.email,
                    COALESCE(b.branch_name, w.warehouse_name) AS location,
                    cc.type_customer as type,
                    cc.customer_state as state,
                    COALESCE(inv.total_invoiced, 0) AS total_invoiced,
                    COALESCE(pay.total_paid, 0) AS total_paid,
                    (COALESCE(inv.total_invoiced, 0) - COALESCE(pay.total_paid, 0)) AS outstanding_balance

                FROM customers cc

                LEFT JOIN branches b
                    ON cc.registered_location = b.branch_id
                LEFT JOIN warehouses w
                    ON cc.registered_location = w.warehouse_id                
                
                /* ---- TOTAL INVOICED PER CUSTOMER ---- */
                LEFT JOIN (
                    SELECT
                        customer,
                        SUM(total_bill) AS total_invoiced
                    FROM saved_orders
                    WHERE order_state IN (0,1,2)
                    GROUP BY customer
                ) inv ON inv.customer = cc.customer_id

                /* ---- TOTAL PAID PER CUSTOMER ---- */
                LEFT JOIN (
                    SELECT
                        so.customer,
                        SUM(p.amount_paid) AS total_paid
                    FROM invoice_payments p
                    INNER JOIN saved_orders so 
                        ON so.order_number = p.invoice_number
                    GROUP BY so.customer
                ) pay ON pay.customer = cc.customer_id";

        // ------------------------------------
        // Dynamic WHERE conditions
        // ------------------------------------
        $conditions = [];
        $params = [];

        // Location access rule
        // 7000 = super location (can see all customers)
        if ($location != 7000) {
            $conditions[] = "cc.registered_location = ?";
            $params[] = $location;
        }



        // Filter by customer types
        if (!empty($types)) {
            $placeholders = implode(",", array_fill(0, count($types), "?"));
            $conditions[] = "cc.type_customer IN ($placeholders)";
            $params = array_merge($params, $types);
        }

        // 🔥 UPDATED: customer_state filter (single OR multiple)
        if ($state !== null) {
            if (is_array($state)) {
                $placeholders = implode(",", array_fill(0, count($state), "?"));
                $conditions[] = "cc.customer_state IN ($placeholders)";
                $params = array_merge($params, $state);
            } else {
                $conditions[] = "cc.customer_state = ?";
                $params[] = $state;
            }
        }

        // Apply WHERE clause if needed
        if (!empty($conditions)) {
            $sql .= " WHERE " . implode(" AND ", $conditions);
        }

        // Grouping & ordering
        $sql .= "
            GROUP BY 
                cc.customer_id,
                cc.customer_name,
                cc.type_customer,
                cc.contact_number,
                cc.email
            ORDER BY cc.customer_name ASC
        ";

        $stmt = $this->link->prepare($sql);
        $stmt->execute($params);

        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getCustomerById($customer_id) {
        $sql = "SELECT 
                    customer_id,
                    customer_name,
                    contact_number as phone,
                    email,
                    CASE
                        WHEN type_customer = 'Dealer' THEN 'Dealer'
                        WHEN type_customer = 'Corporate' THEN 'corporate'
                        WHEN type_customer = 'Individual' THEN 'individual Creditors'
                        ELSE type_customer
                    END AS type,
                    (SELECT 
                        COALESCE(SUM(total_bill), 0) 
                     FROM 
                        saved_orders 
                     WHERE 
                        customer = customers.customer_id 
                        AND order_state IN (0,1,2)
                    ) -
                    (SELECT 
                        COALESCE(SUM(amount_paid), 0) 
                     FROM 
                        invoice_payments 
                     WHERE 
                        invoice_number IN (
                            SELECT order_number FROM saved_orders WHERE customer = customers.customer_id AND order_state IN (0,1,2)
                        )
                    ) AS balance
                FROM 
                    customers
                WHERE 
                    customer_id = :id
                LIMIT 1";

        $stmt = $this->link->prepare($sql);
        $stmt->bindValue(':id', $customer_id);
        $stmt->execute();
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
    /**
     * Fetch all transactions for a given customer
     * Returns array of transactions with date, invoice number, amount, paid, balance
     */
    public function getTransactions($customer_id) {
        try {
            $sql = "SELECT 
                        so.orderdate AS date,
                        so.datecaptured AS order_created,
                        so.order_number AS document_no,
                        'Invoice' AS description,
                        so.total_bill AS amount,
                        IFNULL(SUM(ip.amount_paid), 0) AS paid
                    FROM saved_orders so
                    LEFT JOIN invoice_payments ip 
                        ON ip.invoice_number = so.order_number
                    WHERE so.customer = :customer_id
                    AND so.order_state IN (0,1,2)
                    GROUP BY so.order_number

                    UNION ALL

                    SELECT
                        p.payment_date AS date,
                        p.datecaptured AS order_created,
                        p.sale_id AS document_no,
                        'Cash Sale' AS description,
                        p.amount_paid AS amount,
                        p.amount_paid AS paid
                    FROM payments p
                    WHERE p.customer = :customer_id
                    AND p.payment_type = 'Cash Sale'
                    AND p.payment_state = 1

                    UNION ALL

                    SELECT
                        qt.transaction_date AS date,
                        qt.transaction_date AS order_created,
                        qt.quote_number AS document_no,
                        'Quotation' AS description,
                        qt.total AS amount,
                        0 AS paid
                    FROM quotations qt
                    WHERE qt.customer = :customer_id
                    AND qt.quotation_state IN (0,1,2)

                    ORDER BY date ASC, order_created ASC";

            $stmt = $this->link->prepare($sql);
            $stmt->bindValue(':customer_id', $customer_id, PDO::PARAM_INT);
            $stmt->execute();

            return $stmt->fetchAll(PDO::FETCH_ASSOC);

        } catch (PDOException $e) {
            error_log("getTransactions Error: " . $e->getMessage());
            return [];
        }
    }



    /**
     * Optional: Get customer summary (number of invoices, total balance)
     */
    public function getCustomerSummary($customer_id) {
        try {
            $sql = "SELECT
                    COUNT(so.order_number) AS invoice_count,
                    COALESCE(SUM(so.total_bill), 0) AS total_invoiced,
                    COALESCE(p.total_paid, 0) AS total_paid,
                    COALESCE(SUM(so.total_bill), 0) - COALESCE(p.total_paid, 0) AS total_balance
                FROM 
                    saved_orders so
                LEFT JOIN (
                    SELECT 
                        customer,
                        SUM(amount_paid) AS total_paid
                    FROM invoice_payments
                    WHERE customer = :customer_id
                    GROUP BY customer
                ) p ON p.customer = so.customer
                WHERE so.customer = :customer_id
                AND so.order_state IN (0,1,2)
            ";

            $stmt = $this->link->prepare($sql);
            $stmt->bindValue(':customer_id', $customer_id, PDO::PARAM_INT);
            $stmt->execute();

            return $stmt->fetch(PDO::FETCH_ASSOC);

        } catch (PDOException $e) {
            error_log("getCustomerSummary Error: " . $e->getMessage());
            return [
                'invoice_count'   => 0,
                'total_invoiced'  => 0,
                'total_paid'      => 0,
                'total_balance'   => 0
            ];
        }
    }

    public function getCustomerLastActivity($customer_id) {
        try {
            $sql = "
                (
                    SELECT 
                        so.datecaptured AS activity_date,
                        CONCAT('Invoice Created: ', so.order_number) AS description
                    FROM saved_orders so
                    WHERE so.customer = :customer_id
                )
                UNION ALL
                (
                    SELECT
                        p.datecaptured AS activity_date,
                        CASE 
                            WHEN p.payment_type = 'Cash Sale' THEN 'Cash Sale'
                            WHEN p.payment_type = 'Order Payment' THEN 'Invoice Payment'
                            ELSE 'Payment'
                        END AS description
                    FROM payments p
                    WHERE p.customer = :customer_id
                )
                UNION ALL
                (
                    SELECT
                        c.datecreated AS activity_date,
                        'Account Created' AS description
                    FROM customers c
                    WHERE c.customer_id = :customer_id
                )
                ORDER BY activity_date DESC
                LIMIT 1
            ";

            $stmt = $this->link->prepare($sql);
            $stmt->bindValue(':customer_id', $customer_id, PDO::PARAM_INT);
            $stmt->execute();

            return $stmt->fetch(PDO::FETCH_ASSOC);

        } catch (PDOException $e) {
            error_log('getCustomerLastActivity Error: ' . $e->getMessage());
            return null;
        }
    }

    public function getNextCustomerId(){
        $sql = "SELECT MAX(customer_id) AS max_id FROM customers";
        $stmt = $this->link->prepare($sql);
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($row && $row['max_id'] !== null) {
            return intval($row['max_id']) + 1;
        } else {
            return 8000; // Starting ID when table is empty
        }
    }

    public function addCustomer($data){
        $sql = "INSERT INTO customers (customer_id, customer_name, contact_number, whatsapp_number, email, gender, customer_address, location, registered_location,   type_customer, nationality, operator, datecreated, customer_state) 
                VALUES (:customer_id, :customer_name, :contact_number, :whatsapp_number, :email, :gender, :customer_address, :location, :registration_loc, :type_customer, :nationality, :operator, :datecreated, :customer_state)";

        $stmt = $this->link->prepare($sql);

        $stmt->bindParam(':customer_id',     $data['customer_id'], PDO::PARAM_INT);
        $stmt->bindParam(':customer_name',   $data['customer_name'], PDO::PARAM_STR);
        $stmt->bindParam(':contact_number',  $data['contact_number'], PDO::PARAM_STR);
        $stmt->bindParam(':whatsapp_number', $data['whatsapp_number'], PDO::PARAM_STR);
        $stmt->bindParam(':email',           $data['email'], PDO::PARAM_STR);
        $stmt->bindParam(':gender',          $data['gender'], PDO::PARAM_STR);
        $stmt->bindParam(':customer_address',$data['customer_address'], PDO::PARAM_STR);
        $stmt->bindParam(':location',        $data['location'], PDO::PARAM_STR);
        $stmt->bindParam(':registration_loc',$data['registration_loc'], PDO::PARAM_STR);
        $stmt->bindParam(':type_customer',   $data['type_customer'], PDO::PARAM_STR);
        $stmt->bindParam(':nationality',     $data['nationality'], PDO::PARAM_INT);
        $stmt->bindParam(':operator',        $data['operator'], PDO::PARAM_INT);
        $stmt->bindParam(':datecreated',     $data['datecreated'], PDO::PARAM_STR);
        $stmt->bindParam(':customer_state',  $data['customer_state'], PDO::PARAM_INT);

        if ($stmt->execute()) {
            return true;
        } else {
            return false;
        }
    }

    public function emailExists($email){
        $sql = "SELECT 1 FROM customers WHERE email = ? LIMIT 1";
        $stmt = $this->link->prepare($sql);
        $stmt->execute([$email]);
        return $stmt->fetchColumn() !== false;
    }
    
    public function phoneNumberExists($phone){
        $sql = "SELECT 1 FROM customers WHERE contact_number = ? LIMIT 1";
        $stmt = $this->link->prepare($sql);
        $stmt->execute([$phone]);
        return $stmt->fetchColumn() !== false;
    }

    public function insertCustomer($data){
        try {

            $sql = "INSERT INTO customers 
                (
                    customer_id,
                    customer_name,
                    contact_number,
                    registered_location,
                    type_customer,
                    operator,
                    datecreated,
                    customer_state
                ) 
                VALUES 
                (
                    :customer_id,
                    :customer_name,
                    :contact_number,
                    :registered_loc,
                    :type_customer,
                    :operator,
                    :datecreated,
                    :customer_state
                )
            ";

            $stmt = $this->link->prepare($sql);

            $success = $stmt->execute([
                ':customer_id'    => $data['customer_id'],
                ':customer_name'  => $data['customer_name'],
                ':contact_number' => $data['phone'],
                ':registered_loc' => $data['registered_loc'],
                ':type_customer'  => $data['customer_type'],
                ':operator'       => $data['operator'],
                ':datecreated'    => $data['date_created'],
                ':customer_state' => $data['customer_state']
            ]);

            if (!$success) {
                $error = $stmt->errorInfo(); // [SQLSTATE, driver_code, driver_message]

                return [
                    'status' => false,
                    'error'  => $error[2] ?? 'Unknown database error'
                ];
            }

            return [
                'status' => true
            ];

        } catch (PDOException $e) {
            return [
                'status' => false,
                'error'  => $e->getMessage()
            ];
        }
    }


    public function getCustomerByContactNumber($contact_number){
        $sql = "SELECT 
                    id, 
                    customer_id 
                FROM 
                    customers 
                WHERE 
                    contact_number = :contact_number 
                LIMIT 1";

        $stmt = $this->link->prepare($sql);
        $stmt->execute([':contact_number' => $contact_number
        ]);

        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    public function getCustomerByPhone($phone){
        $sql = "SELECT 
                    customer_id, customer_name 
                FROM 
                    customers 
                WHERE 
                    contact_number = :phone 
                LIMIT 1";

        $stmt = $this->link->prepare($sql);
        $stmt->execute([
            ':phone' => $phone
        ]);

        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    public function customerDatabase($location) {
        $sql = "SELECT 
                    customer_id,
                    customer_name,
                    contact_number AS phone,
                    type_customer AS customer_type,
                    customer_state,
                    datecreated AS date_created,
                    branches.branch_name AS registered_location
                FROM 
                    customers
                LEFT JOIN
                    branches ON customers.registered_location = branches.branch_id
                WHERE 
                    registered_location = :location
                ORDER BY 
                    customer_name ASC";

        $stmt = $this->link->prepare($sql);
        $stmt->bindValue(':location', $location, PDO::PARAM_STR);
        $stmt->execute();

        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    /**
     * Get last N transactions for a customer
     *
     * @param int $customerId
     * @param int $limit
     * @return array
     */
    public function getLastTransactions($customerId, $limit = 5) {
        $sql = "SELECT 
                    q.quotation_number AS ref_number,
                    q.transaction_date AS date,
                    'Quotation' AS type,
                    q.total AS total,
                    '' AS description
                FROM quotations q
                WHERE q.customer = :customer_id
            
            UNION ALL
                SELECT 
                    o.order_number AS ref_number,
                    o.orderdate AS date,
                    'Invoice' AS type,
                    o.total_bill AS total,
                    '' AS description
                FROM saved_orders o
                WHERE o.customer = :customer_id
                AND o.order_state IN (0,1,2)
            
            UNION ALL
            (
                -- Payments
                SELECT 
                    p.sale_id AS ref_number,
                    p.payment_date AS date,
                    CASE 
                        WHEN p.payment_type = 'Cash Sale' THEN 'Cash Sale'
                        ELSE 'Invoice Payment'
                    END AS type,
                    p.amount_paid AS total,
                    p.payment_type AS description
                FROM payments p
                WHERE p.customer = :customer_id
            )
            ORDER BY date DESC
            LIMIT :limit
        ";

        $stmt = $this->link->prepare($sql);
        $stmt->bindValue(':customer_id', $customerId, PDO::PARAM_INT);
        $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
        $stmt->execute();

        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    /**
     * Get customer sales summary with last transactions
     *
     * @param int $customerId
     * @param int $transactionLimit
     * @return array
     */
    public function getCustomerSalesSummary($customerId, $transactionLimit = 5) {
        // 1️⃣ Total Sales (Invoices + Cash Sales)
        $sqlTotalInvoices = "SELECT SUM(total_bill) FROM saved_orders WHERE customer = :customer_id AND order_state IN (0,1,2)";
        $stmt = $this->link->prepare($sqlTotalInvoices);
        $stmt->bindValue(':customer_id', $customerId, PDO::PARAM_INT);
        $stmt->execute();
        $invoiceTotal = floatval($stmt->fetchColumn());

        $sqlTotalCash = "SELECT SUM(amount_paid) FROM payments WHERE customer = :customer_id AND payment_type = 'Cash Sale'";
        $stmt = $this->link->prepare($sqlTotalCash);
        $stmt->bindValue(':customer_id', $customerId, PDO::PARAM_INT);
        $stmt->execute();
        $cashTotal = floatval($stmt->fetchColumn());

        $totalSales = $invoiceTotal + $cashTotal;

        // 2️⃣ Last Sale Date (latest invoice OR cash sale)
        $sqlLastInvoice = "SELECT MAX(orderdate) FROM saved_orders WHERE customer = :customer_id AND order_state IN (0,1,2)";
        $stmt = $this->link->prepare($sqlLastInvoice);
        $stmt->bindValue(':customer_id', $customerId, PDO::PARAM_INT);
        $stmt->execute();
        $lastInvoice = $stmt->fetchColumn();

        $sqlLastCash = "SELECT MAX(payment_date) FROM payments WHERE customer = :customer_id AND payment_type = 'Cash Sale'";
        $stmt = $this->link->prepare($sqlLastCash);
        $stmt->bindValue(':customer_id', $customerId, PDO::PARAM_INT);
        $stmt->execute();
        $lastCash = $stmt->fetchColumn();

        $lastSaleDate = max($lastInvoice, $lastCash);

        // 3️⃣ Last Transactions (Quotations / Invoices / Payments / Cash Sales)
        $sqlTransactions = "SELECT q.quote_number AS ref_number,
                q.transaction_date AS date,
                'Quotation' AS type,
                q.total AS total,
                '' AS description
            FROM quotations q
            WHERE q.customer = :customer_id

            UNION ALL

            SELECT o.order_number AS ref_number,
                o.orderdate AS date,
                'Invoice' AS type,
                o.total_bill AS total,
                '' AS description
            FROM saved_orders o
            WHERE o.customer = :customer_id
            AND o.order_state IN (0,1,2)

            UNION ALL

            SELECT p.sale_id AS ref_number,
                p.payment_date AS date,
                CASE WHEN p.payment_type = 'Cash Sale' THEN 'Cash Sale' ELSE 'Invoice Payment' END AS type,
                p.amount_paid AS total,
                '' AS description
            FROM payments p
            WHERE p.customer = :customer_id

            ORDER BY date DESC
            LIMIT :limit
        ";

        $stmt = $this->link->prepare($sqlTransactions);
        $stmt->bindValue(':customer_id', $customerId, PDO::PARAM_INT);
        $stmt->bindValue(':limit', $transactionLimit, PDO::PARAM_INT);
        $stmt->execute();
        $transactions = $stmt->fetchAll(PDO::FETCH_ASSOC);

        // 4️⃣ Add color-coding and tooltips
        foreach ($transactions as &$t) {
            switch ($t['type']) {
                case 'Quotation':      $t['color'] = 'badge-secondary'; break;
                case 'Invoice':        $t['color'] = 'badge-primary';   break;
                case 'Cash Sale':      $t['color'] = 'badge-success';   break;
                case 'Invoice Payment':$t['color'] = 'badge-info';      break;
                default:               $t['color'] = 'badge-light';     break;
            }

            // Tooltip includes ref number, type, and total amount
            $t['tooltip'] = "{$t['type']} - {$t['ref_number']} - ".number_format($t['total'],2);
        }

        return [
            'total_sales'    => $totalSales,
            'last_sale_date' => $lastSaleDate,
            'transactions'   => $transactions
        ];
    }






}
?>
