<?php

include_once 'config.php';

class reports_model
{
    public $link;

    public function __construct()
    {
        $db_connection = new dbConnection();
        $this->link = $db_connection->connect();
    }

    public function getSalesSummary($location, $from_date, $to_date) {
        // Sum from payments
        $sql = "SELECT
                    IFNULL(SUM(p.amount_paid), 0) AS total_payments,
                    IFNULL(SUM(
                        CASE 
                            WHEN p.payment_type = 'Cash Sale' THEN p.amount_paid 
                            ELSE 0 
                        END
                    ), 0) AS total_cash_sales,
                    IFNULL(SUM(
                        CASE 
                            WHEN p.payment_type = 'Order Payment' THEN p.amount_paid 
                            ELSE 0 
                        END
                    ), 0) AS total_invoice_payments,
                    (IFNULL(SUM(
                        CASE 
                            WHEN p.payment_type = 'Cash Sale' THEN p.amount_paid 
                            ELSE 0 
                        END
                    ), 0)
                    +
                    IFNULL((
                        SELECT 
                            SUM(so.total_bill)
                        FROM 
                            saved_orders so
                        WHERE 
                            so.invoice_location = :location
                            AND so.orderdate >= :start_date
                            AND so.orderdate <= :end_date
                            AND so.order_state IN (0,1,2)
                    ), 0)
                ) AS total_sales
                FROM 
                    payments p
                WHERE 
                    p.location = :location
                    AND p.datecaptured >= :start_date
                    AND p.datecaptured <= :end_date
                    AND p.payment_state = 1
        ";

        $stmt = $this->link->prepare($sql);
        $stmt->bindParam(':location', $location, PDO::PARAM_INT);
        $stmt->bindParam(':start_date', $from_date, PDO::PARAM_STR);
        $stmt->bindParam(':end_date', $to_date, PDO::PARAM_STR);
        $stmt->execute();

        return $stmt->fetch(PDO::FETCH_ASSOC);
    }


    public function getSalesByCategoryGross($location, $start_date, $end_date){
        $sql = "SELECT
            pc.categoryname,
            SUM(so.quantity) AS qty_sold,
            SUM(so.tax) AS total_tax,

            SUM(
                CASE
                    WHEN so.type = 'Cash Sale' THEN
                        so.total - (so.total / cs.order_total * cs.discount)
                    WHEN so.type = 'Customer Order' THEN
                        so.total - (so.total / inv.order_total * inv.discount)
                END
            ) AS total_sales

        FROM sales_order so

        /* PRODUCT + CATEGORY */
        INNER JOIN products p 
            ON so.product_code = p.product_id
        INNER JOIN product_categories pc 
            ON p.category = pc.category_id

        /* CASH SALE HEADER */
        LEFT JOIN payments pay
            ON pay.sale_id = so.sales_number
            AND so.type = 'Cash Sale'
            AND pay.payment_state = 1

        /* INVOICE HEADER */
        LEFT JOIN saved_orders invh
            ON invh.order_number = so.sales_number
            AND so.type = 'Customer Order'
            AND invh.order_state IN (0,1,2)

        /* CASH SALE ORDER TOTALS */
        LEFT JOIN (
            SELECT
                so2.sales_number,
                SUM(so2.total) AS order_total,
                MAX(p2.discount) AS discount
            FROM sales_order so2
            INNER JOIN payments p2
                ON p2.sale_id = so2.sales_number
                AND p2.payment_type = 'Cash Sale'
            GROUP BY so2.sales_number
        ) cs ON cs.sales_number = so.sales_number

        /* INVOICE ORDER TOTALS */
        LEFT JOIN (
            SELECT
                so3.sales_number,
                SUM(so3.total) AS order_total,
                MAX(o2.discount) AS discount
            FROM sales_order so3
            INNER JOIN saved_orders o2
                ON o2.order_number = so3.sales_number
            GROUP BY so3.sales_number
        ) inv ON inv.sales_number = so.sales_number

        WHERE
        (
            so.type = 'Cash Sale'
            AND pay.location = :location
            AND DATE(pay.datecaptured) BETWEEN :start_date AND :end_date
        )
        OR
        (
            so.type = 'Customer Order'
            AND invh.invoice_location = :location
            AND DATE(invh.orderdate) BETWEEN :start_date AND :end_date
        )

        GROUP BY
            pc.category_id,
            pc.categoryname

        ORDER BY
            total_sales DESC
        ";

        $stmt = $this->link->prepare($sql);
        $stmt->bindValue(':location', $location, PDO::PARAM_INT);
        $stmt->bindValue(':start_date', $start_date);
        $stmt->bindValue(':end_date', $end_date);
        $stmt->execute();

        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }


    public function getSalesTrend($location, $from_date, $to_date, $period){
        switch ($period) {
            case 'weekly':
                $groupBy = "YEAR(sale_date), WEEK(sale_date)";
                $label   = "CONCAT('Wk ', WEEK(sale_date), ' ', YEAR(sale_date))";
                break;

            case 'monthly':
                $groupBy = "YEAR(sale_date), MONTH(sale_date)";
                $label   = "DATE_FORMAT(sale_date, '%b %Y')";
                break;

            default: // daily
                $groupBy = "sale_date";
                $label   = "DATE_FORMAT(sale_date, '%d %b, %Y')";
        }

        $sql = "SELECT 
                {$label} AS label,
                SUM(line_total) AS total_sales
            FROM (
                /* CASH SALES */
                SELECT 
                    DATE(p.datecaptured) AS sale_date,
                    p.amount_paid AS line_total
                FROM payments p
                WHERE p.payment_type = 'Cash Sale'
                AND p.payment_state = 1
                AND p.location = :location

                UNION ALL

                /* INVOICE SALES */
                SELECT 
                    DATE(o.orderdate) AS sale_date,
                    o.total_bill AS line_total
                FROM saved_orders o
                WHERE o.invoice_location = :location
                AND o.order_state IN(0,1,2)
            ) sales
            WHERE sale_date BETWEEN :from_date AND :to_date
            GROUP BY {$groupBy}
            ORDER BY sale_date ASC
        ";

        $stmt = $this->link->prepare($sql);
        $stmt->execute([
            ':location'  => $location,
            ':from_date' => $from_date,
            ':to_date'   => $to_date
        ]);

        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getTopProducts($location, $from_date, $to_date) {
        $sql = "SELECT 
                    so.product_code,
                    p.product_name,
                    SUM(so.quantity) AS total_qty,
                    SUM(
                        CASE 
                            WHEN so.type = 'Cash Sale' THEN 
                                so.total - (so.total / t.total_order * t.discount)
                            WHEN so.type = 'Customer Order' THEN
                                so.total - (so.total / s.total_order * s.discount)
                            ELSE so.total
                        END
                    ) AS total_sales
                FROM 
                    sales_order so
                LEFT JOIN 
                    products p ON p.product_id = so.product_code

                /* Cash Sale totals and discount */
                LEFT JOIN (
                    SELECT 
                        p.sale_id, 
                        SUM(so.total) AS total_order,
                        p.discount
                    FROM 
                        payments p
                    INNER JOIN 
                        sales_order so ON so.sales_number = p.sale_id
                    WHERE 
                        p.payment_type = 'Cash Sale' 
                        AND DATE(p.datecaptured) BETWEEN :from_date AND :to_date
                        AND p.location =:location
                    GROUP BY 
                        p.sale_id, p.discount
                ) t ON t.sale_id = so.sales_number

                /* Invoice totals and discount */
                LEFT JOIN (
                    SELECT 
                        s.order_number,
                        SUM(so.total) AS total_order, 
                        s.discount
                    FROM 
                        saved_orders s
                    INNER JOIN 
                        sales_order so ON so.sales_number = s.order_number
                    WHERE 
                        DATE(s.orderdate) BETWEEN :from_date AND :to_date
                        AND invoice_location =:location
                    GROUP BY 
                        s.order_number, s.discount
                ) s ON s.order_number = so.sales_number

                WHERE 
                    so.type IN ('Cash Sale','Customer Order')
                    AND so.date_captured BETWEEN :from_date AND :to_date
                    AND so.status = 1
                    AND so.department =:location
                GROUP BY 
                    so.product_code, p.product_name
                ORDER BY 
                    total_qty DESC";
        
        $stmt = $this->link->prepare($sql);
        $stmt->bindValue(':location', $location);
        $stmt->bindValue(':from_date', $from_date);
        $stmt->bindValue(':to_date', $to_date);
        $stmt->execute();

        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getSalesByCustomer($location, $from_date, $to_date) {
        $sql = "SELECT 
                    c.customer_name,
                    SUM(so.quantity) AS total_qty,

                    SUM(
                        CASE 
                            WHEN so.type = 'Cash Sale' THEN 
                                so.total - (so.total / cs.order_total * cs.discount)
                            WHEN so.type = 'Customer Order' THEN
                                so.total - (so.total / inv.order_total * inv.discount)
                        END
                    ) AS total_sales

                FROM sales_order so

                /* CASH SALE HEADER */
                LEFT JOIN payments p 
                    ON p.sale_id = so.sales_number
                    AND so.type = 'Cash Sale'
                    AND p.payment_state = 1

                /* INVOICE HEADER */
                LEFT JOIN saved_orders o 
                    ON o.order_number = so.sales_number
                    AND so.type = 'Customer Order'
                    AND o.order_state IN (0,1,2)

                /* CUSTOMER */
                LEFT JOIN customers c 
                    ON c.customer_id = COALESCE(p.customer, o.customer)

                /* CASH SALE ORDER TOTALS */
                LEFT JOIN (
                    SELECT 
                        so2.sales_number,
                        SUM(so2.total) AS order_total,
                        MAX(p2.discount) AS discount
                    FROM sales_order so2
                    INNER JOIN payments p2 
                        ON p2.sale_id = so2.sales_number
                        AND p2.payment_type = 'Cash Sale'
                    GROUP BY so2.sales_number
                ) cs ON cs.sales_number = so.sales_number

                /* INVOICE ORDER TOTALS */
                LEFT JOIN (
                    SELECT 
                        so3.sales_number,
                        SUM(so3.total) AS order_total,
                        MAX(o2.discount) AS discount
                    FROM sales_order so3
                    INNER JOIN saved_orders o2 
                        ON o2.order_number = so3.sales_number
                    GROUP BY so3.sales_number
                ) inv ON inv.sales_number = so.sales_number

                WHERE 
                (
                    so.type = 'Cash Sale'
                    AND p.location = :location
                    AND DATE(p.datecaptured) BETWEEN :from_date AND :to_date
                )
                OR
                (
                    so.type = 'Customer Order'
                    AND o.invoice_location = :location
                    AND DATE(o.orderdate) BETWEEN :from_date AND :to_date
                )

                GROUP BY 
                    c.customer_id,
                    c.customer_name

                ORDER BY 
                    total_sales DESC";

        $stmt = $this->link->prepare($sql);
        $stmt->bindValue(':location', $location);
        $stmt->bindValue(':from_date', $from_date);
        $stmt->bindValue(':to_date', $to_date);
        $stmt->execute();

        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }


    public function getPaymentModesSummary($location, $start_date, $end_date){
        $sql = "SELECT
                    payment_method,
                    SUM(amount_paid) AS total_amount
                FROM 
                    payments
                WHERE
                    payment_state = 1
                    AND location = :location
                    AND DATE(datecaptured) BETWEEN :start_date AND :end_date
                GROUP BY 
                    payment_method
                ORDER BY 
                    total_amount DESC";

        $stmt = $this->link->prepare($sql);
        $stmt->bindValue(':location', $location, PDO::PARAM_INT);
        $stmt->bindValue(':start_date', $start_date);
        $stmt->bindValue(':end_date', $end_date);
        $stmt->execute();

        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getSalesVsStock($location, $from_date, $to_date, $limit = 20){
        $sql = "SELECT
            p.product_id,
            p.product_name,
            SUM(so.quantity) AS qty_sold,
            COALESCE(bp.quantity, wp.quantity, 0) AS stock_balance,
            COALESCE(bp.min_level, wp.min_level, 0) AS min_level

        FROM 
            sales_order so

        INNER JOIN products p 
            ON p.product_id = so.product_code

        /* CASH SALES */
        LEFT JOIN payments pay
            ON pay.sale_id = so.sales_number
            AND so.type = 'Cash Sale'
            AND pay.payment_state = 1

        /* INVOICES */
        LEFT JOIN saved_orders inv
            ON inv.order_number = so.sales_number
            AND so.type = 'Customer Order'
            AND inv.order_state IN (0,1,2)

        /* BRANCH STOCK */
        LEFT JOIN branch_products bp
            ON bp.product_id = p.product_id
            AND bp.branch_id = :location

        /* WAREHOUSE STOCK */
        LEFT JOIN warehouse_products wp
            ON wp.product_id = p.product_id
            AND wp.warehouse_id = :location

        WHERE
        (
            so.type = 'Cash Sale'
            AND pay.location = :location
            AND DATE(pay.datecaptured) BETWEEN :from_date AND :to_date
        )
        OR
        (
            so.type = 'Customer Order'
            AND inv.invoice_location = :location
            AND DATE(inv.orderdate) BETWEEN :from_date AND :to_date
        )

        GROUP BY
            p.product_id,
            p.product_name,
            stock_balance

        ORDER BY
            qty_sold DESC

        LIMIT :limit
        ";

        $stmt = $this->link->prepare($sql);
        $stmt->bindValue(':location', $location, PDO::PARAM_INT);
        $stmt->bindValue(':from_date', $from_date);
        $stmt->bindValue(':to_date', $to_date);
        $stmt->bindValue(':limit', (int)$limit, PDO::PARAM_INT);
        $stmt->execute();

        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getDiscountSummary($location, $from_date, $to_date){
        $sql1 = "SELECT
                    SUM(discount) as cash_discounts
                FROM 
                    payments p
                WHERE 
                    p.location = :location
                    AND p.payment_state = 1
                    AND p.datecaptured BETWEEN :from_date AND :to_date
                    AND p.discount > 0";

        $stmt1 = $this->link->prepare($sql1);
        $stmt1->execute([':location' => $location, ':from_date' => $from_date, ':to_date' => $to_date]);

        $row1 = $stmt1->fetch(PDO::FETCH_ASSOC);
        
        $sql2 = "SELECT
                    SUM(discount) as invoice_discounts
                FROM
                    saved_orders so
                WHERE
                    so.invoice_location =:location
                    AND so.orderdate BETWEEN :from_date AND :to_date
                    AND so.order_state IN (0,1,2)
                    AND so.discount > 0";  
        $stmt2 = $this->link->prepare($sql2);
        $stmt2->execute([':location' => $location, ':from_date' => $from_date, ':to_date' => $to_date]);

        $row2 = $stmt2->fetch(PDO::FETCH_ASSOC);


        return [
            'cash_discounts'    => (float)$row1['cash_discounts'],
            'invoice_discounts' => (float)$row2['invoice_discounts'],
            'total_discounts'   => (float)$row1['cash_discounts'] + (float)$row2['invoice_discounts']
        ];
    }

    public function getOutstandingInvoicesSummary($location, $start, $end){
        $sql = "SELECT
                    COUNT(so.order_number) AS invoice_count,
                    SUM(so.total_bill) AS total_invoices,
                    IFNULL(SUM(p.total_paid), 0) AS total_paid,
                    SUM(so.total_bill) - IFNULL(SUM(p.total_paid), 0) AS total_outstanding
                FROM saved_orders so
                LEFT JOIN (
                    SELECT 
                        invoice_number,
                        SUM(amount_paid) AS total_paid
                    FROM invoice_payments
                    GROUP BY invoice_number
                ) p ON p.invoice_number = so.order_number
                WHERE
                    so.invoice_location = :location
                    AND so.orderdate BETWEEN :start AND :end
                    AND so.order_state IN (0,1)";

        $stmt = $this->link->prepare($sql);
        $stmt->execute([':location' => $location, ':start'=> $start, ':end'=>$end]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
    public function getInvoiceAging($location){
        $sql = "SELECT
                SUM(CASE WHEN DATEDIFF(CURDATE(), so.orderdate) BETWEEN 0 AND 30 
                    THEN (so.total_bill - IFNULL(paid.total_paid,0)) ELSE 0 END) AS amt_0_30,
                COUNT(CASE WHEN DATEDIFF(CURDATE(), so.orderdate) BETWEEN 0 AND 30 
                    THEN so.order_number END) AS cnt_0_30,

                SUM(CASE WHEN DATEDIFF(CURDATE(), so.orderdate) BETWEEN 31 AND 60 
                    THEN (so.total_bill - IFNULL(paid.total_paid,0)) ELSE 0 END) AS amt_31_60,
                COUNT(CASE WHEN DATEDIFF(CURDATE(), so.orderdate) BETWEEN 31 AND 60 
                    THEN so.order_number END) AS cnt_31_60,

                SUM(CASE WHEN DATEDIFF(CURDATE(), so.orderdate) BETWEEN 61 AND 90 
                    THEN (so.total_bill - IFNULL(paid.total_paid,0)) ELSE 0 END) AS amt_61_90,
                COUNT(CASE WHEN DATEDIFF(CURDATE(), so.orderdate) BETWEEN 61 AND 90 
                    THEN so.order_number END) AS cnt_61_90,

                SUM(CASE WHEN DATEDIFF(CURDATE(), so.orderdate) > 90 
                    THEN (so.total_bill - IFNULL(paid.total_paid,0)) ELSE 0 END) AS amt_90_plus,
                COUNT(CASE WHEN DATEDIFF(CURDATE(), so.orderdate) > 90 
                    THEN so.order_number END) AS cnt_90_plus

            FROM saved_orders so
            LEFT JOIN (
                SELECT invoice_number, SUM(amount_paid) AS total_paid
                FROM invoice_payments
                GROUP BY invoice_number
            ) paid ON paid.invoice_number = so.order_number

            WHERE
                so.invoice_location = :location
                AND so.order_state IN (0,1)
                AND (so.total_bill - IFNULL(paid.total_paid,0)) > 0
        ";

        $stmt = $this->link->prepare($sql);
        $stmt->execute([':location' => $location]);

        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

     
}

